/* Agent Tcl
   Bob Gray
   1 September 1995

   genQueue.cc

   This file implements a FIFO or queue.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "suppDString.h"
#include "genDStringQueue.h"

/*
 * initial and minimum queue size
 */

const int DEFAULT_QUEUE_SIZE = 8;
const int MINIMUM_QUEUE_SIZE = 8;

/*
 * constructors, assignment operator and destructor for DynamicStringQueue
 */

DynamicStringQueue::DynamicStringQueue (void)
{
	/* elements */

    elements = new DynamicString [DEFAULT_QUEUE_SIZE];
   
	/* number of elements and number of slots */

    count = 0;
    slots = DEFAULT_QUEUE_SIZE;

	/* expand and shrink count */

    expand_count = slots;

    if (slots <= MINIMUM_QUEUE_SIZE) {
	shrink_count = -1;
    } else {
	shrink_count = slots / 4;
    }

	/* head and tail indices */

    tail = 0;
    head = 0;
}

static DynamicString *copyElements (int slots, int count, DynamicString *elements, int head)
{
    DynamicString *newElements = new DynamicString [slots];

    int copyIndex = head;

    for (int i = 0; i < count; ++i) {

	newElements[copyIndex] = elements[copyIndex];

	if (++copyIndex >= slots) {
	   copyIndex = 0;
	}
    }

    return (newElements);
}

DynamicStringQueue::DynamicStringQueue (const DynamicStringQueue &queue):
    count        (queue.count),
    slots        (queue.slots),
    shrink_count (queue.shrink_count),
    expand_count (queue.expand_count), 
    head         (queue.head),
    tail         (queue.tail)
{
	/*
	 * copy in the elements from the other queue
	 */

    DELETE_ARRAY_OBJECT(elements);
    elements = copyElements (slots, count, queue.elements, head);
}

DynamicStringQueue &DynamicStringQueue::operator= (const DynamicStringQueue &queue)
{
    if (this != &queue) {

	    /*
	     * copy in the queue size, etc.
	     */

        count        = queue.count;
        slots        = queue.slots;
        shrink_count = queue.shrink_count;
        expand_count = queue.expand_count;
        head         = queue.head;
        tail         = queue.tail;

	    /*
	     * copy in the elements from the other queue
	     */

        DELETE_ARRAY_OBJECT(elements);
        elements = copyElements (slots, count, queue.elements, head);
    }

    return (*this);
}

DynamicStringQueue::~DynamicStringQueue ()
{
    DELETE_ARRAY_OBJECT(elements);
}

/* DynamicStringQueue::resize_queue

   Purpose: Expand or shrink the queue

     Input: which = e_QUEUE_SHRINK or e_QUEUE_EXPAND

    Output: The procedure doubles or halves the size of the queue.
*/

void DynamicStringQueue::resize_queue (QueueSizing which)
{
  int new_slots;
  DynamicString *new_elements;

  if (which == e_QUEUE_EXPAND) {
    new_slots = slots << 1;
  } else {
    new_slots = slots >> 1;
  }

  new_elements = new DynamicString [new_slots];

    /* copy over the queue elements */

  for (int i = 0; i < count; i++) {
    new_elements[i] = elements[head];
    head = (head + 1) % slots;
  }

  DELETE_ARRAY_OBJECT(elements);

    /* reinitialize all the data fields */

  elements     = new_elements;
  slots        = new_slots;
  head         = 0;
  tail         = count;
  expand_count = new_slots;

  if (slots <= MINIMUM_QUEUE_SIZE) {
    shrink_count = -1;
  } else {
    shrink_count = slots / 4;
  }
}

/* DynamicStringQueue::enqueue

   Purpose: Add an element to the the tail of the queue

     Input: item = DynamicString to add to the tail of the queue

    Output: The procedure adds the DynamicString to the queue.
*/

void DynamicStringQueue::enqueue (DynamicString item)
{
    /* assertions on the parameters */

  assert (item != NULL);
  
    /* add the item to the queue */

  elements[tail] = item;
  tail           = (tail + 1) % slots;
  count         += 1;

    /* expand the queue if necessary */

  if (count >= expand_count) {
    resize_queue (e_QUEUE_EXPAND);
  }
}

/* DynamicStringQueue::push

   Purpose: Add an element to the the head of the queue

     Input: item = DynamicString to add to the head of the queue

    Output: The procedure adds the DynamicString to the queue.
*/

void DynamicStringQueue::push (DynamicString item)
{
    /* assertions on the parameters */

  assert (item != NULL);
  
    /* add the item to the queue */

  head           = (head - 1) % slots;
  elements[head] = item;
  count         += 1;

    /* expand the queue if necessary */

  if (count >= expand_count) {
    resize_queue (e_QUEUE_EXPAND);
  }
}

/* DynamicStringQueue::dequeue

   Purpose: Get the item at the head of the queue

     Input: None

    Output: The procedure returns NULL if the queue is empty.  Otherwise the
            procedure returns the DynamicString at the head of the queue and 
            removes that DynamicString from the queue.
*/

DynamicString DynamicStringQueue::dequeue (void)
{
  DynamicString item;

    /* check if the queue is empty */

  if (head == tail) {
    return NULL;
  }

    /* get the head of the queue */

  item           = elements[head];
  elements[head] = DynamicString();
  head           = (head + 1) % slots;
  count         -= 1;  

    /* shrink the queue if necessary */

  if (count <= shrink_count) {
    resize_queue (e_QUEUE_SHRINK); 
  }

    /* return the head of the queue */

  return (item);
}

/* DynamicStringQueue::peek

   Purpose: Peek at an item in the queue

     Input: i = position of the DynamicString within the queue

    Output: The procedure returns NULL if the queue has i or fewer items.
            Otherwise the procedure returns the DynamicString that is in
	    the iTH position in the queue. 
*/

DynamicString DynamicStringQueue::peek (int i) const
{
  int temp;

  if (i >= count) {
    DynamicString empty;
    return (empty);
  } 

  temp = (head + i) % slots; 
  return (elements[temp]);
}
