/* Agent Tcl
   Bob Gray
   19 April 1997

   genEncrypt.cc

   This file defines the "meta-messages" that are used to pass around encrypted 
   and digitially-signed messages.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "suppStrings.h"
#include "mesgTcpip.h"		// TCP/IP routines
#include "mesgMachId.h"		// MachineId class
#include "suppDString.h"	// DynamicString
#include "genFile.h"		// FileUtility

/* ------------------------------------------------ */

#include "genEncrypt.h"
#include "genFile.h"
#include "genManager.h"
#include "genPGP.h"
#include "genTransmit.h"

    /*
     * prefix for server keys
     */

const char *const c_SERVER_KEY_PREFIX = "Agent Tcl server";

    /*
     * security-related messages
     */

static MESSAGE_TYPE protectionTypes[NUM_PROT] =
{
    {PROT_BOTH	    , 1, {MESG_TYPE_BINARY,   /* signed/encrypted message(s) */
	     	          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {PROT_SIGNED    , 1, {MESG_TYPE_BINARY,   /* signed message(s)           */
	     	          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {PROT_ENCRYPTED , 1, {MESG_TYPE_BINARY,   /* encrypted message(s)        */
	     	          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {PROT_NONE	    , 0, {0,         
	     	          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {PROT_KEY	    , 4, {MESG_TYPE_STR,      /* owner keyname               */
			  MESG_TYPE_BYTE,     /* owner authenticated flag    */
			  MESG_TYPE_STR,      /* server keyname              */
			  MESG_TYPE_BYTE,     /* server authenticated flag   */
	     	          0, 0, 0, 0, 0, 0, 0, 0, 0}}
};

MESSAGE_SET protectionMessages = {NUM_PROT, protectionTypes};

/* getEncryptionError

   Purpose: Get the text associated with the last encryption error

     Input: None

    Output: The procedure returns a DynamicString that contains the text 
            associated with the last encryption error.
*/

DynamicString getEncryptionError (void) {
    return (PGP_error());
}

/* makeServerKeyname

   Purpose: Construct the server keyname

     Input: machine = server name and IP address
		      (struct MachineId &)

    Output: The procedure returns a dynamically allocated string that 
	    contains the server keyname.
*/

char *makeServerKeyname (const MachineId &machine)
{
    char *keyname;

	/* assertions on the parameters */

    assert (machine.getIp() != UNKNOWN_IP);

	/* convert the IP address to a string */

    SmartCharPtr ipstring (tcpip_IPToString (machine.getIp()));

	/* make the server keyname (or more precisely a prefix of the   */
	/* server key name that is enough to uniquely identify the key) */

    keyname = new char [strlen(c_SERVER_KEY_PREFIX) + strlen(ipstring) + 4];
    strcpy (keyname, c_SERVER_KEY_PREFIX);
    strcat (keyname, " <");
    strcat (keyname, ipstring);
    strcat (keyname, ">");

	/* return the keyname */

    return (keyname);
}

/* makeFullServerKeyname

   Purpose: Construct the full server keyname

     Input: machine = server name and IP address
		      (struct MachineId &)

    Output: The procedure returns a dynamically allocated string that 
	    contains the server keyname.
*/

char *makeFullServerKeyname (const MachineId &machine)
{
    char *keyname;

	/* get the machine name and convert to lowercase */

    DynamicString machineName = machine.getName();  
    machineName.convertToLowercase ();

	/* get the partial keyname */

    SmartCharPtr partialKeyname (makeServerKeyname (machine));

	/* construct the full keyname */

    keyname = new char [strlen(partialKeyname) + machineName.length() + 4];
    strcpy (keyname, partialKeyname);
    strcat (keyname, " <");
    strcat (keyname, machineName.value());
    strcat (keyname, ">");

	/* done */

    return (keyname);
}

/* secureForServer

   Purpose: Encrypt and/or digitally sign a message for transmission to the
	    agent server

     Input: machine  = destination server
		       (struct MachineId &)

	    security = security characteristics
		       (class BASE_SECURITY &)

	    message  = message that is to be encrypted or digitally signed
		       (class MESSAGE &)

    Output: The procedure returns one of the following codes.

	    MESSAGE_OK        = success
	    MESSAGE_ENCRYPT   = encryption routines failed
	    MESSAGE_SYS_ERROR = system error while attempting to encrypt
 
	    On success "securedMessage" points to the encrypted and/or
	    digitally signed message (or is set to NULL of neither encryption
	    nor digital signatures were spefied in the security 
	    characteristics).
*/

static MESSAGE *constructKeyMessage (const BASE_SECURITY &security)
{
    SEC_IDENTITY owner;
    SEC_IDENTITY server;
    MESSAGE *keyMessage;

	/* get the owner and the machine */

    owner  = security.getOwner ();
    server = security.getMachine ();

	/* construct the key message */

    keyMessage = new MESSAGE (protectionMessages.messages[PROT_KEY]);
    keyMessage -> addString (0, strcpyWithAlloc (owner.m_keyname.value()), e_TRUE);
    keyMessage -> addByte (1, owner.m_authenticated ? 1 : 0);
    keyMessage -> addString (2, strcpyWithAlloc (server.m_keyname.value()), e_TRUE);
    keyMessage -> addByte (3, server.m_authenticated ? 1 : 0);

	/* return the key message */

    return (keyMessage);
};

int secureForServer (const MachineId &machine, const BASE_SECURITY &security, MESSAGE &message, MESSAGE *&securedMessage)
{
    int rc;
    int code;
    int nread;
    char *buffer;
    SEC_IDENTITY owner;
    SEC_IDENTITY server;
    SEC_IDENTITY signer;
    int plaintextFd = -1;
    int encryptedFd = -1;
    UINT_8 protectionFlag;   
    char *serverKeyname = NULL;
    MESSAGE *keyMessage = NULL;
    char *plaintextFilename = NULL;
    char *encryptedFilename = NULL;

	/* assume no encryption or signatures */

    securedMessage = NULL;

	/* determine the encryption type */

    if (security.shouldEncrypt() && security.shouldSign()) {
	protectionFlag = PROT_BOTH;
    } else if (security.shouldEncrypt()) {
	protectionFlag = PROT_ENCRYPTED;
    } else if (security.shouldSign()) {
	protectionFlag = PROT_SIGNED;
    } else {
	protectionFlag = PROT_KEY;
    }

	/* while loop - just something to break out of */

    while (1) {

	    /* are we doing an insecure send? */

	if (protectionFlag == PROT_KEY) {
	    securedMessage = constructKeyMessage (security);
	    rc = MESSAGE_OK;
	    break;  // BREAK OUT TO CLEANUP
	}

	    /* otherwise we need to get the name of the signer */

	signer = security.getSigner (); 

	if (signer.m_keyname.isEmpty() || signer.m_phrase.isEmpty()) {
	    rc = MESSAGE_ENCRYPT;
	    break;  // BREAK OUT TO CLEANUP
	}

	    /* and possibly the server keyname */ 

	if (protectionFlag != PROT_SIGNED) {
	    if ((serverKeyname = makeServerKeyname (machine)) == NULL) {
	  	rc = MESSAGE_ENCRYPT;
		break;  // BREAK OUT TO CLEANUP
	    }
	}

	    /* create two temporary files */

	if ((plaintextFilename = FileUtility::createTemporaryFile ("plain", (char *) NULL)) == NULL) {
	    rc = MESSAGE_SYS_ERROR;
	    break;  // BREAK OUT TO CLEANUP
	}

	if ((encryptedFilename = FileUtility::createTemporaryFile ("encry", (char *) PGP_extension())) == NULL) {
	    rc = MESSAGE_SYS_ERROR;
	    break;  // BREAK OUT TO CLEANUP
	}

	    /* dump the plaintext into the first file -- first open the file */ 

	if ((plaintextFd = FileUtility::open (plaintextFilename, O_WRONLY)) < 0) {
	    rc = MESSAGE_SYS_ERROR;
	    break;  // BREAK OUT TO CLEANUP
	}

	    /* construct and write out the PROT_KEY message (owner/machine) */

	keyMessage = constructKeyMessage (security);
 
	if ((rc = message_send (plaintextFd, *keyMessage)) != MESSAGE_OK) {
	    break;  // BREAK OUT TO CLEANUP
	}

	    /* then write out the message itself */

	if ((rc = message_send (plaintextFd, message)) != MESSAGE_OK) {
	    break;  // BREAK OUT TO CLEANUP
	}	
	 
	    /* encrypt and sign the message */

	if (protectionFlag == PROT_BOTH) {
	    code = PGP_EncryptAndSign (plaintextFilename, encryptedFilename, serverKeyname, signer);
	} else if (protectionFlag == PROT_SIGNED) {
	    code = PGP_Sign (plaintextFilename, encryptedFilename, signer);
	} else if (protectionFlag == PROT_ENCRYPTED) {
	    code = PGP_Encrypt (plaintextFilename, encryptedFilename, serverKeyname);
	} else {
	    abort_with_message ("protectionFlag is PROT_KEY at a point where this should be impossible!");
	    code = -1;	// prevent compiler warning
	}

	if (code < 0) {
	    rc = MESSAGE_ENCRYPT;
	    break;  // BREAK OUT TO CLEANUP
	}	

	    /* read in the encrypted data */

	if ((encryptedFd = FileUtility::open (encryptedFilename, O_RDONLY)) < 0) {
	    rc = MESSAGE_SYS_ERROR;
	    break;  // BREAK OUT TO CLEANUP
	}	

	    /* get the size of the file */

	FileUtility::FileInformation stats;

	if (FileUtility::getFileStats (encryptedFd, stats) != FileUtility::e_OK) {
	    rc = MESSAGE_SYS_ERROR;
	    break;  // BREAK OUT TO CLEANUP
	}	

	    /* allocate the buffer and load from the file */

	buffer = new char [stats.size];

	if ((tcpip_blockingReadn (encryptedFd, buffer, stats.size, nread) != e_TCPIP_OK) 
	    || (nread < (int) stats.size)) {
	    rc = MESSAGE_SYS_ERROR;
	    DELETE_ARRAY_OBJECT(buffer);
	    break;  // BREAK OUT TO CLEANUP
	}	

	    /* construct the secured message */

	securedMessage = new MESSAGE (protectionMessages.messages[protectionFlag]);
	securedMessage -> addBinary (0, stats.size, buffer, e_TRUE);
	break;
    }

	/* cleanup */

    if (plaintextFilename != NULL) {

	unlink (plaintextFilename);
	DELETE_ARRAY_OBJECT(plaintextFilename);

	if (plaintextFd >= 0) {
	    close (plaintextFd);
	}
    }

    if (encryptedFilename != NULL) {

	unlink (encryptedFilename);
	DELETE_ARRAY_OBJECT(encryptedFilename);

	if (encryptedFd >= 0) {
	    close (encryptedFd);
	}
    }


    DELETE_OBJECT(keyMessage);
    DELETE_ARRAY_OBJECT(serverKeyname);
    return (rc);
}  

/* unsecure

   Purpose: Extract the enclosed message from an encrypted/digitally signed
	    message

     Input: set      = message set in which the enclosed message appears
		       (const class MESSAGE_SET &)

	    receiver = security characteristics of the receiver
		       (const BASE_SECURITY &receiver)

	    message  = the encrypted/digitally signed message
		       (const class MESSAGE &)

    Output: The procedure returns one of the following codes.

	    MESSAGE_OK        = success
	    MESSAGE_SYS_ERROR = system error while extracting the enclosed message
	    MESSAGE_DECRYPT   = unable to decrypt the message
	    MESSAGE_INVALID   = the message is invalid
 
	    On success

	    sender           = filled in with the security characteristics of 
			       the sender
		               (class BASE_SECURITY &)

	    unsecuredMessage = pointer to a dynamically allocated MESSAGE
			       instance that contains the enclosed message
			       (class MESSAGE *&)
*/

int unsecure (const MESSAGE_SET &set, const BASE_SECURITY &receiver, MESSAGE &message, BASE_SECURITY &sender, MESSAGE *&unsecuredMessage)
{
    int rc;
    int nwritten;
    UINT_32 length;
    SmartCharPtr buffer;
    char *signer = NULL;
    int plaintextFd = -1;
    int encryptedFd = -1;
    SEC_IDENTITY receiverId;
    SEC_IDENTITY ownerId;
    SEC_IDENTITY machineId;
    MESSAGE *keyMessage = NULL;
    char *plaintextFilename = NULL;
    char *encryptedFilename = NULL;

	/* assume that there is no enclosed message to extract  */

    unsecuredMessage = NULL;

	/* while loop -- simply something to break out of */

    while (1) {

	    /* PROT_NONE and PROT_KEY -- just get the security information */  

	if ((message.getFlag() == PROT_NONE) || (message.getFlag() == PROT_KEY)) {

	    sender.setSignatures (e_FALSE);
	    sender.setEncryption (e_FALSE);
	    sender.setOwnerSigned (e_TRUE);

	    if (message.getFlag() == PROT_KEY) {

		SmartCharPtr ownerKeyname   (message.getString (0));
		UINT_8 ownerClaimed 	= message.getByte (1);
		SmartCharPtr machineKeyname (message.getString (2));
		UINT_8 machineClaimed 	= message.getByte (3);

		ownerId.m_keyname = (char *) ownerKeyname;
		ownerId.m_claimed = ownerClaimed ? e_TRUE : e_FALSE;
		machineId.m_keyname = (char *) machineKeyname;
		machineId.m_claimed = machineClaimed ? e_TRUE : e_FALSE;

	    } else {

		// ownerId and machineId are empty by default
	    }

	    sender.setOwner (ownerId);
	    sender.setMachine (machineId);
	    rc = MESSAGE_OK;
	    break;	// BREAK OUT TO CLEANUP
	}

	    /* PROT_BOTH, PROT_SIGNED and PROT_ENCRYOTED -- decrypt and verify */

	if (message.getFlag() == PROT_BOTH) {
	    sender.setSignatures (e_TRUE);
	    sender.setEncryption (e_TRUE);
	} else if (message.getFlag() == PROT_SIGNED) {
	    sender.setSignatures (e_TRUE);
	    sender.setEncryption (e_FALSE);
	} else if (message.getFlag() == PROT_ENCRYPTED) {
	    sender.setSignatures (e_FALSE);
	    sender.setEncryption (e_TRUE);
	} else {
	    rc = MESSAGE_INVALID;
	    break;	// BREAK OUT TO CLEANUP
	}

	    /* get the receiver keyname */

	receiverId = receiver.getSigner ();

	if (receiverId.m_keyname.isEmpty() || receiverId.m_phrase.isEmpty()) {
	    rc = MESSAGE_DECRYPT;
	    break;	// BREAK OUT TO CLEANUP
	}

	    /* create two temporary files -- encrypted and plaintext */

	if ((plaintextFilename = FileUtility::createTemporaryFile ("plain", (char *) NULL)) == NULL) {
	    rc = MESSAGE_SYS_ERROR;
	    break;	// BREAK OUT TO CLEANUP
	}

	if ((encryptedFilename = FileUtility::createTemporaryFile ("encry", (char *) PGP_extension())) == NULL) {
	    rc = MESSAGE_SYS_ERROR;
	    break;	// BREAK OUT TO CLEANUP
	}

	    /* write the encrypted data into the encrypted file */

	if ((encryptedFd = FileUtility::open (encryptedFilename, O_CREAT | O_TRUNC | O_WRONLY, 0600)) < 0) {
	    rc = MESSAGE_SYS_ERROR;
	    break;	// BREAK OUT TO CLEANUP
	}

	buffer = message.getBinary (0, length);

	if ((tcpip_blockingWriten (encryptedFd, buffer, length, nwritten) != e_TCPIP_OK) 
	    || (nwritten < (int) length)) {
	    rc = MESSAGE_SYS_ERROR;
	    break;	// BREAK OUT TO CLEANUP
	}

	    /* decrypt the encrypted file */

	if (message.getFlag() == PROT_ENCRYPTED) {

	    if (PGP_Decrypt (encryptedFilename, plaintextFilename, receiverId) < 0) {
		rc = MESSAGE_DECRYPT;
		break;	// BREAK OUT TO CLEANUP
	    }

	} else {

	    if ((signer = PGP_DecryptSigned (encryptedFilename, plaintextFilename, receiverId)) == NULL) {
		rc = MESSAGE_DECRYPT;
		break;	// BREAK OUT TO CLEANUP
	    }
	}

	    /* open the decrypted file */ 

	if ((plaintextFd = FileUtility::open (plaintextFilename, O_RDONLY)) < 0) {
	    rc = MESSAGE_SYS_ERROR;
	    break;	// BREAK OUT TO CLEANUP
	}

	if ((keyMessage = message_receive (plaintextFd, protectionMessages)) == NULL) {
	    rc = MESSAGE_INVALID;
	    break;	// BREAK OUT TO CLEANUP
	}

	if (keyMessage -> getFlag() != PROT_KEY) {
	    rc = MESSAGE_INVALID;
	    break;	// BREAK OUT TO CLEANUP
	}


	SmartCharPtr ownerKeyname	(keyMessage -> getString (0));
	UINT_8 ownerClaimed 		= keyMessage -> getByte (1);
	SmartCharPtr machineKeyname	(keyMessage -> getString (2));
	UINT_8 machineClaimed 		= keyMessage -> getByte (3);

	if ((char *) ownerKeyname == NULL) {
	    rc = MESSAGE_INVALID;
	    break;	// BREAK OUT TO CLEANUP
	}
	 
	ownerId.m_keyname   = (char *) ownerKeyname;
	ownerId.m_claimed   = ownerClaimed ? e_TRUE : e_FALSE;
	machineId.m_keyname = (char *) machineKeyname;
	machineId.m_claimed = machineClaimed ? e_TRUE : e_FALSE;

	    /* check everything */

	if (message.getFlag() != PROT_ENCRYPTED) {

	    if (signer == ownerId.m_keyname) {
		ownerId.m_authenticated = e_TRUE;
		sender.setOwnerSigned (e_TRUE);
	    } else if (machineId.m_keyname.isEmpty() || (signer != machineId.m_keyname)) {
		rc = MESSAGE_INVALID;
		break;	// BREAK OUT TO CLEANUP
	    } else {
		machineId.m_authenticated = e_TRUE;
	        sender.setOwnerSigned (e_FALSE);
	    }

	} else {

	    sender.setOwnerSigned (e_FALSE);
	}
 
	    /* finish filling in the security information */

	sender.setOwner (ownerId);
	sender.setMachine (machineId);

	    /* get the enclosed message out of the file */ 

        if ((unsecuredMessage = message_receive (plaintextFd, set)) == NULL) {
	    rc = MESSAGE_INVALID;
        } else {
	    rc = MESSAGE_OK;
        }

	break;
    }

	/* cleanup */

    if (plaintextFilename != NULL) {

	unlink (plaintextFilename);
	DELETE_ARRAY_OBJECT(plaintextFilename);

	if (plaintextFd >= 0) {
	    FileUtility::close (plaintextFd);
	}
    }

    if (encryptedFilename != NULL) {

	unlink (encryptedFilename);
	DELETE_ARRAY_OBJECT(encryptedFilename);

	if (encryptedFd >= 0) {
	    FileUtility::close (encryptedFd);
	}
    }

    DELETE_OBJECT(keyMessage);
    DELETE_ARRAY_OBJECT(signer);
    return (rc);
}
