/* Agent Tcl
   Bob Gray
   10 March 1995

   genError.cc

   This file implements class ERROR which provides an error logging
   facility.  ERROR is based on the error logging facilities in
   "UNIX Network Programming" by W. Richard Stevens [Prentice-Hall, 1990].

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "suppStrings.h"
#include "mesgTcpip.h"
#include "genError.h"
#include "genFile.h"		// FileUtility

#ifndef STDERR_FILENO
#define STDERR_FILENO 2
#endif

#ifdef FIX_LATER

    /*
     * 1. Fix the counter so that we are couting locks per file rather than
     *    locks per file per ErrorLog instance
     *
     * 2. Get the exit out of the constructor
     */

#endif

/* ERROR::openLog and ERROR::closeLog

   Purpose: These procedures open and close the error log.
*/ 

int ERROR::openLog (void)
{
	/* if we are using stderr, just setup its fd */

    if (useStderr) {
	logfd = STDERR_FILENO;
	return (1);
    }

	/* otherwise we have to open the log file */

    if ((logfd = FileUtility::open (logname, O_RDWR | O_CREAT, 0600)) < 0) {
	return (0);
    }

    (void) FileUtility::acquireFileLock (logfd);
    lseek (logfd, 0, SEEK_END);
    return (1);
}

void ERROR::closeLog (void)
{
    if (!useStderr) {
	(void) FileUtility::releaseFileLock (logfd);
	FileUtility::close (logfd);
    }
}

/* ERROR::ERROR and ERROR::~ERROR

   Purpose: These procedures are the constructors and destructor for class 
	    ERROR.
*/

ERROR::ERROR (const ERROR& errorLog):
    count	(0),
    useStderr	(errorLog.useStderr),
    logname	(strcpyWithAlloc (errorLog.logname))
{
    //empty
}
   
ERROR::ERROR (void):
    count	(0),
    useStderr	(e_TRUE) ,
    logname	((char *) NULL)
{
    //empty
}

ERROR::ERROR (char *p_logname):
    count	(0),
    useStderr   (e_FALSE),
    logname	(strcpyWithAlloc (p_logname))
{
	/* assertions */ 

    assert (logname != NULL);

	/* make sure the error log is accessible */

    if (!openLog()) {
	fprintf (stderr, "ERROR: unable to open error log \"%s\"\n", logname);
	exit (1);
    }

    closeLog ();
}

ERROR::~ERROR ()
{
    DELETE_ARRAY_OBJECT(logname);
}

/* ERROR::operator=

   Purpose: This procedure is the assignment operator for class ERROR.
*/

ERROR& ERROR::operator= (const ERROR& errorLog)
{
    if (this != &errorLog) {
	DELETE_ARRAY_OBJECT(logname);
	logname = strcpyWithAlloc (errorLog.logname);
	useStderr = errorLog.useStderr;
    }

    return (*this);
}

/* ERROR::error_date

   Purpose: This procedure logs the date and time.
*/

void ERROR::error_date (void)
{
    time_t current_time = time (NULL);
    char * ascii_time   = ctime (&current_time);
    int length = strlen(ascii_time);

    if (ascii_time[length - 1] == '\n') {
	length -= 1;
    }

    int nwritten;
    tcpip_blockingWriten (logfd, ascii_time, length, nwritten);
    tcpip_blockingWriten (logfd, " - ", 3, nwritten);
}
 
/* ERROR::error_app_cont

   Purpose: This procedure logs a NON-FATAL application error.
*/

void ERROR::error_app_cont (char *message)
{
    if (openLog()) {
	int nwritten;
	error_date ();
	tcpip_blockingWriten (logfd, message, strlen(message), nwritten);
	tcpip_blockingWriten (logfd, "\n", 1, nwritten);
	closeLog ();
    }
}

void ERROR::error_app_cont (char **argv)
{
    if (openLog()) {

	int nwritten;

	error_date ();
	
	while (*argv != NULL) {
	    tcpip_blockingWriten (logfd, *argv, strlen(*argv), nwritten);
	    argv += 1;
	}

	tcpip_blockingWriten (logfd, "\n", 1, nwritten);
	closeLog ();
    }
}

/* ERROR::error_app_quit

   Purpose: This procedure logs a fatal application error.
*/

void ERROR::error_app_quit (char *message)
{
    error_app_cont (message);
    exit (1);
}

void ERROR::error_app_quit (char **argv)
{
    error_app_cont (argv);
    exit (1);
}

/* ERROR::error_sys_cont

   Purpose: This procedure logs a NON-FATAL system error.
*/

void ERROR::error_sys_cont (char *message)
{
    char temp[32];
 
    if (openLog ()) {
	int nwritten;
	sprintf (temp, "%d", errno);
	error_date ();
	tcpip_blockingWriten (logfd, message, strlen(message), nwritten); 
	tcpip_blockingWriten (logfd, " (errno = ", 10, nwritten);
	tcpip_blockingWriten (logfd, temp, strlen(temp), nwritten);
	tcpip_blockingWriten (logfd, ")\n", 2, nwritten);
	closeLog ();
    }
}

void ERROR::error_sys_cont (char **argv)
{
    char temp[32];
 
    if (openLog ()) {

	int nwritten;
	sprintf (temp, "%d", errno);

	    /* date */ 

	error_date ();

	    /* message pieces */

	while (*argv != NULL) {
	    tcpip_blockingWriten (logfd, *argv, strlen(*argv), nwritten);
	    argv += 1;
	}

	    /* errno */

	sprintf (temp, "%d", errno);
	tcpip_blockingWriten (logfd, " (errno = ", 10, nwritten);
	tcpip_blockingWriten (logfd, temp, strlen(temp), nwritten);
	tcpip_blockingWriten (logfd, ")\n", 2, nwritten);
	closeLog ();
    }
}

/* ERROR::error_sys_quit

   Purpose: This procedure logs a FATAL system error.
*/

void ERROR::error_sys_quit (char *message)
{
    error_sys_cont (message);
    exit (1);
}

void ERROR::error_sys_quit (char **argv)
{
    error_sys_cont (argv);
    exit (1);
}
