/* Agent Tcl
   Bob Gray
   1 July 1995

   locks.cc

   This file implements the routines that set and release file locks.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "suppStrings.h"
#include "suppDString.h"
#include "suppFileTable.h"
#include "genDefaults.h"
#include "genError.h"
#include "genFile.h"

    /* declaration for flock */

#ifdef NO_FCNTL_LOCKING
extern "C" int flock (int, int);
#endif

/* FileUtility::getFileStats

   Purpose: Get information about a file

     Input: filename = name of the file
	               (const char *)    

    Output: The procedure returns -1 on error.  Otherwise the procedure
	    returns 0 and fills in the FileInformation structure.
*/

static void fillInFileInformation (const struct stat &statBuf, FileUtility::FileInformation &fileInformation)
{
	/* size */

    fileInformation.size = (UINT_32) statBuf.st_size;

	/* permission bits */

    int flags  = 0;
    if (statBuf.st_mode & S_IRUSR) { flags |= FileUtility::e_USER_READ  ; }
    if (statBuf.st_mode & S_IWUSR) { flags |= FileUtility::e_USER_WRITE ; }
    if (statBuf.st_mode & S_IXUSR) { flags |= FileUtility::e_USER_EXEC  ; }
    if (statBuf.st_mode & S_IRGRP) { flags |= FileUtility::e_GROUP_READ ; }
    if (statBuf.st_mode & S_IWGRP) { flags |= FileUtility::e_GROUP_WRITE; }
    if (statBuf.st_mode & S_IXGRP) { flags |= FileUtility::e_GROUP_EXEC ; }
    if (statBuf.st_mode & S_IROTH) { flags |= FileUtility::e_OTHER_READ ; }
    if (statBuf.st_mode & S_IWOTH) { flags |= FileUtility::e_OTHER_WRITE; }
    if (statBuf.st_mode & S_IXOTH) { flags |= FileUtility::e_OTHER_EXEC ; }
    fileInformation.permissions = flags;

	/* type */

    if (S_ISLNK(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_LINK;
    } else if (S_ISREG(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_REGULAR;
    } else if (S_ISDIR(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_DIRECTORY;
    } else if (S_ISCHR(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_CHAR_DEVICE;
    } else if (S_ISBLK(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_BLOCK_DEVICE;
    } else if (S_ISFIFO(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_FIFO;
    } else if (S_ISSOCK(statBuf.st_mode)) {
	fileInformation.type = FileUtility::e_SOCKET;
    } else {
	fileInformation.type = FileUtility::e_UNKNOWN;
    }
}

FileUtility::FileError FileUtility::getFileStats (const char *filename, FileInformation &fileInformation)
{
    struct stat statBuf;

	/* get the statistics */

#ifdef FIX_LATER
	/* 
	 * we need more error codes to distinguish among the things that
	 * might have gone wrong with the call to stat()
	 */
#endif

    if (stat (filename, &statBuf) < 0) {
	return (e_NONEXISTENT);
    }

	/* fill in the file information structure */

    fillInFileInformation (statBuf, fileInformation);
    return (e_OK);
}

/* FileUtility::getFileStats

   Purpose: Get information about a file

     Input: fd = file descriptor 
		 (const char *)    

    Output: The procedure returns -1 on error.  Otherwise the procedure
	    returns 0 and fills in the FileInformation structure.
*/

FileUtility::FileError FileUtility::getFileStats (int fd, FileInformation &fileInformation)
{
    struct stat statBuf;

	/* get the statistics */

#ifdef FIX_LATER
	/* 
	 * we need more error codes to distinguish among the things that
	 * might have gone wrong with the call to fstat()
	 */
#endif
	
    if (fstat (fd, &statBuf) < 0) {
	return (e_INVALID_FD);
    }

	/* fill in the file information structure */

    fillInFileInformation (statBuf, fileInformation);
    return (e_OK);
}

/* FileUtility::open

   Purpose: Open a file reliably -- i.e., do not die on interrupt.

     Input: filename = name of the file
		       (const char *)

	    flags    = OR'ed combination of O_TRUNC, O_CREAT, etc.
		       (int)

	    mode     = permission bits
		       (mode_t)

    Output: The procedure returns a file descriptor on success and -1 otherwise.
*/

int FileUtility::open (const char *filename, int flags, mode_t mode)
{
    int fd;

	// open the file

    do {
	fd = ::open ((char *) filename, flags, mode);
    } while ((fd < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	// record the file descriptor if file recording is on

    if (fd >= 0) {
	if (FileTable::isFileRecordingOn()) {
	    FileTable::add (fd, FileTable::e_FILE);
	}
    }

	// done

    return (fd);
}

/* FileUtility::dup

   Purpose: Duplicate a file descriptor

     Input: fd = the file descriptor
                 (int)

    Output: The function returns a new file descriptor (that refers to the
            same underlying file device as the given file descriptor) on 
	    success.  The function returns -1 on failure.
*/

int FileUtility::dup (int fd)
{
    int newFd;

	// duplicate the file descriptor

    newFd = ::dup (fd);

	// add the new file descriptor to the file table

    if (newFd >= 0) {
	if (FileTable::isFileRecordingOn ()) {
	    FileTable::dup (fd, newFd);
	}
    }

	// done

    return (newFd);
}

/* FileUtility::close

   Purpose: Close a file

     Input: fd = the file descriptor
		 (int)

    Output: The function closes the file and returns e_OK if the file 
	    descriptor referred to a valid open file.  Otherwise the 
	    function returns e_INVALID_FD.
*/

FileUtility::FileError FileUtility::close (int fd) 
{
    FileError rc;

	// remove the file descriptor from the table

    if (FileTable::isFileRecordingOn()) {
	FileTable::remove (fd);
    }

	// close the file descriptor

    if (::close (fd) < 0) {
	rc = e_INVALID_FD;
    } else {
	rc = e_OK;
    }

    return (rc);
}

/* FileUtility::acquireFileLock

   Purpose: Acquire a file lock

     Input: fd = file descriptor      
		 (int)
  
    Output: The procedure returns e_OK on success and e_FAILURE on error.
*/

FileUtility::FileError FileUtility::acquireFileLock (int fd)
{
    int rc;
#ifndef NO_FCNTL_LOCKING
    struct flock lockInfo;
#endif

    do {
#ifdef NO_FCNTL_LOCKING
	rc = flock (fd, LOCK_EX);
#else
	lockInfo.l_type   = F_WRLCK;
	lockInfo.l_start  = 0;
	lockInfo.l_whence = SEEK_SET;
	lockInfo.l_len    = 0;
	rc = fcntl (fd, F_SETLKW, &lockInfo);
#endif
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

    if (rc < 0) {
	return (e_FAILURE);
    }

    return (e_OK);
}

/* FileUtility::releaseFileLock

   Purpose: Release a file lock

     Input: fd = file descriptor      
		 (int)
  
    Output: The procedure returns e_OK on success and e_UNKNOWN on error.
*/

FileUtility::FileError FileUtility::releaseFileLock (int fd)
{
    int rc;
#ifndef NO_FCNTL_LOCKING
    struct flock lockInfo;
#endif
   
    do {
#ifdef NO_FCNTL_LOCKING
	rc = flock (fd, LOCK_UN);
#else
	lockInfo.l_type  = F_UNLCK;
	lockInfo.l_start  = 0;
	lockInfo.l_whence = SEEK_SET;
	lockInfo.l_len    = 0;
	rc = fcntl (fd, F_SETLKW, &lockInfo);
#endif
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

    if (rc < 0) {
	return (e_FAILURE);
    }

    return (e_OK);
}

FileUtility::FileError FileUtility::fileRedirect (void)
{
    int fd;

	/* close all open files */

#ifdef FIX_LATER
	/* we should not be using a hardcoded constant (i..e, 4096) */
#endif

    for (fd = 0; fd < 4096; fd++) {
        close (fd);
    }
  
    errno = 0;

        /* redirect stdin */

    if ((fd = FileUtility::open ("/dev/null", O_RDONLY)) < 0) {
        return (e_FAILURE);
    }

    if (fd != 0) {
        dup2 (fd, 0);
        close (fd);
    }

        /* redirect stdout and stderr */

    if ((fd = FileUtility::open ("/dev/null", O_WRONLY)) < 0) {
        return (e_FAILURE);
    }

    if (fd != 1) {
        dup2 (fd, 1);
        close (fd);
    }

    dup2 (1, 2);
    return (e_OK);
}

/* FileUtility::fileTildeSubst

   Purpose: Replace the leading ~ (and following characters) in a filename with 
	    the home directory of the given user 

     Input: filename = filename in which to replace ~
		       (char *)

    Output: The procedure returns a dynamically allocated string that
	    contains (1) an exact copy of the given filename if the filename
	    did *not* contain a ~ or (2) a copy in which the ~ has been
	    replaced appropriately if the filename does contain a ~.  On error
	    the procedure returns NULL.

      Note: This is code is based on the corresponding procedure from Tcl 7.5
	    which is (C) 1996, Sun Microsystems, Inc.  The license for Tcl
	    7.5 allows free distribution of the code as long as copyright
	    notices are maintained.
*/

char *FileUtility::fileTildeSubst (const char *filename)
{
    const char *ep;
    char *dir;
    char *username;
    char *newFilename;
    struct passwd *pwPtr;

	/* assertions on the parameters */

    assert (filename != NULL);

	/* return an exact copy if there is no tilde (~) */

    if (filename[0] != '~') {
	return (strcpyWithAlloc (filename));
    }

	/* a tilde (~) with no username should be replaced with the home */
	/* directory of the current user		 	         */

    if ((filename[1] == '/') || (filename[1] == '\0')) {

	if ((dir = getenv("HOME")) == NULL) {
	    newFilename = ((char *) NULL);
	} else {
	    newFilename = new char [strlen(dir) + strlen(filename) + 1];
	    strcpy (newFilename, dir);
	    strcat (newFilename, &filename[1]);
	}

	return (newFilename);
    }

	/* otherwise we have a tilde (~) followed a username which must */
	/* be replaced with the home directory of that user             */

    for (ep = &filename[1]; (*ep != 0) && (*ep != '/'); ep++) {
	/* Null body;  just find end of name. */
    }

    username                = strcpyWithAlloc (filename);
    username[ep - filename] = '\0';

    if ((pwPtr = getpwnam (username)) == NULL) {
	newFilename = (char *) NULL;
    } else {
	newFilename = new char [strlen(pwPtr -> pw_dir) + strlen(ep) + 1];
	strcpy (newFilename, pwPtr -> pw_dir);
	strcat (newFilename, ep);
    }

    endpwent ();
    DELETE_ARRAY_OBJECT(username);
    return (newFilename);
}

/* FileUtility::createTemporaryFile

   Purpose: Create a temporary file

     Input: prefix    = prefix for the temporary filename
		        (const char *)

	    extension = extension for the temporary filename
			(const char *)
 
    Output: The procedure returns NULL if it is unable to create the
	    temporary file.  Otherwise the pointer returns a pointer to 
	    a dynamically allocated string that contains the filename (the
	    procedure creates the file with access permission 0600 and 
	    truncates the file to 0 length).
*/

char *FileUtility::createTemporaryFile (const char *prefix, const char *extension)
{
    int fd;
    char *filename;   
    const char *dir;
    char *extendedFilename;

	/* assertions on the parameters */

    assert (prefix != NULL);

	/* directory in which to create the filename */

    if ((dir = getenv (AGENT_TEMP_ENV)) == NULL) {
	dir = DEFAULT_AGENT_TEMP;
    }    

	/* get a name for the temporary file */

    if ((filename = tempnam (dir, prefix)) == NULL) {
	return ((char *) NULL);
    }    

    if (extension == NULL) {

	extendedFilename = filename;

    } else {

	extendedFilename = new char [strlen(filename) + strlen(extension) + 2];
	strcpy (extendedFilename, filename);
	strcat (extendedFilename, ".");
	strcat (extendedFilename, extension);
	DELETE_ARRAY_OBJECT(filename);
    }
	
	/* create the temporary file */

    if ((fd = FileUtility::open (extendedFilename, O_WRONLY | O_TRUNC | O_CREAT, 0600)) < 0) {
	DELETE_ARRAY_OBJECT(extendedFilename);
	return ((char *) NULL);
    }

    close (fd);

	/* return the filename */

    return (extendedFilename);
}

/* FileUtility::createDirectory

   Purpose: Create a directory

     Input: directory   = name of the directory
			  (const char *)

            permissions = permissions for the new directory
			  (int, an OR'ed combination of FileUtility::FilePermissions)

    Output: The procedure returns one of the following error codes:

            e_OK               = the directory was created successfully
	    e_NOT_DIRECTORY    = a file with the given name already exists, 
			         but is not a directory
	    e_DIRECTORY_EXISTS = the directory already exists
	    e_INACCESIBLE_PATH = no access to one of the parent directories on 
                                 the path
	    e_INVALID_PATH     = one of the parent directorues on the path does
				 not exist
	    e_FAILURE	       = any other error

	    e_DIRECTORY_BAD_PERMISSIONS =  directory exists but has the wrong
                                           permissions
*/

FileUtility::FileError FileUtility::createDirectory (const char *directory, int permissions)
{
    mode_t permissionBits = 0;

	/*
	 * construct the permission bits
	 */

    if (permissions & e_USER_READ  ) { permissionBits |= S_IRUSR; }
    if (permissions & e_USER_WRITE ) { permissionBits |= S_IWUSR; }
    if (permissions & e_USER_EXEC  ) { permissionBits |= S_IXUSR; }
    if (permissions & e_GROUP_READ ) { permissionBits |= S_IRGRP; }
    if (permissions & e_GROUP_WRITE) { permissionBits |= S_IWGRP; }
    if (permissions & e_GROUP_EXEC ) { permissionBits |= S_IXGRP; }
    if (permissions & e_OTHER_READ ) { permissionBits |= S_IROTH; }
    if (permissions & e_OTHER_WRITE) { permissionBits |= S_IWOTH; }
    if (permissions & e_OTHER_EXEC ) { permissionBits |= S_IXOTH; }

	/*
	 * attempt to create the directory
	 */

    if (mkdir ((char *) directory, permissionBits) < 0) {

	if (errno == EEXIST) {

	    FileInformation stats;
	   
	    if (getFileStats (directory, stats) != e_OK) {
		return (e_FAILURE);
	    }

	    if (stats.type != e_DIRECTORY) {
		return (e_NOT_DIRECTORY);
	    }

	    if (stats.permissions != permissions) {
	   	return (e_DIRECTORY_BAD_PERMISSIONS);
	    }

	    return (e_DIRECTORY_EXISTS);

	} else if (errno == EACCES) {

	    return (e_INACCESIBLE_PATH);

	} else if (errno == ENOENT) {

	    return (e_INVALID_PATH);

	} else {

	    return (e_FAILURE);
	}
    }

	/*
	 * done
	 */

    return (e_OK);
}
