/* Agent Tcl
   Bob Gray
   1 July 1995

   genFile.h

   Thie file defines various file utilities.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _GEN_FILE_H
#define _GEN_FILE_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _GEN_UTILITY_H
#include "genUtility.h"		// UINT_32
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"		// BOOLEAN
#endif

    /* forward declarations */

class ERROR;
class DynamicString;

    /* file utilities */

class FileUtility
{
    public:

	enum FileError {
	    e_NONEXISTENT,
	    e_OK,
	    e_INVALID_FD,
	    e_NOT_DIRECTORY,
	    e_DIRECTORY_EXISTS,
	    e_DIRECTORY_BAD_PERMISSIONS,
	    e_INACCESIBLE_PATH,
	    e_INVALID_PATH,
	    e_FAILURE
	};

	enum FilePermissions {
	    e_USER_READ		= 0x1,
	    e_USER_WRITE	= 0x2,
	    e_USER_EXEC		= 0x4,
	    e_GROUP_READ	= 0x8,
	    e_GROUP_WRITE	= 0x10,
	    e_GROUP_EXEC	= 0x20,
	    e_OTHER_READ	= 0x40,
	    e_OTHER_WRITE	= 0x80,
	    e_OTHER_EXEC	= 0x100
	};


	enum FileType {
	    e_LINK		= 0x1,
	    e_REGULAR		= 0x2,
	    e_DIRECTORY		= 0x3,
	    e_CHAR_DEVICE	= 0x4,
	    e_BLOCK_DEVICE	= 0x5,
	    e_FIFO		= 0x6,
	    e_SOCKET		= 0x7,
	    e_UNKNOWN		= 0x8
	};

	struct FileInformation {
	    UINT_32 size;		 	// size of the file
	    int permissions;			// permissions bits
	    FileType type;			// type of the file
	};

	    /* access tests */

	static BOOLEAN isUserAccessible (int permissions) {
	    return (
		((permissions & e_USER_READ) || 
		(permissions & e_USER_WRITE) ||
		(permissions & e_USER_EXEC))
		? e_TRUE : e_FALSE
	    );
	}

	static BOOLEAN isGroupAccessible (int permissions) {
	    return (
		((permissions & e_GROUP_READ) || 
		(permissions & e_GROUP_WRITE) ||
		(permissions & e_GROUP_EXEC))
		? e_TRUE : e_FALSE
	    );
	}

	static BOOLEAN isOtherAccessible (int permissions) {
	    return (
		((permissions & e_OTHER_READ) || 
		(permissions & e_OTHER_WRITE) ||
		(permissions & e_OTHER_EXEC))
		? e_TRUE : e_FALSE
	    );
	}

	    /* create a directory */

	static FileError createDirectory 
	    (const char *directory, int permissions);

	    /* open a file reliably -- i.e., do not die on an interrupt */

#ifdef FIX_LATER
	    /*
	     * open should take a FilePermissions (or int) argument rather
	     * than a mode_t
	     */
#endif

	static int open (const char *filename, int flags, mode_t mode = 0600);

	    /* create a temporary file */

	static char *createTemporaryFile 
	    (const char *prefix, const char *extension);

	    /* duplicate a file descriptor */

	static int dup (int fd);

	    /* close a file descriptor */

	static FileError close (int fd);

	    /* acquire or release a file lock */

	static FileError acquireFileLock (int fd);
	static FileError releaseFileLock (int fd);

	    /* get information about a file */

	static FileError getFileStats 
	    (int fd, FileInformation &fileInformation);
	static FileError getFileStats 
	    (const char *filename, FileInformation &fileInformation);

	    /* close all open files except for stdin, stderr and stdout */
	    /* and redirect stdin, stderr and stdout to /dev/null       */

	static FileError fileRedirect (void);

	    /* replace a ~ in a filename with the user's home directory */

	static char *fileTildeSubst (const char *filename);
};

    /* ensure that a file is closed */

class FileCloser
{
public:

	/* constructor and destructor */

    FileCloser (int fd):
	m_fd (fd)
    {
	// empty
    }

   ~FileCloser () {
	FileUtility::close (m_fd);
    }

private:

	/* file descriptor */

    int m_fd;

	/* unimplemented */

    FileCloser (const FileCloser&);
    FileCloser& operator= (const FileCloser&);
};

#endif
