/* Agent Tcl
   Bob Gray
   10 January 1996

   genLocation.cc

   This file implements class LOCATION which keeps track of an agent's
   location within the network.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "mesgTcpip.h"			// TCP/IP routines
#include "agentAgentId.h"		// AgentId
#include "agentPermit.h"		// Permit
#include "genAgent.h"
#include "genLocation.h"
#include "genManager.h"
#include "genServerInterface.h"		// ServerInterface::SubmissionTypes
#include "genUtility.h"

    /* number of seconds when getting IP address from server */

const int IP_SECONDS = 2;
 
/* LOCATION::startingBootstrap

   Purpose: Bootstrap the location system for agents that are *starting* 
            on the local machine - e.g., figure out the full and IP address 
            of the actual machine.

     Input: None

    Output: The procedure returns 0 on success and -1 otherwise.
*/

int LOCATION::startingBootstrap (void)
{
    struct timeval stop;   /* stop time            */
    struct timeval delay;  /* maximum server delay */

	/* RESTRICT structure */

    RESTRICT *restrict = agent -> get_restrict ();
    assert (restrict != NULL);

	/* default delay */

    delay.tv_sec  = IP_SECONDS;
    delay.tv_usec = 0;

	/* add the restriction */

    Permit permit;
    permit.setWallLimit (delay);
    RestrictMgr restrictMgr (*restrict, permit, (TimerTrap *) NULL);
    restrict -> getWallLimit (stop);

	/* try to get the host name and ip address from the agent server; */
	/* otherwise resort to the "evil" gethostname and gethostbyaddr   */
	/* functions						          */

    if (agent -> agentIp (delay, machine) != AGENT_OK) {

	char *hostname;

	if ((hostname = tcpip_getActualHostname ()) == NULL) {

  	    abort_with_message ("fatal error: LOCATION::startingBootstrap was unable to get name of local host!");

	} else {

	    DynamicString hostnameString (hostname, BufferOwnership::e_TAKE);
	    machine.setName (hostnameString);

	    if (tcpip_getIP (machine, stop) != e_TCPIP_OK) {
		DynamicString string;
		string.append ("fatal error: LOCATION::startingBootstrap was unable to get IP address of local host \"");
		string.append (hostname);
		string.append ("\"");
		abort_with_message (string.value());
	    }
	}
    }

	/* actual IP address */ 

    tcpip_setActualIPAndHostname (machine.getIp(), machine.getName().value());
    return (0);
}

/* LOCATION::arrivingBootstrap

   Purpose: Bootstrap the location system for agents that are *arriving* 
            from another machine - e.g., figure out the full and IP address 
            of the actual machine.

     Input: None

    Output: The procedure returns 0 on success and -1 otherwise.
*/

int LOCATION::arrivingBootstrap 
	(ServerInterface::SubmissionTypes submissionType,
	 const AgentId &newRoot,
	 const AgentId &newRootHome,
	 const AgentId &newLocal,
	 const AgentId &newLocalHome,
	 const DynamicString &directory)
{
	/* set the local, root and home identifications */

    if ((submissionType == ServerInterface::e_SUB_JUMP) ||
	(submissionType == ServerInterface::e_SUB_AGLET_JUMP)) {

	    /* server, registered and home */

	server     = AGENT::e_INCOMING_AGENT;
	registered = AGENT::e_REGISTERED;
	home       = AGENT::e_AWAY_FROM_HOME;

            /* see if we are a root agent */

	if ((newRootHome.getServerIp() == newLocalHome.getServerIp()) && (newRootHome.getId() == newLocalHome.getId())) {

	    rootId     = newLocal;
	    rootHomeId = newLocalHome;
	    localId    = newLocal;
	    homeId     = newLocalHome;
	    root       = AGENT::e_ROOT_AGENT;

	} else {

	    rootId     = newRoot;
	    rootHomeId = newRootHome;
	    localId    = newLocal;
	    homeId     = newLocalHome;
	    root       = AGENT::e_CHILD_AGENT;
	}

    } else {

	    /* register, server, root and home */

        server     = AGENT::e_INCOMING_AGENT;
        registered = AGENT::e_REGISTERED;
        home       = AGENT::e_AT_HOME;
        root       = AGENT::e_CHILD_AGENT;

	    /* update local and root identifications */

        rootId     = newRoot;
        rootHomeId = newRootHome;
        localId    = newLocal;
        homeId     = newLocal;
    }

	/* set the actual name and IP address of the local machine */

    machine.setName (localId.getServerName());
    machine.setIp   (localId.getServerIp());
    tcpip_setActualIPAndHostname (machine.getIp(), machine.getName().value());

	/* set the temporary agent directory */

#ifdef FIX_LATER 
        /*
	 * we are assuming that (1) the base directory name does not have
	 * a slash on the end already and (2) a backslash (/) is the correct
	 * path separator for the platform that we are on 
	 */
#endif

    char buffer[16]; 
    tempDirectory = directory;
    tempDirectory.append ("/");
    fastIntegerToAscii ((int) localId.getId(), buffer);
    tempDirectory.append (buffer);
    return (0);
}

/* LOCATION::LOCATION

   Purpose: This procedure is the constructor for class LOCATION.

     Input: p_agent = agent associated with this LOCATION insance
                      (class AGENT *)
*/

LOCATION::LOCATION (AGENT *p_agent)
{
	/* assertions on the parameters */

    assert (p_agent != NULL);

	/* base initialization */ 
   
    agent       = p_agent; 
    server      = AGENT::e_LOCAL_AGENT;
    root        = AGENT::e_ROOT_AGENT;
    home        = AGENT::e_AT_HOME;
    registered  = AGENT::e_NOT_REGISTERED;

	/* localhost (IP address 127.0.0.1) */

    localhost = tcpip_getLocalhostIP ();
}

/* LOCATION::turnIntoRoot

   Purpose: Turn the agent into a root agent

     Input: None

    Output: The procedure turns the agent into a root agent.
*/

void LOCATION::turnIntoRoot (void)
{
    if (root != AGENT::e_ROOT_AGENT) {

	root   = AGENT::e_ROOT_AGENT;
	rootId = localId;

	if (home == AGENT::e_AT_HOME) {
	    rootHomeId = homeId;
	}

	changeNotification();	// notify derviced class that there has been a changed
    }
}

/* LOCATION::updateName

   Purpose: Update the name of the agent

     Input: name = the new name

    Output: The procedure updates the name.
*/

void LOCATION::updateName (char *name)
{
    localId.setName (name);

    if (home == AGENT::e_AT_HOME) {
	homeId.setName (name);
    }

    if (root == AGENT::e_ROOT_AGENT) {

	rootId.setName (name);

	if (home == AGENT::e_AT_HOME) {
	    rootHomeId.setName (name);
	}
    }

    changeNotification();    // notify derived class that there has been a changed
}

/* LOCATION::update

   Purpose: This procedure updates the location information for a root agent
	    that did not arrive via the server.  

     Input: id = new agent identification
		 (const AgentId &)

    Output: The procedure updates the local and root identifications and marks
	    the agent as a ROOT.
*/

void LOCATION::update (const AgentId &id)
{
	// agent is now registered and is now a root

    server     = AGENT::e_LOCAL_AGENT;
    registered = AGENT::e_REGISTERED;
    root       = AGENT::e_ROOT_AGENT;
    home       = AGENT::e_AT_HOME;

	// remember the local, root and home identifications

    localId    = id;
    rootId     = id;
    homeId     = id;
    rootHomeId = id;

	// notify dervived class that there has been a change

    changeNotification();
}

/* LOCATION::reset

   Purpose: Reset the local and root identifications to empty

     Input: None

    Output: The procedure resets the local and root identifications to empty.
*/

void LOCATION::reset (void)
{
	// NOTE that we do not reset the root, home and server method variables,
	// since we might still want to test those values after the agent has
	// ended

	// agent is no longer registered

    registered = AGENT::e_NOT_REGISTERED;

	// reset the identifications

    rootId.empty ();
    homeId.empty ();
    localId.empty ();
    rootHomeId.empty ();

	// notify dervived class that there has been a change

    changeNotification();
}
