/* Agent Tcl
   Bob Gray
   10 January 1996

   genLocation.h

   This file defines class LOCATION which keeps track of an agent's location
   within the network. 

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _GEN_LOCATION_H
#define _GEN_LOCATION_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _AGENT_AGENT_ID_H
#include "agentAgentId.h"	  // AgentId 
#endif
#ifndef _SUPP_MACH_ID_H
#include "mesgMachId.h"          // MachineId
#endif
#ifndef _GEN_AGENT_H
#include "genAgent.h"		  // AGENT
#endif
#ifndef _GEN_STRING_H
#include "suppDString.h"		  // DynamicString
#endif
#ifndef _GEN_SERVER_FLAGS_H
#include "genServerInterface.h"	  // ServerInterface::SubmissionTypes
#endif

    /* agent's location within the network */

class LOCATION
{
    friend class AGENT;

private:	// friend class AGENT calls these methods

	/*
	 * update location information for a root agent that did not arrive 
	 * via the server
	 */

    virtual void update (const AgentId &id);
	/* 
	 * update the symbolic name of the agent
	 */

    virtual void updateName (char *name);

	/* 
	 * reset the local and root identifications to empty
	 */

    virtual void reset (void);

	/* 
	 * turn the agent into a root agent
	 */

    void turnIntoRoot (void);

	/* 
	 * bootstrap location for an incoming agent
	 */

    int arrivingBootstrap 
	(ServerInterface::SubmissionTypes submissionType,
	 const AgentId &root,
	 const AgentId &rootHome,
	 const AgentId &local,
	 const AgentId &localHome,
	 const DynamicString &tempDirectory);

	/* 
	 * bootstrap location for an agent started on the current machine
	 */

    int startingBootstrap 
	(void);

protected:	// data

	/* variable data */

    AGENT::AgentSource server;       /* AGENT_SERVER or AGENT_LOCAL           */
    AGENT::AgentRelation root;       /* AGENT_ROOT or AGENT_CHILD             */
    AGENT::AgentLocation home;       /* AGENT_HOME or AGENT_AWAY              */
    AGENT::AgentRegister registered; /* AGENT_REGISTERD or AGENT_UNREGISTERED */  
    AgentId homeId;                  /* home identification                   */
    AgentId localId;                 /* local identification                  */
    AgentId rootId;                  /* root identification                   */
    AgentId rootHomeId;              /* root home identificaton               */
    MachineId machine;		     /* machine on which agent is executing   */
    UINT_32 localhost;               /* IP address 127.0.0.1                  */
    DynamicString tempDirectory;     /* temporary agent directory             */

	/* agent associated with this LOCATION structure */

    AGENT *agent;

protected:	// constructor and destructor

	/* constructor and destructor */

    LOCATION (AGENT *agent);
    virtual ~LOCATION () {};

public:		// "get" routines

	/* IP address 127.0.0.1 */

    UINT_32 getLocalhostIp (void) {
	return (localhost);
    }

	/* get the actual machine or actual IP on which the agent is executing */

    MachineId getMachine (void) {
	return (machine);
    }

    UINT_32 getMachineIp (void) {
	return (machine.getIp());
    }

	/* get the agent's local identification or local IP */

    AgentId getLocal (void) const {
	return (localId);
    } 

    UINT_32 getLocalIp (void) const {
	return (localId.getServerIp());
    }

	/* get the identifcation or IP of the root agent */

    AgentId getRoot (void) const {
	return (rootId);
    } 

    UINT_32 getRootIp (void) const {
	return (rootId.getServerIp());
    }

	/* get the agent's home identification or home IP */

    AgentId getHome (void) const {
	return (homeId);
    }

    UINT_32 getHomeIp (void) const {
	return (homeId.getServerIp());
    }

	/* get the home identification or home IP of the root agent */

    AgentId getRootHome (void) const {
	return (rootHomeId);
    } 

    UINT_32 getRootHomeIp (void) const {
	return (rootHomeId.getServerIp());
    }

	/* get or set the agent temporary directory */

    DynamicString getTempDirectory (void) const {
	return (tempDirectory);
    }

	/* check whether or not the agent arrived via the server */

    int isServer (void) const {
	return (server == AGENT::e_INCOMING_AGENT);
    };

	/* check whether or not the agent is a root agent */

    int isRoot (void) const {
	return (root == AGENT::e_ROOT_AGENT);
    };

	/* check whether or not the agent is home */

    int isHome (void) const {
	return (home == AGENT::e_AT_HOME);
    };

	/* check whether or not the agent is registered */

    int isRegistered (void) const {
	return (registered == AGENT::e_REGISTERED);
    };

public:		// "callbacks"

	/*
	 * notify the derived class that something has changed
	 */

    virtual void changeNotification (void) = 0;
};

#endif    /* STACK */
