/* Agent Tcl
   Bob Gray
   10 January 1996

   genMasks.cc

   This file implements the mask structures.

   Copyright (c) 1995-1998, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "platCharact.h"
#include "platSigio.h"
#include "suppStrings.h"
#include "mesgPort.h"			// RemotePort
#include "mesgTcpip.h"			// TCP/IP routines
#include "agentAgentId.h"		// AgentId
#include "genAgentConv.h"		// Agent_AgentMaskEntryToString()
#include "genAgentIO.h"			// AgentIO

#include "genAgent.h"
#include "suppHash.h"
#include "genAgentConn.h"
#include "genLocation.h"
#include "genManager.h"
#include "genMasks.h"
#include "genServerInterface.h"		// ServerInterface::SubmissionTypes
#include "genTransmit.h"

UINT_32 AgentMaskEntry::localIp	= 0;
UINT_32 AgentMaskEntry::actualIp	= 0;

/* AgentMaskHandler::AgentMaskHandler and AgentMaskHandler::~MASK_HANDLER

   Purpose: These procedures are the constructor and destructor for class
            AgentMaskHandler.
*/

AgentMaskHandler::AgentMaskHandler (HandlerTypes p_type)
{
	/* assertions on the type */

    assert ((p_type == EVENT_HANDLER) || (p_type == INTERRUPT_HANDLER));

	/* remember the type */

    AgentMaskHandler::type = p_type;
};

AgentMaskHandler::~AgentMaskHandler () 
{
    // empty
}

/* AgentMaskEntry::AgentMaskEntry

   Purpose: These procedures are the constructors for class AgentMaskEntry.
*/

AgentMaskEntry::AgentMaskEntry (void)
{
    id      = NULL;
    tag     = NULL;
    handler = NULL;
}

AgentMaskEntry::AgentMaskEntry (const AgentMaskEntry &entry)
{
    if (entry.id == NULL) {
	id = NULL;
    } else {
	id = new AgentId (*(entry.id));
    }

    tag  = strcpyWithAlloc (entry.tag);

    if (entry.handler == NULL) {
	handler = (AgentMaskHandler *) NULL;
    } else {
	handler = entry.handler -> clone(); 
    }
}

AgentMaskEntry::AgentMaskEntry (AgentId *p_id, char *p_tag, AgentMaskHandler *p_handler)
{
	/* assertions */

    assert ((p_id != NULL) || (p_handler != NULL));

	/* initialize */

    if (p_id == NULL) {
	AgentMaskEntry::id = NULL;
    } else {
	AgentMaskEntry::id = new AgentId (*p_id);
    }

    AgentMaskEntry::tag = strcpyWithAlloc (p_tag);

    if (p_handler == NULL) {
	AgentMaskEntry::handler = NULL;
    } else {
	AgentMaskEntry::handler = p_handler -> clone(); 
    }
}

/* AgentMaskEntry::operator =

   Purpose: This procedure defines the assignment operator for class
	    AgentMaskEntry.
*/

AgentMaskEntry &AgentMaskEntry::operator= (const AgentMaskEntry &mask)
{
	/* check for assignment to self */

    if (this == &mask) {
	return (*this);
    }

	/* assign */

    DELETE_OBJECT(id);

    if (mask.id == NULL) {
	id = (AgentId *) NULL;
    } else {
	id = new AgentId (*mask.id);
    }

    DELETE_ARRAY_OBJECT(tag);
    tag = strcpyWithAlloc (mask.tag);

    DELETE_OBJECT(handler);

    if (mask.handler == NULL) {
	handler = (AgentMaskHandler *) NULL;
    } else {
	handler = mask.handler -> clone ();
    }
 
	/* return self-reference */

    return (*this);
}
     
/* AgentMaskEntry::~AgentMaskEntry

   Purpose: This procedure is the destructor for class AgentMaskEntry.
*/

AgentMaskEntry::~AgentMaskEntry ()
{
    DELETE_OBJECT(id);
    DELETE_OBJECT(handler);
    DELETE_ARRAY_OBJECT(tag);
}

/* AgentMaskEntry::exactCompare

   Purpose: Perform an exact comparison of this mask entry and another mask
	    entry

     Input: entry = the other mask entry
		    (class AgentMaskEntry &)

    Output: The procedure returns TRUE if the entries are identifcal and FALSE
	    if the entries are different.
*/

int AgentMaskEntry::exactCompare (AgentMaskEntry &entry) const
{
    int code;

	/* check the agent identification */

    if (id == entry.id) {
	code = TRUE;
    } else if ((id == NULL) || (entry.id == NULL)) {
	code = FALSE;
    } else if (id -> getServerIp() != entry.id -> getServerIp()) {
	code = FALSE;
    } else if (id -> getName() != entry.id -> getName()) {
	code = FALSE;
    } else if (id -> getId() != entry.id -> getId()) {
	code = FALSE;
    } else {
	code = TRUE;
    }

	/* check the tag and handler */

    if (code) {

	if (strcmpNullAndEmpty (tag, entry.tag)) {
	    code = FALSE;
	} else if (handler == entry.handler) {
	    code = TRUE;
	} else if ((handler == NULL) || (entry.handler == NULL)) {
	    code = FALSE;
	} else if (handler -> compare (*entry.handler)) {
 	    code = TRUE;
	} else {
	    code = FALSE;
	}
    }

    return (code);
}

/* AgentMaskEntry::inexactCompare

   Purpose: Perform an inexact comparison between the entry and an incoming
	    item

     Input: which = EVENT_ITEM, MEETING_ITEM or MESSAGE_ITEM
		    (int)

	    item  = the incoming item
		    (class INCOMING &)

    Output: The procedure returns TRUE if the item matches the entry and
	    FALSE otherwise.
*/

int AgentMaskEntry::inexactCompare (int which, INCOMING &item) const
{
	/* check the agent identification if necessary */

    if (id != NULL) {

	    /* check the IP address */

	if (id -> getServerIp() != ANY_IP) {

	    if (id -> getServerIp() != item.id -> getServerIp()) {
		if ((id -> getServerIp() != localIp) || (item.id -> getServerIp() != actualIp)) {
		    return FALSE;
		}
	    }
	}

	    /* check the numeric id */

	if (id -> getId() != ANY_ID) {
	    if (id -> getId() != item.id -> getId()) {
		return FALSE;
	    } 
	} 

	    /* check the name */

	if (id -> getName().length() != 0) {
	    if (id -> getName() != item.id -> getName()) {
		return FALSE; 
	    }
	}
    }

	/* check the event tags if this is an EVENT_ITEM */

    if (which == EVENT_ITEM) {
	if (tag != ANY_TAG) {
	    if (strcmpNullAndEmpty (tag, ((INCOMING_EVENT &) item).tag)) {
		return FALSE; 
	    }
	}
    }

    return TRUE;
}

/* AgentMask::AgentMask

   Purpose: These procedures are the constructors for class MASK.

     Input: type   = MASK_ACCEPT_ALL or MASK_ACCEPT_NONE
		     (int)

	    handle = mask handle
		     (UINT_32)
*/

AgentMask::AgentMask (void)
{
    type   = MASK_ACCEPT_NONE;
    handle = 0;
}

AgentMask::AgentMask (int p_type, UINT_32 p_handle)
{
	/* assertions on the parameters */

    assert ((p_type == MASK_ACCEPT_NONE) || (p_type == MASK_ACCEPT_ALL));

	/* initialize */
 
    AgentMask::type   = p_type;
    AgentMask::handle = p_handle;
}

/* AgentMask::AgentMask

   Purpose: This procedure is the copy constructor for class MASK.

     Input: mask = mask that we are copying
		   (const MASK &)
*/

AgentMask::AgentMask (const AgentMask& mask)
{
    int n;
    AgentMaskEntry *entry;
    AgentMaskEntry *copiedEntry;

	/* copy the easy parts */

    type   = mask.type;
    handle = mask.handle;

	/* copy the queue of mask entrues */

    n = mask.entries.get_count ();
 
    for (int i = 0; i < n; i++) {
	entry = (AgentMaskEntry *) mask.entries.peek (i); 
	copiedEntry = new AgentMaskEntry (*entry);
	entries.enqueue ((void *) copiedEntry);
    }
}

/* AgentMask::operator =

   Purpose: This procedure is the assignment operator for class MASK.

     Input: mask = mask that we are copying
		   (const AgentMask &)
*/

AgentMask &AgentMask::operator= (const AgentMask &mask)
{
    int n;
    AgentMaskEntry *entry;
    AgentMaskEntry *copiedEntry;

	/* make sure that we are not assigning to ourself */

    if (this == &mask) {
	return (*this);
    }

	/* copy the easy stuff */

    type   = mask.type;
    handle = mask.handle;

	/* copy the queue of mask entries */

    while ((entry = (AgentMaskEntry *) entries.dequeue ()) != NULL) {
	DELETE_OBJECT(entry);
    }
 
    n = mask.entries.get_count();
 
    for (int i = 0; i < n; i++) {
	entry = (AgentMaskEntry *) mask.entries.peek (i); 
	copiedEntry = new AgentMaskEntry (*entry);
	entries.enqueue ((void *) copiedEntry);
    }

	/* return self-reference */

    return (*this);
}

/* AgentMask::~AgentMask

   Purpose: This procedure is the destructor for class MASK.
*/

AgentMask::~AgentMask ()
{
    AgentMaskEntry *entry;

	/* remove all the mask entries */

    while ((entry = (AgentMaskEntry *) entries.dequeue ()) != NULL) {
	DELETE_OBJECT(entry); 
    }   
}

/* AgentMask::empty_and_set

   Purpose: Remove all of the mask entries and set the type

     Input: type = MASK_ACCEPT_ALL or MASK_ACCEPT_NONE

    Output: The procedure removes all of the mask entries and sets the type.
*/

void AgentMask::empty_and_set (int p_type)
{ 
    AgentMaskEntry *entry;

	/* assertions on the paramter */

    assert ((p_type == MASK_ACCEPT_ALL) || (p_type == MASK_ACCEPT_NONE));

	/* empty out the mask */

    while ((entry = (AgentMaskEntry *) entries.dequeue()) != NULL) {
	DELETE_OBJECT(entry);
    }
    
    AgentMask::type = p_type;
}

/* AgentMask::add

   Purpose: Add an entry to a mask

     Input: entry = the mask entry
		    (class AgentMaskEntry &)

    Output: The procedure returns one of the following codes.

	    MASK_OK        = success
	    MASK_DUP_ENTRY = entry is already in the mask 
*/

int AgentMask::add (AgentMaskEntry &entry)
{
    int code = MASK_OK;
    AgentMaskEntry *current;

	/* check that we do not have a duplicate entry */

    int i = 0;

    while ((current = (AgentMaskEntry *) entries.peek (i)) != NULL) {

	if (current -> exactCompare (entry)) {
	    code = MASK_DUP_ENTRY;
	    break;
	}
	
	i += 1;
    }

	/* add the entry if it is distinct */

    if (code == MASK_OK) {
	type = MASK_ACCEPT_SOME;
	current = new AgentMaskEntry (entry);
	entries.enqueue ((void *) current);
    }

    return (code);
}

/* AgentMask::remove

   Purpose: Remove an entry from a mask

     Input: entry = the mask entry
		    (class AgentMaskEntry &)

    Output: The procedure returns one of the following codes.

	    MASK_OK            = success
	    MASK_INVALID_ENTRY = the mask does not contain the given entry
*/

int AgentMask::remove (AgentMaskEntry &entry)
{
    register int i;
    int totalEntries;
    AgentMaskEntry *current;
    int code = MASK_INVALID_ENTRY;

	/* remove the entry -- note that we can not break out of the loop */
        /* once we remove the entry since the entries have to end up in   */
        /* the same order; thus we keep going even after we remove the    */
        /* entry except that we don't try to remove again                 */

    totalEntries = entries.get_count ();
 
    for (i = 0; i < totalEntries; i++) {

	current = (AgentMaskEntry *) entries.dequeue ();

	if (code != MASK_OK) {
	    if (current -> exactCompare (entry)) {
	        code = MASK_OK;
	 	continue;
	    }
	}

	entries.enqueue ((void *) current);
    }

	/* is the mask now empty */

    if (entries.get_count() == 0) {
	type = MASK_ACCEPT_NONE;
    }

    return (code);
}

/* AgentMask::filter

   Purpose: Filter an incoming item through the mask

     Input: which = EVENT_ITEM, MEETING_ITEM or MESSAGE_ITEM
		    (int)

	    item  = the incoming item
		    (class INCOMING &)

    Output: The procedure returns TRUE if there is an entry in the mask
            that matches the incoming item.  Otherwise the procedure returns
            FALSE.
*/

int AgentMask::filter (int which, INCOMING &item)
{
    int i;
    AgentMaskEntry *entry;
    int totalEntries;

	/* assertions on the parameters */

    assert ((which == EVENT_ITEM) || (which == MESSAGE_ITEM) || (which == MEETING_ITEM));

	/* check if the the list contains ALL masks or NO masks */

    if (type == MASK_ACCEPT_ALL) {
	return TRUE;
    } else if (type == MASK_ACCEPT_NONE) {
	return FALSE;
    }

        /* check each mask entry */

    totalEntries = entries.get_count ();

    for (i = 0; i < totalEntries; i++) {

	entry = (AgentMaskEntry *) entries.peek (i);

	if (entry -> inexactCompare (which, item)) {
	    return TRUE;
	}
    }

	/* no mask entry matches the incoming item */

    return FALSE;
}

/* AgentMasks::AgentMasks

   Purpose: This procedure is the constructor for class AgentMasks.

     Input: p_agent = agent associated with this AgentMasks instance
                      (class AGENT *) 
*/

AgentMasks::AgentMasks (AGENT *p_agent)
{ 
    LOCATION *locations;

	/* assertions on the parameters */

    assert (p_agent != NULL);

	/* agent, number of masks, and next untried handle */

    agent              = p_agent;
    mask_count         = 0;
    mask_handle        = 0;

	/* initially do not allow event or interrupt handlers */

    allowEvents     = FALSE;
    allowInterrupts = FALSE;

	/* LOCATIONS structure */

    locations = agent -> get_location ();
    assert (locations != NULL);

	/* hash table */

    maskTable = new Hash (MASK_TABLE_SIZE);

	/* message, meeting and event masks */

    masks[MESSAGE_ITEM] = NULL; 
    masks[MEETING_ITEM] = NULL; 
    masks[ EVENT_ITEM ]	= NULL; 
}

/* AgentMasks::arrivingBootstrap

   Purpose: Bootstrap the masks system for agents that are *arriving* from
	    another machine - e.g., create the default event, meeting and
	    message masks only if this is a new agent created with the
            SUBMIT or AGLET_SUBMIT command

     Input: None

    Output: The procedure returns one of the following codes.

	    MASK_OK = success 
*/

int AgentMasks::arrivingBootstrap (ServerInterface::SubmissionTypes type)
{
    LOCATION *locations = agent -> get_location();
    assert (locations != NULL);

	// local and actual IP addresses

    AgentMaskEntry::localIp  = locations -> getLocalhostIp ();
    AgentMaskEntry::actualIp = locations -> getMachineIp ();

	// default masks if needed

    if ((type == ServerInterface::e_SUB_SCRIPT) ||
	(type == ServerInterface::e_SUB_AGLET_SUBMIT)) {

	masks[MESSAGE_ITEM] = maskCreateInt (MASK_ACCEPT_ALL);
	masks[MEETING_ITEM] = maskCreateInt (MASK_ACCEPT_ALL);
	masks[ EVENT_ITEM ] = maskCreateInt (MASK_ACCEPT_ALL);
    }

    return (MASK_OK);
}

/* AgentMasks::startingBootstrap

   Purpose: Bootstrap the masks system for agents that are *starting* on the
	    local machine - e.g., create the default event, meeting and
	    message masks.

     Input: None

    Output: The procedure returns one of the following codes.

	    MASK_OK = success 
*/

int AgentMasks::startingBootstrap (void)
{
    LOCATION *locations = agent -> get_location();
    assert (locations != NULL);

	/* local and actual IP addresses */

    AgentMaskEntry::localIp  = locations -> getLocalhostIp ();
    AgentMaskEntry::actualIp = locations -> getMachineIp ();

	/* default masks */

    masks[MESSAGE_ITEM] = maskCreateInt (MASK_ACCEPT_ALL);
    masks[MEETING_ITEM] = maskCreateInt (MASK_ACCEPT_ALL);
    masks[ EVENT_ITEM ]	= maskCreateInt (MASK_ACCEPT_ALL);
    return (MASK_OK);
}

/* AgentMasks::maskCreate

   Purpose: Create a new mask

     Input: None

    Output: The procedure returns one of the following error codes.
	   
   	    MASK_OK       = success
	    MASK_TOO_MANY = maximum number of masks has been reached

	    On success 

	    handle       = the mask handle
			   (UINT_32 &)
*/ 

int AgentMasks::maskCreate (UINT_32 &handle)
{
    int code;
    AgentMask *mask;

	/* guard the access (lock is automatically released) */

    Guard guard (agent -> getMonitorLock());

        /* create the new entry */
 
    if ((mask = maskCreateInt (MASK_ACCEPT_NONE)) == NULL) {
	code = MASK_TOO_MANY;
    } else {
	handle = mask -> handle;
	code = MASK_OK;
    }

    return (code);
}

/* AgentMasks_TCL::maskCreateKnown

   Purpose: Create a mask with the given handle

     Input: handle = desired handle
		     (UINT_32)

    Output: The procedure returns one of the following error codes.
	   
   	    MASK_OK          = success
	    MASK_TOO_MANY    = maximum number of masks has been reached
	    MASK_USED_HANDLE = desired handle is in use
*/ 

int AgentMasks::maskCreateKnown (UINT_32 handle)
{
    int code;

	/* guard the access (lock is automatically released) */

    Guard guard (agent -> getMonitorLock());

        /* create the new entry */
 
    code = maskCreateKnownInt (MASK_ACCEPT_NONE, handle);
    return (code);
}

/* AgentMasks::maskDelete

   Purpose: Delete a mask

     Input: handle = mask handle
		     (UINT_32)

    Output: The procedure returns one of the following codes.

	    MASK_OK             = success
            MASK_INVALID_HANDLE = no mask with the specified handle
	    MASK_IN_USE         = mask is the current meeting, message or event mask
*/

int AgentMasks::maskDelete (UINT_32 handle)
{
    int i; 
    HashNode *node;
    int code = MASK_OK;

	/* guard the access (lock is automatically released) */

    Guard guard (agent -> getMonitorLock());

	/* check the the mask is not in use */

    for (i = 0; i < NUM_ITEMS; i++) {
	if (masks[i] -> handle == handle) {
	    code = MASK_IN_USE;
	}
    }

	/* remove the mask if it is not in use */

    if (code != MASK_IN_USE) {
	if ((node = maskTable -> lookup (handle, (char *) NULL)) == NULL) {
	    code = MASK_INVALID_HANDLE;
	} else {
	    DELETE_OBJECT_WITH_CAST(AgentMask*,node->data);
	    maskTable -> remove (handle, (char *) NULL);
	    mask_count -= 1;    
	}
    }

    return (code);
}

/* AgentMasks::maskFill

   Purpose: Fill a mask so that it accepts any incoming item

     Input: handle = mask handle
		     (UINT_32)

    Output: The procedure returns one of the following codes.

	    MASK_OK             = success
	    MASK_INVALID_HANDLE = no mask with the specified handle
*/

int AgentMasks::maskFill (UINT_32 handle)
{
    int code;
    AgentMask *mask;

	/* guard the access (lock is automatically released) */

    Guard guard (agent -> getMonitorLock());

	/* lookup the mask and then fill*/ 

    if ((mask = maskLookupInt (handle)) == NULL) {
        code = MASK_INVALID_HANDLE;
    } else {
	mask -> fill ();
	code = MASK_OK;
    }

    return (code);
}

/* AgentMasks::maskEmpty

   Purpose: Empty a mask so that it accepts no incoming items

     Input: handle = mask handle
		     (UINT_32)

    Output: The procedure returns one of the following codes.

	    MASK_OK             = success
	    MASK_INVALID_HANDLE = no mask with the specified handle
*/

int AgentMasks::maskEmpty (UINT_32 handle)
{
    int code;
    AgentMask *mask;

	/* guard the access (lock is automatically released) */
 
    Guard guard (agent -> getMonitorLock());

	/* lookup the mask and then fill*/ 

    if ((mask = maskLookupInt (handle)) == NULL) {
        code = MASK_INVALID_HANDLE;
    } else {
	mask -> empty ();
	code = MASK_OK;
    }

    return (code);
}

/* AgentMasks::maskGet

   Purpose: Get a mask

     Input: handle = the mask handle
		     (UINT_32)

    Output: The procedure returns NULL if there is no mask with the specified
	    handle.  Otherwise the procedure returns a pointer to a
	    dynamically allocated MASK structure that contains a complete
	    *COPY* of the mask.
*/

AgentMask *AgentMasks::maskGet (UINT_32 handle)
{
    AgentMask *mask;
    AgentMask *copiedAgentMask;

	/* guard the access (lock is automatically released) */

    Guard guard (agent -> getMonitorLock());

	/* lookup the mask and then copy */

    if ((mask = maskLookupInt (handle)) == NULL) {
	copiedAgentMask = NULL;
    } else {
	copiedAgentMask = new AgentMask (*mask);
    }

    return (copiedAgentMask);
}

/* AgentMasks:maskAdd

   Purpose: Add an entry to a mask

     Input: handle = the mask handle
		     (UINT_32);

	    entry  = the entry to be added
		     (class AgentMaskEntry &)

    Output: The procedure returns one of the following codes.

	    MASK_OK             = success
	    MASK_INVALID_HANDLE = no mask with the specified handle
	    MASK_DUP_ENTRY      = entry is exactly the same as another entry
*/

int AgentMasks::maskAdd (UINT_32 handle, AgentMaskEntry &entry)
{
    AgentMask *mask;
    int code = MASK_OK;

	/* guard the access (lock is automatically released) */
 
    Guard guard (agent -> getMonitorLock());

	/* make sure that we are allowed to specify an event or interrupt handler */

    if (entry.handler != NULL) {

	if ((entry.handler -> getType() == AgentMaskHandler::EVENT_HANDLER) && !(allowEvents)) {
	    code = MASK_NO_EVENTS;
        } else if ((entry.handler -> getType() == AgentMaskHandler::INTERRUPT_HANDLER) && !(allowInterrupts)) {
	    code = MASK_NO_INTERRUPTS;
	}
    }

	/* lookup the mask and then add */

    if (code == MASK_OK) {

	if ((mask = maskLookupInt (handle)) == NULL) {
            code = MASK_INVALID_HANDLE;
	} else {
	    code = mask -> add (entry);
	}

	    /* fire off event handlers if everything is okay */

	if (code == MASK_OK) {
	    for (int i = 0; i < NUM_ITEMS; i++) {
		if (masks[i] == mask) {
		    fireAllEvents_noLock (i);
		}
	    }
	}
    }

    return (code);
}

/* AgentMasks:maskRemove

   Purpose: Remove an entry from a mask

     Input: handle = the mask handle
		     (UINT_32);

	    entry  = the entry to be removed
		     (class AgentMaskEntry &)

    Output: The procedure returns one of the following codes.

	    MASK_OK             = success
	    MASK_INVALID_HANDLE = no mask with the specified handle
	    MASK_INVALID_ENTRY  = mask does not contain the given entry
*/

int AgentMasks::maskRemove (UINT_32 handle, AgentMaskEntry &entry)
{
    int code;
    AgentMask *mask;

	/* guard the access (lock is automatically released) */

    Guard guard (agent -> getMonitorLock());

	/* lookup the mask and then fill*/ 

    if ((mask = maskLookupInt (handle)) == NULL) {
        code = MASK_INVALID_HANDLE;
    } else {
	code = mask -> remove (entry);
    }

    return (code);
}

/* AgentMasks::maskLookupInt

   Purpose: Find a mask in the table

     Input: handle = mask handle
		     (UINT_32)

    Output: The procedure returns NULL if there is no mask with the
	    specified handle.  Otherwise the procedure returns a pointer to
	    the appropriate MASK structure.
*/

AgentMask *AgentMasks::maskLookupInt (UINT_32 handle)
{
    HashNode *node;

    if ((node = maskTable -> lookup (handle, (char *) NULL)) == NULL) {
	return ((AgentMask *) NULL);
    }

    return ((AgentMask *) node -> data);
}

/* AgentMasks::maskCreateKnownInt

   Purpose: Create a new mask with a specific handle (internal version)

     Input: type   = MASK_ACCEPT_NONE or MASK_ACCEPT_ALL
		     (int)

	    handle = desired handle
		     (UINT_32)

    Output: The procedure returns one of the following codes.

	    MASK_OK          = success
	    MASK_TOO_MANY    = too many masks
	    MASK_USED_HANDLE = handle is in use
*/

int AgentMasks::maskCreateKnownInt (int type, UINT_32 handle)
{
    AgentMask *mask;
    int there = 1;
    HashNode *node;

	/* check the number of masks */

    if (mask_count >= (UINT_32) MAX_U32) {
	return MASK_TOO_MANY; 
    }

	/* create the mask */

    mask = new AgentMask (type, handle); 
    node = maskTable -> add (handle, (char *) NULL, mask, there);

    if (there) {
	DELETE_OBJECT(mask);
	return MASK_USED_HANDLE;
    }

	/* success */

    return MASK_OK; 
}

/* AgentMasks::maskCreateInt

   Purpose: Create a new mask (internal version)

     Input: type  = MASK_ACCEPT_NONE or MASK_ACCEPT_ALL
		    (int)
*/

AgentMask *AgentMasks::maskCreateInt (int type)
{
  AgentMask *mask;
  int there = 1;
  HashNode *node;

    /* check the number of masks */

  if (mask_count >= (UINT_32) MAX_U32) {
    return NULL;
  }

    /* create the mask */

  mask = new AgentMask (type, mask_handle); 

    /* find an unused mask handle */

  while (there) {

    node = maskTable -> add (mask -> handle, (char *) NULL, mask, there);

    if (!there) {
	mask_count += 1;
    } else {
	mask->handle = (mask->handle + 1) % MAX_U32;
    } 
  } 

    /* pass back the mask */

  mask_handle = (mask -> handle % MAX_U32) + 1;
  return (mask);
}

/* AgentMasks::~AgentMasks

   Purpose: This procedure is the destructor for class AgentMasks.
*/

AgentMasks::~AgentMasks ()
{
    HashNode *node;
    HashSearch search (maskTable);

	/* turn off the background handler and delete all queued items */

    deleteAllIncomingItems ();

	/* delete every entry in the hash table */

    while ((node = search.next ()) != (HashNode *) NULL) {
	DELETE_OBJECT_WITH_CAST(AgentMask*,node -> data);
    }
}

void AgentMasks::deleteAllIncomingItems (void)
{
    int i;
    EventData *event;
    INCOMING *incoming;

	/* delete all incoming communication */

    for (i = 0; i < NUM_ITEMS; i++) {
	while ((incoming = (INCOMING *) queues[i].dequeue()) != NULL) {
	    DELETE_OBJECT(incoming);
	}
    }

	/* delete all pending events */

    while ((event = (EventData *) eventQueue.dequeue()) != NULL) {
	DELETE_OBJECT(event);
    }

	/* delete all pending interrupts */

    while ((event = (EventData *) interruptQueue.dequeue()) != NULL) {
	DELETE_OBJECT(event);
    }
}

/* AgentMasks::getIncoming_noLock

   Purpose: Get an incoming item

     Input: which    = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
		       (int)

    Output: The procedure returns NULL if there is no incoming item.
            Otherwise it returns a pointer to a dynamically allocated
            INCOMING instance.
*/

INCOMING *AgentMasks::getIncoming_noLock (int which)
{
    int i;
    INCOMING *iptr;
    int totalEntries;
    INCOMING *item = NULL;

    totalEntries = queues[which].get_count ();

    for (i = 0; i < totalEntries; i++) {

	iptr = (INCOMING *) queues[which].dequeue ();

	if ((item == NULL) && (masks[which] -> filter (which, *iptr))) {
	    item = iptr;
	} else {
	    queues[which].enqueue ((void *) iptr);
	}
    }

    return (item);
}

BOOLEAN AgentMasks::checkIncoming_noLock (int which)
{
    int i;
    INCOMING *iptr;
    int totalEntries;

    totalEntries = queues[which].get_count ();

    for (i = 0; i < totalEntries; i++) {

	iptr = (INCOMING *) queues[which].peek (i);

	if (masks[which] -> filter (which, *iptr)) {
	    return (e_TRUE);
	}
    }

    return (e_FALSE);
}

/* MASK_SET::addIncomingToQueue_noLock

   Purpose: Add an incoming item to one of the internal queues

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
		    (int)

            item  = the incoming item
		    (INCOMING *)

    Output: The procedure adds the incoming item to the specified queue.

      Note: The procedure assumes that the lock has already been acquired.
*/

void AgentMasks::addIncomingToQueue_noLock (int which, INCOMING *item)
{
	/* assertions on the parameters */

    assert (item != NULL);
    assert ((which == MESSAGE_ITEM) || (which == EVENT_ITEM) || (which == MEETING_ITEM));

	/* fire event handlers if defined; otherwise add to queue */

    if (!fireEvents_noLock (which, item)) {
	queues[which].enqueue ((void *) item);
    }

	/* notify the IO subsystem */

    AgentIO *io = agent -> get_io ();
    assert (io != NULL);
    io -> setQueuedItem ();
} 

/* AgentMasks::addMeetingToQueue

   Purpose: Add a meeting to the queue

     Input: meeting = the meeting
		      (struct INCOMING_MEETING *)

    Output: The procedure adds the meeting to the meeting queue.
*/

void AgentMasks::addMeetingToQueue (INCOMING_MEETING *meeting)
{
	/* assertions on the parameters */

    assert (meeting != NULL);
    assert (meeting -> status == MEET_REQUESTED);

	/* mutual exclusion and then add */

    Guard guard (agent -> getMonitorLock());
    addIncomingToQueue_noLock (MEETING_ITEM, meeting);
} 

/* AgentMasks::handleIncoming_noLock

   Purpose: Add an incoming item to the appropriate queue

     Input: item = the incoming item
	           (struct MESSAGE *)

    Output: The procedure returns one of the following codes.

	    MASK_OK              = success
	    MASK_LOST_BACKGROUND = connection to server has failed

      Note: The procedure assumes that the lock has already been acquired.
*/

int AgentMasks::handleIncoming_noLock (MESSAGE *item)
{
    int code = MASK_OK;

	/* break apart the incoming item */

    if (item -> getFlag() == RESP_MESSAGE) {

	INCOMING_MESSAGE *message;

	    /* break apart the incoming message */
   
        AgentId *source       = item -> getId (1);      // don't delete 
	BASE_SECURITY *secure = item -> getSecurity (2);  // don't delete 
	UINT_32 code          = item -> getLong (3); 
	char *string          = item -> getString (4);  // don't delete 

	     /* construct the INCOMING_MESSAGE structure */

	message = new INCOMING_MESSAGE (source, code, string, secure);

	     /* add the incoming message to the appropriate queue */

	addIncomingToQueue_noLock (MESSAGE_ITEM, message);

    } else if (item -> getFlag() == RESP_MEETING) {

	RemotePort *port;
	AgentMeetings *meetings;
	INCOMING_MEETING *meeting;

	    /* break apart the incoming message */

	AgentId *source         = item -> getId (1);        // don't delete 
	BASE_SECURITY *secure   = item -> getSecurity (2);  // don't delete 
	UINT_8 request	        = item -> getByte (3); 
	UINT_32 localHandle     = item -> getLong (4); 
	UINT_32 remoteHandle    = item -> getLong (5); 
	SmartCharPtr actualMach (item -> getString (6));
	UINT_32 actualIp        = item -> getIp (7); 
	UINT_32 actualPort      = item -> getLong (8);

	    /* assemble the RemotePort and then the INCOMING_MEETING */

	port = new RemotePort (actualMach, actualIp, actualPort);
	meeting = new INCOMING_MEETING (source, request, localHandle, remoteHandle, port, secure);

	    /* now handle the meeting */ 

	meetings = agent -> get_meetings ();
	meetings -> handleIncomingMeeting (meeting);

    } else if (item -> getFlag() == RESP_EVENT) {

	INCOMING_EVENT *event;

	    /* break apart the incoming event */

	AgentId *source      = item -> getId (1);        // don't delete
        BASE_SECURITY *secure = item -> getSecurity (2);  // don't delete
	char *tag	      = item -> getString (3);    // don't delete 
	char *string	      = item -> getString (4);    // don't delete 

	     /* construct the INCOMING_EVENT structure */

	event = new INCOMING_EVENT (source, tag, string, secure);

	     /* add the incoming event to the appropriate queue */

	addIncomingToQueue_noLock (EVENT_ITEM, event);

    } else {

	code = MASK_LOST_BACKGROUND;
    }

    return (code);
}

/* AgentMasks::updateHandle

   Purpose: Change one of the three distinguished masks

     Input: which  = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
		     (int)

            handle = mask handle 
		     (UINT_32)

    Output: The procedure returns one of the following codes.

            MASK_OK             = success
            MASK_INVALID_HANDLE = no mask with the specified handle
*/

int AgentMasks::updateHandle (int which, UINT_32 handle)
{
    int code;
    AgentMask *mask;

	/* assertions on the parameters */

    assert ((which == MESSAGE_ITEM) || (which == MEETING_ITEM) || (which == EVENT_ITEM));

	/* guard the access (lock is automatically released) */ 

    Guard guard (agent -> getMonitorLock());

	/* find the mask with the given handle */

    if ((mask = maskLookupInt (handle)) == NULL) {

	code = MASK_INVALID_HANDLE; 

    } else {

	masks[which] = mask;
	code = MASK_OK;	
    }

	/* handle events */

    if (code == MASK_OK) {
	fireAllEvents_noLock (which);
    }

    return (code);
}

/* AgentMasks::fireAllEvents_noLock

   Purpose: Fire all event handlers that are ready

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
		    (int)

    Output: The procedure fires the event handlers

      Note: The procedure assums that the lock has already been acquired.
*/

void AgentMasks::fireAllEvents_noLock (int which)
{
    int i;
    int stop;
    INCOMING *item;

	/* assertions on the parameters */

    assert ((which == MESSAGE_ITEM) || (which == MEETING_ITEM) || (which == EVENT_ITEM));

	/* check that we have a handler proc and a mask with entries */

    if (masks[which] -> type == MASK_ACCEPT_SOME) { 

	    /* look for event handlers */

	stop = queues[which].get_count ();

	for (i = 0; i < stop; i++) {

	    item = (INCOMING *) queues[which].dequeue();

	    if (!fireEvents_noLock (which, item)) {

		queues[which].enqueue ((void *) item);

	    } else {

		/* Do NOT delete item since control of item has been */
		/* handed off to fireEvents.                         */
	    }
	}
    } 
}

/* MASK_SET::fireEvents_noLock

   Purpose: Fire all event handlers that are defined for a given item. 

     Input: which = MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM
		    (int)

            item  = the incoming item
		    (struct INCOMING *)
 
    Output: The procedure returns TRUE if one or more event handlers fired.
	    Otherwise the procedure returns FALSE.
*/

int AgentMasks::fireEvents_noLock (int which, INCOMING *item)
{
    int i;
    int stop;
    int rc = FALSE;
    int matches = 0;
    int intMatches = 0;
    int eveMatches = 0;
    AgentMaskEntry *entry;
    EventData *eventData = NULL;
    AgentMaskEntry **intEntries = NULL;
    AgentMaskEntry **eveEntries = NULL;

	/* assertions on the parameters */

    assert (item != NULL);
    assert ((which == MEETING_ITEM) || (which == MESSAGE_ITEM) || (which == EVENT_ITEM));

	/* make sure that we have a mask with entries */

    if (masks[which] -> type == MASK_ACCEPT_SOME) {

	    /* check the item against each mask entry */

	stop = masks[which] -> entries.get_count ();

	if (stop > 0) {

	    intEntries = new AgentMaskEntry * [stop];
            eveEntries = new AgentMaskEntry * [stop];

	    for (i = 0; i < stop; i++) {

		entry = (AgentMaskEntry *) masks[which] -> entries.peek (i);

		if ((entry -> handler != NULL) && (entry -> inexactCompare (which, *item))) {

		    if (entry -> handler -> type == AgentMaskHandler::EVENT_HANDLER) {

			eveEntries[eveMatches++] = entry;
			matches += 1;

		    } else {

			intEntries[intMatches++] = entry;
			matches += 1;
		    }
		}
	    }

		/* fire off the event handlers if we had a match */

	    if (matches > 0) {

		if (intMatches > 0) {
		    eventData = new EventData (intMatches, AgentMaskHandler::INTERRUPT_HANDLER, which, item, intEntries);
		    interruptQueue.enqueue ((void *) eventData);
		}

		if (eveMatches > 0) {
		    eventData = new EventData (eveMatches, AgentMaskHandler::EVENT_HANDLER, which, item, eveEntries);
		    eventQueue.enqueue ((void *) eventData);
		}

		event ();
		rc = TRUE;
	    }

	    DELETE_ARRAY_OBJECT(intEntries);
	    DELETE_ARRAY_OBJECT(eveEntries);
	}
    }

    return (rc);
}

/* EventData::EventData

   Purpose: This procedure is the constructor for class EventData.

     Input: count     = number of matching mask entries
		        (int)

	    eventType = INTERRUPT_HANDLER or EVENT_HANDLER
			(AgentMaskHandler::HandlerTypes)

	    itemType  = MESSAGE_ITEM, EVENT_ITEM or MEETING_ITEM
		 	(int)

	    item      = the incoming item
			(struct INCOMING *)

	    entries   = the matching mask entries 
			(class AgentMaskEntry **)
*/

EventData::EventData (int count, AgentMaskHandler::HandlerTypes eventType, int itemType, INCOMING *item, AgentMaskEntry **entries)
{
    int i;
    int j = 0;
    int actualCount = 0;
    int defaultCount = 0;
    int regularCount = 0;

	/* assertions on the parameters */

    assert (count > 0);
    assert ((eventType == AgentMaskHandler::INTERRUPT_HANDLER) || (eventType == AgentMaskHandler::EVENT_HANDLER));
    assert ((itemType == MESSAGE_ITEM) || (itemType == EVENT_ITEM) || (itemType == MEETING_ITEM));
    assert (item != NULL);
    assert (entries != NULL);

    for (i = 0; i < count; i++) {
	assert (entries[i] != NULL);
    }

	/* remember the base stuff */

    m_eventType = eventType;
    m_itemType  = itemType;
    m_item      = item;

	/* count the number of default and the number of regular entries */

    for (i = 0; i < count; i++) {
	if (entries[i] -> isDefault()) {
	    defaultCount += 1;
	} else {
	    regularCount += 1;
	}
    }

    if (regularCount > 0) {
	actualCount = regularCount;
    } else {
	actualCount = defaultCount;
    }

	/* allocate space for the handlers */

    m_count    = 0; 
    m_handlers = new AgentMaskHandler * [actualCount];

	/* now copy the handlers but do not include duplicate handlers */

#ifdef FIX_LATER
	/* the check for duplicate handlers can be made much for efficient */
	/* with a simple hash table arrangement                            */
#endif

    for (i = 0; i < count; i++) {

	    /* skip over default handlers if we have any regular handlers */

	if ((regularCount > 0) && (entries[i] -> isDefault())) {
	    continue;
	}

	    /* skip over duplicate handlers */

	for (j = 0; j < m_count; ++j) {

	    const AgentMaskHandler& handler = entries[i] -> getHandler();

	    if (m_handlers[j] -> compare (handler)) {
		break;
	    }
	}

	if (j < m_count) {
	    continue;
	}

	    /* add the handler to the list */

	m_handlers[m_count++] = entries[i] -> getHandler().clone();
    }
}

/* EventData::~EventData

   Purpose: This procedure is the destructor for class EventData.
*/

EventData::~EventData ()
{
   DELETE_OBJECT(m_item);

   for (int i = 0; i < m_count; i++) {
	DELETE_OBJECT(m_handlers[i]);
   }

   DELETE_ARRAY_OBJECT(m_handlers);
}


/* AgentMasks::maskCodeToShortString

   Purpose: Convert an error code into a short string

     Input: code = the error code
		   (int)

    Output: The procedure returns a DynamicString that contains the string. 
*/

static char *error[] = {
	"OK",
	"TOO_MANY_AgentMasks",
	"INVALID_HANDLE",
	"IN_USE",
	"DUPLICATE_ENTRY",
	"INVALID_ENTRY",
	"LOST_SERVER",
        "TIMEOUT",
	"NO_EVENTS",
	"NO_INTERRUPTS",
	"USED_HANDLE"
};

DynamicString AgentMasks::maskCodeToShortString (int code)
{
    DynamicString string;

    if ((code >= MASK_BEG_GENERAL) && (code <= MASK_END_GENERAL)) {
	string.append (error[code - MASK_BEG_GENERAL]);
    } else {
	string.append ("UNKNOWN");
    }

    return (string);
}

/* AgentMasks::maskCodeToLongString

   Purpose: Convert a mask error code into a long human-readable string

     Input: code   = the error code
		     (int)

	    handle = the mask handle
		     (UINT_32)

	    entry  = a mask entry that we were trying to add or remove
		     (const AgentMaskEntry *)
 
   Output: The procedure returns a DynamicString that contains the long string.
*/

DynamicString AgentMasks::maskCodeToLongString (int code, UINT_32 handle, const AgentMaskEntry *entry) 
{
    char temp[16];
    char *entryString;
    DynamicString string; 

    switch (code) {

	case MASK_OK:
	    break;

	case MASK_TOO_MANY:
	    string.append ("too many masks");
	    break;

	case MASK_INVALID_HANDLE:
	    sprintf (temp, "%u", handle);
	    string.append ("no mask with handle ");
	    string.append (temp);
	    break;

	case MASK_IN_USE:
	    string.append ("mask is in use as the current meeting, message or event mask");
	    break;

	case MASK_DUP_ENTRY:
	    entryString = Agent_AgentMaskEntryToString (entry);
	    string.append ("mask already contains the entry \"");
	    string.append (entryString);
	    string.append ("\"");
	    DELETE_ARRAY_OBJECT(entryString);
	    break;

	case MASK_INVALID_ENTRY:
	    entryString = Agent_AgentMaskEntryToString (entry);
	    string.append ("mask does not contain the entry \"");
	    string.append (entryString);
	    string.append ("\"");
	    DELETE_ARRAY_OBJECT(entryString);
	    break;

        case MASK_LOST_BACKGROUND:
	    string.append ("background connection to server has been lost");
	    break;

	case MASK_TIMEOUT:
	    string.append ("timeout while waiting for an incoming item");
	    break;

	case MASK_NO_EVENTS:
	    string.append ("event handlers are not allowed");
	    break;

	case MASK_NO_INTERRUPTS:
	    string.append ("interrupt handlers are not allowed");
	    break;

	case MASK_USED_HANDLE:
	    sprintf (temp, "%u", handle);
	    string.append ("desired handle ");
	    string.append (temp);
	    string.append (" is in use");
	    break;

	default:
	    string.append ( "unknown error");
	    break;
    }

    return (string);
}

/* AgentMasksPacker::getPackedSize

   Purpose: Calculate and return the packed size (in bytes) of the AgentMasks
*/

UINT_32 AgentMasksPacker::getPackedSize (void)
{
    UINT_32 size = 0;

	/* number of masks */

    size += sizeof(UINT_32);

	/* handles of the message, meeting and event masks */

    size += 3 * sizeof(UINT_32);

	/* get the size of each individual mask */

    HashNode * hashNode;
    HashSearch hashSearch (m_masks.maskTable);

    while ((hashNode = hashSearch.next()) != NULL) {
 	AgentMask *mask = (AgentMask *) hashNode -> data;
	AgentMaskPacker packer (*mask);
	size += packer.getPackedSize ();
    }

	/* done */

    return (size);
}

/* AgentMasksPacker::pack

   Purpose: Pack the masks into a bytestream
*/

char *AgentMasksPacker::pack (char *bp)
{
    UINT_32 hostLong;
    UINT_32 netLong;

	/* add the number of masks */

    hostLong = m_masks.mask_count;
    netLong  = htonl (hostLong);
    bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	/* add the handles of the message, event and meeting masks */

    for (int i = 0; i < NUM_ITEMS; ++i) {
	hostLong = m_masks.masks[i] -> getHandle();
	netLong  = htonl (hostLong);
	bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));
    }

	/* add each individual mask */

    HashNode * hashNode;
    HashSearch hashSearch (m_masks.maskTable);

    while ((hashNode = hashSearch.next()) != NULL) {
 	AgentMask *mask = (AgentMask *) hashNode -> data;
	AgentMaskPacker packer (*mask);
	bp = packer.pack (bp);
    }

	/* done */

    return (bp);
}

/* AgentMaskPacker::getPackedSize

   Purpose: Calculate and return the packed size (in bytes) of the AgentMask
*/

UINT_32 AgentMaskPacker::getPackedSize (void)
{
    UINT_32 size = 0;

	/* mask handle */

    size += sizeof (UINT_32);

	/* mask type */

    size += sizeof (UINT_8);

	/* the mask entries (if we have mask entries) */

    if (m_mask.type == MASK_ACCEPT_SOME) {

	    /* number of mask entries */

	size += sizeof(UINT_32);

	    /* mask entries */

	for (int i = 0; i < m_mask.entries.get_count(); ++i) {
	    AgentMaskEntry *maskEntry = (AgentMaskEntry *) m_mask.entries.peek (i);
	    AgentMaskEntryPacker packer (*maskEntry);
	    size += packer.getPackedSize ();
	}
    }

	/* done */

    return (size);
}

/* AgentMaskPacker::pack

   Purpose: Pack the AgentMask into a bytestream
*/

char *AgentMaskPacker::pack (char *bp)
{
    UINT_8 netByte;
    UINT_32 hostLong;
    UINT_32 netLong;

	/* mask handle */

    hostLong = m_mask.handle;
    netLong  = htonl (hostLong);
    bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	/* mask type */

    netByte = (UINT_8) m_mask.type; 
    bp = fast_memcpy (bp, (char *) &netByte, sizeof(UINT_8));

	/* mask entries (if we have any) */

    if (m_mask.type == MASK_ACCEPT_SOME) {
	
	    /* number of mask entries */

	hostLong = (UINT_32) m_mask.entries.get_count ();
	netLong  = htonl (hostLong);
	bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));
	
	    /* mask entries */

	for (int i = 0; i < m_mask.entries.get_count(); ++i) {
	    AgentMaskEntry *maskEntry = (AgentMaskEntry *) m_mask.entries.peek (i);
	    AgentMaskEntryPacker packer (*maskEntry);
	    bp = packer.pack (bp);
	}
    }

	/* done */

    return (bp);
}

/* AgentMaskEntryPacker::getPackedSize

   Purpose: Calculate and return the packed size (in bytes) of the AgentMaskEntry
*/

UINT_32 AgentMaskEntryPacker::getPackedSize (void)
{
    UINT_32 size = 0;

	/* default or not default flag */

    size += sizeof(UINT_8);

	/* AgentId if it is not a default entry */

    if (m_maskEntry.id != NULL) {
	AgentIdPacker packer (*m_maskEntry.id);
	size += packer.getPackedSize ();
    }

	/* wheter or not we have a tag */

    size += sizeof(UINT_8);

	/* tag */

    if (m_maskEntry.tag != NULL) {
	size += sizeof(UINT_32);
	size += strlen(m_maskEntry.tag) + 1;
    }

	/* whether or not we have a handle */

    size += sizeof (UINT_8);

	/* handle */

    if (m_maskEntry.handler != NULL) {
	size += sizeof (UINT_8);			    // event or interrupt
	size += m_maskEntry.handler -> getPackedSize ();    // language-specific portion
    }

	/* done */

    return (size);
}

/* AgentMaskEntryPacker::pack

   Purpose: Pack the AgentMaskEntry into a bytestream
*/

char *AgentMaskEntryPacker::pack (char *bp)
{
    UINT_8 netByte;

	/* default or not default flag */

    netByte = (m_maskEntry.id == NULL) ? 1 : 0;
    bp = fast_memcpy (bp, (char *) &netByte, sizeof(UINT_8));

	/* AgentId if it is not a default entry */

    if (m_maskEntry.id != NULL) {
	AgentIdPacker packer (*m_maskEntry.id);
	bp = packer.pack (bp);
    }

	/* whether or not we have a tag */

    netByte = (m_maskEntry.tag == NULL) ? 0 : 1;
    bp = fast_memcpy (bp, (char *) &netByte, sizeof(UINT_8));
   
	/* tag */

    if (m_maskEntry.tag != NULL) {
	bp = linearize_string (bp, m_maskEntry.tag);
    }

	/* whether or not we have a handler */

    netByte = (m_maskEntry.handler == NULL) ? 0 : 1;
    bp = fast_memcpy (bp, (char *) &netByte, sizeof(UINT_8));

	/* handler */

    if (m_maskEntry.handler != NULL) {
	netByte = (m_maskEntry.handler -> getType() == AgentMaskHandler::EVENT_HANDLER) ? 1 : 0;
	bp = fast_memcpy (bp, (char *) &netByte, sizeof(UINT_8));
	bp = m_maskEntry.handler -> pack (bp);
    }

	/* done */

    return (bp);
}

/* AgentMasksUnpacker::unpack

   Purpose: Unpack a byte stream into a AgentMasks instance
*/

char *AgentMasksUnpacker::unpack (char *bp, char *end_bp)
{
    UINT_32 hostLong;
    UINT_32 messageHandle;
    UINT_32 eventHandle;
    UINT_32 meetingHandle;

	/* get the number of masks */

    if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	return ((char *) NULL);
    }

    m_masks.mask_count = hostLong;

	/* get the handles of the message, meeting and event masks */

    if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	return ((char *) NULL);
    }

    meetingHandle = hostLong;

    if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	return ((char *) NULL);
    }

    messageHandle = hostLong;

    if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	return ((char *) NULL);
    }

    eventHandle = hostLong;

	/* get the masks */

    UINT_32 maxHandle = 0;

    for (UINT_32 i = 0; i < m_masks.mask_count; ++i) {

	int there;

	    /* get a mask */

	AgentMask *mask = new AgentMask ();
	AgentMaskUnpacker unpacker (*mask, m_masks);

	if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
	    DELETE_OBJECT(mask);
	    return ((char *) NULL);
	}

	    /* add the mask to the table (bailing out of the routine  */
	    /* with an error if there is already a mask with the same */
            /* handle)                                                */

	UINT_32 handle = mask -> getHandle ();
	m_masks.maskTable -> add (handle, (char *) NULL, (void *) mask, there);

	if (there) {
	    DELETE_OBJECT(mask);
	    return ((char *) NULL);
	}

	    /* see if we now have a larger handle */

	if (handle > maxHandle) {
	    maxHandle = handle;
	}	
    }

	/* first handle to try when the agent creates its next mask is */
	/* maxHandle + 1                                               */

    m_masks.mask_handle = (maxHandle % MAX_U32) + 1;
    
	/* make sure that the message, meeting and event masks exist */

    m_masks.masks[MESSAGE_ITEM] = m_masks.maskLookupInt (messageHandle);
    m_masks.masks[MEETING_ITEM] = m_masks.maskLookupInt (meetingHandle);
    m_masks.masks[  EVENT_ITEM] = m_masks.maskLookupInt (eventHandle);

    for (int j = 0; j < NUM_ITEMS; ++j) {
	if (m_masks.masks[j] == NULL) {
	    return ((char *) NULL);
	}
    }

	/* done */

    return (bp);
}

/* AgentMaskUnpacker::unpack

   Purpose: Unpack a byte stream into a AgentMask instance
*/

char *AgentMaskUnpacker::unpack (char *bp, char *end_bp)
{
    UINT_8 hostByte;
    UINT_32 hostLong;

	/* get the mask handle */

    if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	return ((char *) NULL);
    }

    m_mask.handle = hostLong;

	/* get the mask type */

    if ((bp = breakout_byte (bp, end_bp, hostByte)) == NULL) {
	return ((char *) NULL);
    }

    if ((hostByte != MASK_ACCEPT_ALL) && (hostByte != MASK_ACCEPT_NONE) && (hostByte != MASK_ACCEPT_SOME)) {
	return ((char *) NULL);
    }

    m_mask.type = hostByte;

	/* get the mask entries (if we have any) */

    if (m_mask.type == MASK_ACCEPT_SOME) {

	UINT_32 nEntries;

	    /* first get the number of entries */

	if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	    return ((char *) NULL);
	}
    
	nEntries = hostLong;
 
	    /* then get each entry */

	for (UINT_32 i = 0; i < nEntries; ++i) {

	    AgentMaskEntry *maskEntry = new AgentMaskEntry ();
	    AgentMaskEntryUnpacker unpacker (*maskEntry, m_masks);

	    if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
		DELETE_OBJECT(maskEntry);
		return ((char *) NULL);
	    }

	    m_mask.entries.enqueue ((void *) maskEntry);
	}
    }

	/* done */

    return (bp);
}

/* AgentMaskEntryUnpacker::unpack

   Purpose: Unpack a byte stream into a AgentMaskEntry instance
*/

char *AgentMaskEntryUnpacker::unpack (char *bp, char *end_bp)
{
    UINT_8 flag;

	/* unpack the AgentId */

    if ((bp = breakout_byte (bp, end_bp, flag)) == NULL) {
	return ((char *) NULL);
    }

    if (flag) {

	m_maskEntry.id = NULL;

    } else {

	AgentId *id = new AgentId ();
	AgentIdUnpacker unpacker (*id);

	if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
	    DELETE_OBJECT(id);
	    return ((char *) NULL);
	}

	m_maskEntry.id = id;
    }

	/* unpack the tag */

    if ((bp = breakout_byte (bp, end_bp, flag)) == NULL) {
	return ((char *) NULL);
    }
  
    if (!flag) {
	
	m_maskEntry.tag = NULL;

    } else {

	char *string;

	if ((bp = breakout_string (bp, end_bp, &string)) == NULL) {
	    return ((char *) NULL);
	}
	
	m_maskEntry.tag = string;
    }

	/* unpack the handler */

    if ((bp = breakout_byte (bp, end_bp, flag)) == NULL) {
	return ((char *) NULL);
    }
  
    if (!flag) {

	m_maskEntry.handler = NULL;

    } else {

	AgentMaskHandler::HandlerTypes type;

	    /* event or interrupt handler */

	if ((bp = breakout_byte (bp, end_bp, flag)) == NULL) {
	    return ((char *) NULL);
	}

	if (flag) {
	    type = AgentMaskHandler::EVENT_HANDLER;
	} else {
	    type = AgentMaskHandler::INTERRUPT_HANDLER;
	}

	AgentMaskHandler *maskHandler = m_masks.createEmptyHandler (type);

	    /* language-specific portion */

	if ((bp = maskHandler -> unpack (bp, end_bp)) == NULL) {
	    DELETE_OBJECT(maskHandler);
	    return ((char *) NULL);
	}

	m_maskEntry.handler = maskHandler;
    }

	/* we must either have an AgentId (i.e., a non-default handler) */
	/* or a handler (i.e., a default handler)                       */

    if ((m_maskEntry.id == NULL) && (m_maskEntry.handler == NULL)) {
	return ((char *) NULL);
    }

	/* done */

    return (bp);
}
