/* Agent Tcl
   Bob Gray
   10 January 1996

   genAgentMasks.h

   This file defines the mask structures.

   Copyright (c) 1995-1998, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _GEN_MASKS_H
#define _GEN_MASKS_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

    /* forward declarations */

class AgentId;
class Hash;

#ifndef _SUPP_DSTRING_H
#include "suppDString.h"	// DynamicString
#endif
#ifndef _GEN_SERVER_FLAGS_H
#include "genServerInterface.h"	// SubmissionTypes
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"		// BOOLEAN
#endif

#include "genMeeting.h"
#include "genMessage.h"
#include "genQueue.h"
#include "genRestrict.h"
#include "platSigio.h"
#include "platTimers.h"

const int MASK_TABLE_SIZE = 32;

    /* forward definition so we can compile */

class AgentMasks;
class AgentMaskHandler;
class AgentMaskEntry;

    /* error codes */

const int MASK_BEG_GENERAL	   = 0;
const int MASK_OK		   = 0;
const int MASK_TOO_MANY	   = 1;
const int MASK_INVALID_HANDLE  = 2;
const int MASK_IN_USE	   = 3;
const int MASK_DUP_ENTRY	   = 4;
const int MASK_INVALID_ENTRY   = 5; 
const int MASK_LOST_BACKGROUND = 6;
const int MASK_TIMEOUT	   = 7;
const int MASK_NO_EVENTS	   = 8;
const int MASK_NO_INTERRUPTS   = 9;
const int MASK_USED_HANDLE	   = 10;
const int MASK_END_GENERAL	   = 10;

    /* mask types */ 

const int MASK_ACCEPT_ALL	= 0;
const int MASK_ACCEPT_SOME	= 1;
const int MASK_ACCEPT_NONE	= 2;

    /* an event or interrupt handler */

class AgentMaskHandler
{
    friend class AgentMasks;

public:

    enum HandlerTypes {
	EVENT_HANDLER = 0x0,
	INTERRUPT_HANDLER = 0x1
    };

protected:

	/* EVENT_HANDLER or INTERRUPT_HANDLER */

    HandlerTypes type;     

public:

	/* contents */

    HandlerTypes getType (void) const {
	return (type);
    } 

	/* compare one handler to another */

    virtual int compare (const AgentMaskHandler &handle) const = 0;

	/* create a copy of this handler -- can't do this with the   */
        /* assignment operator or a copy constructor since we want   */
	/* to make a copy of a derived class not this abstract class */

    virtual AgentMaskHandler *clone (void) const = 0;

	/* pack and unpack the handler */

    virtual UINT_32 getPackedSize (void) const = 0;
    virtual char *pack (char *bp) const = 0;
    virtual char *unpack (char *bp, char *end_bp) = 0;
 
	/* get a string representation of the handler */

    virtual DynamicString getStringRepresentation (void) const = 0;

	/* constructor */

    AgentMaskHandler (HandlerTypes type);

	/* destructor -- this virtual destructor does nothing but is  */
	/* included so that a user-define "delete" operator in a base */
	/* class will work correctly				      */

    virtual ~AgentMaskHandler ();
};

    /* event data */

struct EventData
{
    AgentMaskHandler::HandlerTypes m_eventType;       /* INTERRUPT_HANDLER or EVENT_HANDLER       */
    int m_itemType;                              /* MESSAGE_ITEM, MEETING_ITEM or EVENT_ITEM */
    INCOMING *m_item;                            /* the incoming item                        */
    int m_count;                                 /* number of handlers                       */
    AgentMaskHandler **m_handlers;                    /* the handlers                             */

	/* constructor and destructor */


    EventData (int count, AgentMaskHandler::HandlerTypes eventType, int itemType, INCOMING *item, AgentMaskEntry **entries);
   ~EventData ();
}; 

    /* an entry in a mask */
    
class AgentMaskEntry          
{
    friend class AgentMaskEntryPacker;
    friend class AgentMaskEntryUnpacker;
    friend class AgentMask;
    friend class AgentMaskUnpacker;
    friend class AgentMasks;
 
	/* static data */

    static UINT_32 localIp;    /* 127.0.0.1         */
    static UINT_32 actualIp;   /* actual machine IP */

	/* instance data */

    AgentId *id;                   /* filter on source agent identification */
    char *tag;                     /* tag for EVENTS                        */
    AgentMaskHandler *handler;     /* handler for this mask                 */

	/* private constructor */

    AgentMaskEntry (void);

public:

	/* public constructors */

    AgentMaskEntry (AgentId *id, char *tag, AgentMaskHandler *handler);
    AgentMaskEntry (const AgentMaskEntry& mask);

	/* assignment operator */

    AgentMaskEntry &operator = (const AgentMaskEntry &mask);

	/* exact comparision between the entry and some other entry */

    int exactCompare (AgentMaskEntry &entry) const;

	/* inexact comparison between the entry and an incoming item */

    int inexactCompare (int which, INCOMING &item) const;

	/* member access */

    BOOLEAN isDefault (void) const {
	return (id == NULL) ? e_TRUE : e_FALSE;
    }

    BOOLEAN hasTag (void) const {
	return (tag != NULL) ? e_TRUE : e_FALSE;
    }

    BOOLEAN hasHandler (void) const {
	return (handler != NULL) ? e_TRUE : e_FALSE;
    }

    const AgentId& getId (void) const {
	assert (id != NULL);
	return (*id);
    }

    const char *getTag (void) const {
	return (tag);
    }

    const AgentMaskHandler& getHandler (void) const {
	assert (handler != NULL);
	return (*handler);
    }

	/* destructor */

   ~AgentMaskEntry ();
};

    /* a single mask */

class AgentMask
{
    friend class AgentMasks;
    friend class AgentMaskPacker;
    friend class AgentMaskUnpacker;
    friend class AgentMasksUnpacker;

	/* MASK_ACCEPT_NONE, MASK_ACCEPT_SOME or MASK_ACCEPT_ALL */

    int type;

	/* handle of this mask */

    UINT_32 handle;

	/* entries in the mask */

    QUEUE entries;

	/* constructors are private so that only class AgentMasks or class */
	/* AgentMasksUnpacker can create an instance of class AgentMask         */

    AgentMask (void);
    AgentMask (int type, UINT_32 handle);
    AgentMask (const AgentMask &mask);

	/* assignment operator */

    AgentMask &operator= (const AgentMask &mask);

	/* empty the mask and set the flag */

    void empty_and_set (int flag);

	/* fill the mask so that it accepts any incoming item */

    void fill (void) {
	empty_and_set (MASK_ACCEPT_ALL);
    };

	/* empty the mask so that it accepts no incoming items */

    void empty (void) {
	empty_and_set (MASK_ACCEPT_NONE);
    };

	/* add an entry to the mask */

    int add (AgentMaskEntry &entry);

	/* remove an entry from the mask */

    int remove (AgentMaskEntry &entry);

public:

	/* contents */

    int getType (void) const {
	return (type);
    }

    int getHandle (void) const {
	return (handle);
    }

    int getCount (void) const {
	return (entries.get_count());
    }

    const AgentMaskEntry &getEntry (int i) const {
	assert (i < entries.get_count());
	return (* (AgentMaskEntry *) entries.peek(i));
    }

	/* filter an incoming item through the mask */

    int filter (int which, INCOMING &item);

	/* destructor */

   ~AgentMask ();
};

    /* all of the masks defined in the current agent */

class AgentMasks
{
    friend class AGENT;
    friend class AgentMeetings;
    friend class AgentMasksPacker;
    friend class AgentMasksUnpacker;

private:   /* member functions */

	/*
	 *--------------------------------------------------------------
	 * Section 2: Events and interrupts 
	 *--------------------------------------------------------------
	 */

	/* 
         * fire all event handlers that are ready or fire all event
	 * handlers that are ready for a particular item
	 */

    void fireAllEvents_noLock (int which); 
    int fireEvents_noLock (int which, INCOMING *item);

	/*
	 *--------------------------------------------------------------
	 * Section 3: AgentMasks
	 *--------------------------------------------------------------
	 */

	/* 
	 * add an incoming item to the appropriate queue
	 */

    void addIncomingToQueue_noLock (int which, INCOMING *item);

protected: /* variables */

	/*
	 *--------------------------------------------------------------
	 * Section 1: Miscellaneous
	 *--------------------------------------------------------------
	 */

	/*
	 * agent associated with this AgentMasks instance
	 */

    AGENT *agent;

	/*
	 *--------------------------------------------------------------
	 * Section 2: Events and interrupts 
	 *--------------------------------------------------------------
	 */

	/* 
	 * allowEvents is TRUE if we can define event handlers;        
	 * allowInterrupts is TRUE if we can define interrupt handlers
	 */

    int allowInterrupts;
    int allowEvents;

	/* 
	 * queue of pending events and of pending interrupts 
	 */

    QUEUE eventQueue;
    QUEUE interruptQueue;

	/*
	 *--------------------------------------------------------------
	 * Section 3: AgentMasks
	 *--------------------------------------------------------------
	 */

	/* 
	 * hash table of masks 
	 */

    Hash *maskTable;

	/* 
	 * number of masks and the next untried handle
	 */

    UINT_32 mask_count;
    UINT_32 mask_handle;

	/* 
	 * message, meeting and event masks and the corresponding queues
	 */

    AgentMask *masks[NUM_ITEMS];
    QUEUE queues[NUM_ITEMS];

protected: /* member functions */

	/*
	 *--------------------------------------------------------------
	 * Section 1: Miscellaneous
	 *--------------------------------------------------------------
	 */

	/* 
	 * constructor and destructor  
	 */

    AgentMasks (AGENT *agent);
    virtual ~AgentMasks ();

	/*
	 * delete all incoming items
	 */

    void deleteAllIncomingItems (void);

	/*
	 * bootstrap masks for an incoming agent
	 */

    int arrivingBootstrap (ServerInterface::SubmissionTypes type);

	/*
	 *--------------------------------------------------------------
	 * Section 2: Events and interrupts 
	 *--------------------------------------------------------------
	 */

	/* 
	 * function that is called when an event or interrupt is ready
	 */
 
    virtual void event (void) = 0;

	/*
	 *--------------------------------------------------------------
	 * Section 3: AgentMasks
	 *--------------------------------------------------------------
	 */

	/* 
         * create a mask or lookup a mask (internal version)
	 */

    int maskCreateKnownInt (int type, UINT_32 handle);
    AgentMask *maskCreateInt (int type);
    AgentMask *maskLookupInt (UINT_32 handle);

	/*
	 * queue up a meeting (called from MEETINGS)
	 */

    void addMeetingToQueue (INCOMING_MEETING *meeting);

public: /* member functions */

	/*
	 *--------------------------------------------------------------
	 * Section 1: Miscellaneous
	 *--------------------------------------------------------------
	 */

	/*
	 * bootstrap masks for an agent started on the current machine
	 */

    int startingBootstrap (void);

	/*
	 * convert mask codes to short and long strings
	 */

    DynamicString maskCodeToShortString (int code);
    DynamicString maskCodeToLongString (int code, UINT_32 handle, const AgentMaskEntry *entry);

	/*
	 * factory method to create an empty instance of the appropriate 
	 * AgentMaskHandler subclass
	 */

    virtual AgentMaskHandler *createEmptyHandler (AgentMaskHandler::HandlerTypes type) = 0;

	/*
	 *--------------------------------------------------------------
	 * Section 2: Events and interrupts 
	 *--------------------------------------------------------------
	 */

	/* 
	 * allow event handlers or interrupt handlers
	 */

    void turnOnEvents (void) {
	allowEvents = TRUE;
    } 

    void turnOnInterrupts (void) {
	allowInterrupts = TRUE;
    } 

	/* 
	 * return TRUE if there are pending event handlers or return 
	 * TRUE if there are pending interrupt handlers
	 */

    int pendingEvents (void) {
	return (eventQueue.get_count() > 0);
    } 

    int pendingInterrupts (void) {
	return (interruptQueue.get_count() > 0);
    } 

	/* 
	 * get a pending event or get a pending interrupt
	 */

    EventData *getNextEvent (void) {
	return ((EventData *) eventQueue.dequeue());
    } 

    EventData *getNextInterrupt (void) {
	return ((EventData *) interruptQueue.dequeue());
    } 

	/*
	 *--------------------------------------------------------------
	 * Section 3: AgentMasks 
	 *--------------------------------------------------------------
	 */

	/*
	 * create a new mask, create a new mask with the given handle
	 * or delete an existing mask
	 */ 

    int maskCreate (UINT_32 &handle);			/* PUBLIC */
    int maskCreateKnown (UINT_32 handle);
    int maskDelete (UINT_32 handle);			/* PUBLIC */

	/* 
	 * fill a mask so that it accepts any incoming item or
	 * empty a mask so that it rejects all incoming items
	 */

    int maskFill (UINT_32 handle);			/* PUBLIC */
    int maskEmpty (UINT_32 handle);			/* PUBLIC */

	/* 
	 * add an entry to a mask or remove an entry from a mask
	 */

    int maskAdd (UINT_32 handle, AgentMaskEntry &entry);	/* PUBLIC */
    int maskRemove (UINT_32 handle, AgentMaskEntry &entry); /* PUBLIC */

	/* 
	 * get a mask
	 */

    AgentMask *maskGet (UINT_32 handle);		/* PUBLIC */

	/* 
	 * get the current handle of one of the three distinguished masks
	 */

    UINT_32 getHandle (int which) {			/* PUBLIC */
	return (masks[which] -> handle);
    }

	/* 
	 * update the current handle of one of the three distinguished masks
	 */

    virtual int updateHandle (int which, UINT_32 handle);	/* PUBLIC */

	/*
	 * handle an incoming item 
	 */

    int handleIncoming_noLock (MESSAGE *item);

	/*
	 * get an incoming item from the queue
	 */

    INCOMING *getIncoming_noLock (int which);

	/*
	 * check if there is an incoming item
	 */

    BOOLEAN checkIncoming_noLock (int which);
};

/*
 * Iterators that pack and unpack the current masks -- we use iterators
 * since eventually we want to be able to pack the masks in fixed-size chunks
 * so that we can overlap the packing and transmission process.
 */

class AgentMasksPacker
{
    AgentMasks &m_masks;	// AgentMasks that we are packing into a byte stream

public:

	// constructor

    AgentMasksPacker (AgentMasks &masks):
	m_masks (masks)
    {
	// empty
    }

	// get the packed size of the AgentMasks

    UINT_32 getPackedSize (void);

	// pack the AgentMasks into a buffer

    char *pack (char *buffer);
};

class AgentMasksUnpacker
{
    AgentMasks &m_masks;	// AgentMasks into which we are unpacking a byte stream

public:

	// constructor

    AgentMasksUnpacker (AgentMasks &masks):
	m_masks (masks)
    {
	// empty
    }

	// unpack from a buffer

    char *unpack (char *buffer, char *endBuffer);
};

/*
 * Iterators that pack and unpack an individual mask -- we use iterators
 * since eventually we want to be able to pack the mask in fixed-size chunks
 * so that we can overlap the packing and transmission process.
 */

class AgentMaskPacker
{
    AgentMask &m_mask;	// AgentMasks that we are packing into a byte stream

public:

	// constructor

    AgentMaskPacker (AgentMask &mask):
	m_mask (mask)
    {
	// empty
    }

	// get the packed size of the AgentMask

    UINT_32 getPackedSize (void);

	// pack the AgentMask into a buffer

    char *pack (char *buffer);
};

class AgentMaskUnpacker
{
    AgentMasks &m_masks;	// set of AgentMasks associated with this AgentMask 
    AgentMask &m_mask;	// AgentMask into which we are unpacking a byte stream

public:

	// constructor

    AgentMaskUnpacker (AgentMask &mask, AgentMasks& masks):
	m_masks (masks),
	m_mask (mask)
    {
	// empty
    }

	// unpack from a buffer

    char *unpack (char *buffer, char *endBuffer);
};

/*
 * Iterators that pack and unpack a mask entry -- we use iterators since
 * eventually we want to be able to pack the mask entry in fixed-size chunks
 * so that we can overlap the packing and transmission process.
 */

class AgentMaskEntryPacker
{
    AgentMaskEntry &m_maskEntry;	// AgentMaskEntry that we are packing into a byte stream

public:

	// constructor

    AgentMaskEntryPacker (AgentMaskEntry &maskEntry):
	m_maskEntry (maskEntry)
    {
	// empty
    }

	// get the packed size of the AgentMaskEntry

    UINT_32 getPackedSize (void);

	// pack the AgentMaskEntry into a buffer

    char *pack (char *buffer);
};

class AgentMaskEntryUnpacker
{
    AgentMasks &m_masks;		// set of AgentMasks associated with this AgentMaskEntry 
    AgentMaskEntry &m_maskEntry;	// AgentMaskEntry into which we are unpacking a byte stream

public:

	// constructor

    AgentMaskEntryUnpacker (AgentMaskEntry &maskEntry, AgentMasks& masks):
	m_masks (masks),
	m_maskEntry (maskEntry)
    {
	// empty
    }

	// unpack from a buffer

    char *unpack (char *buffer, char *endBuffer);
};

#endif
