/* Agent Tcl
   Bob Gray
   4 April 1996

   genMeeting.cc

   This file implements the data structures that store information about
   meetings.

   Copyright (c) 1995-1998, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"	// platform-specific definitions
#include "platDelete.h"
#include "platCharact.h"
#include "platSigio.h"
#include "mesgTcpip.h"		// TCP/IP routines
#include "agentAgentId.h"	// AgentId
#include "genAgentIO.h"		// AgentIO

#include "genMeeting.h"
#include "genAgent.h"
#include "genAgentConn.h"
#include "genLocation.h"	// LOCATION
#include "genManager.h"
#include "genMasks.h"
#include "genTransmit.h"

    /* size of the meeting handle hash table */

const int HANDLE_TABLE_SIZE	= 0x64;

    /* size for the copy buffer */

const int COPY_BUFFER_SIZE	= 4096;

/* MEETING_INFO::MEETING_INFO

   Purpose: These procedures are the constructor for class MEETING_INFO.
*/

MEETING_INFO::MEETING_INFO (const AgentId &p_id):
    status		(MEET_REQUESTING),
    lastStatus		(MEET_REQUESTING),
    messageReady	(e_FALSE),
    fileReady		(e_FALSE),
    id			(p_id),
    security		(),
    port		(),
    localHandle		(UNDEF_MEETING_HANDLE),
    remoteHandle	(UNDEF_MEETING_HANDLE)
{
    /* empty */
}

MEETING_INFO::MEETING_INFO (const INCOMING_MEETING &meeting):
    status		(MEET_REQUESTED),
    lastStatus		(MEET_REQUESTED),
    messageReady	(e_FALSE),
    fileReady		(e_FALSE),
    id			(*(meeting.id)),
    security 		(*(meeting.security)),
    port		(*(meeting.port)),
    localHandle		(meeting.localHandle),
    remoteHandle	(meeting.remoteHandle)
{
    /* empty */
}

MEETING_INFO::MEETING_INFO (const MEETING_INFO& meeting):
    status		(meeting.status),
    lastStatus		(meeting.lastStatus),
    messageReady	(meeting.messageReady),
    fileReady		(meeting.fileReady),
    id			(meeting.id),
    security		(meeting.security),
    port		(meeting.port),
    localHandle		(meeting.localHandle),
    remoteHandle	(meeting.remoteHandle)
{
    /* empty */
}

/* MEETING::MEETING and MEETING::~MEETING

   Purpose: These procedures are the constructor and destructor for class
            MEETING.
*/


MEETING::MEETING (const AgentId &p_id, AgentMeetings *p_meetings):
    MEETING_INFO	(p_id),
    previous		(NULL),
    next		(NULL),
    sockfd		(-1),
    lastSigioFlags	(0),
    handler		((SigioHandler *) NULL),
    meetings		(p_meetings)
{
    /* empty */
}

MEETING::MEETING (const INCOMING_MEETING &p_meeting, AgentMeetings *p_meetings):
    MEETING_INFO	(p_meeting),
    previous		(NULL),
    next		(NULL),
    sockfd		(-1),
    lastSigioFlags	(0),
    handler		((SigioHandler *) NULL),
    meetings		(p_meetings)
{
    /* empty */
} 

MEETING::~MEETING () {
    terminateConnection ();
} 

/* MEETING::turnBackgroundOn and MEETING::turnBackgroundOn

   Purpose: These procedures associate a socket with the meeting.
*/
 
void MEETING::turnBackgroundOn (int flags) 
{
    assert (handler == NULL);
    MonitorGroup group ((meetings -> agent) -> getMonitorLock());
    handler = new SigioHandler (sockfd, (SigioTrap *) this, flags, group);
}

void MEETING::resetBackground (int flags)
{
    assert (handler != NULL);
    handler -> changeFlags (flags);
}

void MEETING::turnBackgroundOff (void)
{
    if (handler != NULL) {
	DELETE_OBJECT(handler);
    }
}

void MEETING::startConnection (int p_sockfd, int flags) 
{
    sockfd = p_sockfd;
    turnBackgroundOn (flags);
}

void MEETING::terminateConnection (void) 
{
    turnBackgroundOff ();

    if (sockfd > 0) {
	close (sockfd);
    }
}

/* MEETING::sigioAlarm 

   Purpose: Called when the meeting socket is ready for reading or writing

     Input: fd     = socket descriptor
		     (int)

	    flags  = OR'ed combination of READ_READY, WRITE_READY, EXCEPT_READY
		     (int)

    Output: The procedure records the flags and calls the 
	    AgentMeetings::meetingStatusChange callback method so that the 
	    status of the meeting can be updated later when the system is in 
	    a "safe" state.
*/

void MEETING::sigioAlarm (int /*fd*/, int flags)
{
	// NOTE -- we do not need to acquire the MEETINGS monitor since it was 
	// already acquired by the lower-level routines that called this
	// function
 
	// NOTE -- We know that this routine is only called when the
	// Tcl interpreter is in a safe state since the SigioHandler
	// class calls up to the Tcl interpreter level.

	// remember the flags

    lastSigioFlags |= flags;

	// check for a status change and wakeup any waiters if necessary

    handleStatusChange ();

    if ((status != lastStatus) || (fileReady) || (messageReady)) {
	(meetings -> agent -> getMonitorLock()).wakeup ();
    }
}

/* MEETING::handleStatusChange

   Purpose: Detect and handle any change in meeting status

     Input: None
  
    Output: None
*/

void MEETING::handleStatusChange (void)
{
    UINT_32 l_port;
    UINT_32 ip;
    int new_status;
    int new_sockfd;

	    /* determine what is happening */

    switch (status) {

    case MEET_ACCEPTING:

        if (!(lastSigioFlags & SigioTrap::e_READ_READY)) {
	    break;
        }

        if (tcpip_accept (sockfd, new_sockfd, SystemUtil::getCurrentWall()) != e_TCPIP_OK) {
	    new_status = MEET_FAILED;
        } else {
	    new_status = MEET_COMPLETE;
        }

        terminateConnection ();

        meetings -> spliceMeetingFromList (this);
        status = new_status;
        meetings -> spliceMeetingIntoList (this);

        if (new_status == MEET_COMPLETE) {
	    startConnection (new_sockfd, SigioTrap::e_READ_READY);
        }

        break;

    case MEET_CONNECTING:

        if (!(lastSigioFlags & SigioTrap::e_WRITE_READY)) {
	    break;
        }

        if (tcpip_getPeername (sockfd, l_port, ip) != e_TCPIP_OK) {
	    new_status = MEET_FAILED;
        } else {
  	    new_status = MEET_COMPLETE;
        }

        if (new_status == MEET_FAILED) {
            terminateConnection ();
        } else {
	    resetBackground (SigioTrap::e_READ_READY);
        }

        meetings -> spliceMeetingFromList (this);
        status = new_status;
        meetings -> spliceMeetingIntoList (this);
        break;

    case MEET_COMPLETE:

        char flag;
	int nread;

        if (!(lastSigioFlags & SigioTrap::e_READ_READY)) {
	    break;
        }

	    /* don't do anything if we have unprocessed messages */

        if ((fileReady) || (messageReady)) {
	    break;
        }

	    /* otherwise see what we got */

	    /* 
	     * NOTE - NOTE - NOTE - NOTE - NOTE - NOTE - NOTE - NOTE
	     *
	     * Do not use "recv" in place of tcpip_blockingReadn!
	     * recv does not work correctly with Sun's JDK 1.1 (%#@&$).
	     */

        if ((tcpip_blockingReadn (sockfd, &flag, 1, nread) != e_TCPIP_OK) || (nread != 1)) {
	    new_status = MEET_BROKEN;
        } else if (flag == MEETING_DONE) {
	    new_status = MEET_TERMINATED;
        } else if (flag == MEETING_MESSAGE) {
	    messageReady = e_TRUE;
	    handler -> changeFlags (0);
	    new_status = MEET_COMPLETE;
        } else if (flag == MEETING_FILE) {
	    fileReady = e_TRUE;
	    handler -> changeFlags (0);
	    new_status = MEET_COMPLETE;
        } else {
	    new_status = MEET_BROKEN;
        }

        if (new_status != MEET_COMPLETE) {
	    terminateConnection ();
	    meetings -> spliceMeetingFromList (this);
	    status = new_status;
	    meetings -> spliceMeetingIntoList (this);
        }

        break;
    }

	/* reset the sigio flags to 0 */

    lastSigioFlags = 0;
}

/* AgentMeetings::AgentMeetings

   Purpose: This procedure is the constructor for class MEETINGS.

     Input: agent = agent associated with this instance
		    (class AGENT *)
*/

AgentMeetings::AgentMeetings (AGENT *p_agent):
    agent (p_agent),
    meetingCount (0),
    nextHandle (1),
    meetingTable (HANDLE_TABLE_SIZE)
{
    for (int i = 0; i < NUM_MEET_STATES; i++) {
	meetingLists[i] = (MEETING *) NULL;
    }
}

/* AgentMeetings::~AgentMeetings

   Purpose: This procedure is the destructor for class MEETINGS.
*/

AgentMeetings::~AgentMeetings ()
{
    HashNode *node;
    HashSearch search (&meetingTable);

	/* remove the hash table entries */

    while ((node = search.next()) != NULL) {
	MEETING *meeting = (MEETING *) node -> data;
	DELETE_OBJECT(meeting);
    }
} 

/* MEETING::spliceMeetingFromList

   Purpose: Splice a meeting out of the appropriate status list

     Input: meeting = the meeting information
		      (class MEETING *)

    Output: The procedure splices the meeting out of the appropriate
	    status list.
*/
   
void AgentMeetings::spliceMeetingFromList (MEETING *meeting)
{
    int status;

	/* assertion on the parameters */

    assert (meeting != NULL);
    status = meeting -> status;

	/* assertion on the status */

    assert ((status >= 0) && (status < NUM_MEET_STATES));

	/* adjust the pointers */

    if (meeting -> next != NULL) {
	meeting -> next -> previous = meeting -> previous;
    }

    if (meeting -> previous != NULL) {
	meeting -> previous -> next = meeting -> next;
    } else {
	meetingLists[status] = meeting -> next;
    }
}

/* MEETING::spliceMeetingIntoList

   Purpose: Splice a meeting into the appropriate status list

     Input: meeting = the meeting information
		      (class MEETING *)

    Output: The procedure splices the meeting into the appropriate 
	    status list.
*/

void AgentMeetings::spliceMeetingIntoList (MEETING *meeting)
{
    int status;

	/* assertion on the parameters */

    assert (meeting != NULL);
    status = meeting -> status;

	/* assertion on the status */

    assert ((status >= 0) && (status < NUM_MEET_STATES));

	/* splice the meeting into the list */

    meeting -> next     = meetingLists[status];
    meeting -> previous = NULL;

    if (meetingLists[status] != NULL) {
	meetingLists[status] -> previous = meeting;
    }

    meetingLists[status] = meeting;
}

/* AgentMeetings::makeNewMeeting

   Purpose: Make a new meeting

     Input: id = identification of the agent on the other end of the meeting
		 (const AgentId &)

    Output: The procedure returns one of the following error codes.

	    MEET_OK       = success
	    MEET_TOO_MANY = too many meetings

	    On success

	    handle = unique handle for this meeting
		     (UINT_32)
*/

int AgentMeetings::makeNewMeeting (const AgentId &id, UINT_32 &handle)
{
    int there = 1;
    HashNode *node;
    MEETING *meeting;

	/* check the number of meetings */

    if (meetingCount >= (UINT_32) MAX_U32) {
	return MEET_TOO_MANY;
    }

	/* create the new meeting */

    meeting = new MEETING (id, this);

	/* find an unused meeting handle */

    while (there) {

	node = meetingTable.add (nextHandle, (char *) NULL, meeting, there);

	if (!there) {
	    meeting -> localHandle = nextHandle; 
	    handle = nextHandle;
	    meetingCount += 1;
	}

	nextHandle = (nextHandle % MAX_U32) + 1;
    }

	/* new handle callback */

    new_handle (handle);

	/* splice the meeting into the correct list */

    spliceMeetingIntoList (meeting);
    return MEET_OK;
}

/* AgentMeetings::killAllMeetings

   Purpose: Kill all the meetings

     Input: None

    Output: The procedure kills all the meetings.
*/

void AgentMeetings::killAllMeetings (void)
{
    int i;

    for (i = 0; i < NUM_MEET_STATES; i++) {
	while (meetingLists[i] != NULL) {
	    killMeetingInternal (meetingLists[i]);
	}
    }
}

/* AgentMeetings::killMeetingInternal

   Purpose: Kill a meeting (internal version)

     Input: meeting = the meeting
		      (class MEETING *)

    Output: The procedure kills the meeting.
*/

void AgentMeetings::killMeetingInternal (MEETING *meeting)
{
    UINT_8 flag;
    int nwritten;

	/* assertions on the parameters */

    assert (meeting != NULL);

	/* If the meeting is connected, notify the other end, *but* */
	/* never block since we do not want to wait.  If the other  */
	/* end does not get our explicit notification, that's okay. */
	 
    if (meeting -> status == MEET_COMPLETE) {
	flag = MEETING_DONE;
	BlockingMode oldMode;
	(void) tcpip_setBlockingMode (meeting -> sockfd, e_NonBlocking, oldMode);
	(void) tcpip_nonblockingWriten (meeting -> sockfd, (char *) &flag, 1, nwritten);
    }
	
	/* splice the meeting out of its list */

    spliceMeetingFromList (meeting);

	/* remove handle callback */
 
    remove_handle (meeting -> localHandle);

	/* delete the meeting */

    (void) meetingTable.remove (meeting -> localHandle, (char *) NULL);
    DELETE_OBJECT(meeting);

	/* now have one less meeting */

    meetingCount -= 1;
}

/* AgentMeetings::killMeeting

   Purpose: Kill a meeting 

     Input: handle = the meeting handle
		     (UINT_32)

    Output: The procedure returns one of the following error codes.

	    MEET_OK          = success
	    MEET_NONEXISTENT = no meeting with the specified handle 
*/

int AgentMeetings::killMeeting (UINT_32 handle)
{
    HashNode *node;
    MEETING *meeting;

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, (char *) NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* kill the meeting */

    killMeetingInternal (meeting);
    return MEET_OK;
}

/* AgentMeetings::receiveFile

   Purpose: Receive a file from a meeting

     Input: delay     = maximum time to wait for the incoming file
			(struct timeval)

	    handle    = the meeting handle
		        (UINT_32)

	    callbacks = file access callbacks
			(class FileCallbacks &)

    Output: The procedure returns one of the following codes.

	    MEET_OK               = success
	    MEET_NONEXISTENT      = meeting does not exist
	    MEET_CANT_RECEIVE     = meeting is not connected
	    MEET_LOST_CONNECTION  = meeting has broken   
	    MEET_TIMEOUT	  = timeout has expired
	    MEET_NEXT_MESSAGE     = next thing is a message
	    MEET_CANT_WRITE       = unable to write to file

	    On success

	    bytes = total number of bytes written to the file
		    (UINT_32 &)
*/

int AgentMeetings::receiveFile (struct timeval delay, UINT_32 handle, FileCallbacks &callbacks, UINT_32 &bytes)
{
    int rc;
    int newStatus;
    HashNode *node;
    MEETING *meeting;
    MESSAGE *message = NULL;
    UINT_8 meetingStatus = MEET_NO_CHANGE;

	/* get the IO instance */

    AgentIO *io = agent -> get_io ();
    assert (io != NULL);

	/* mutual exclusion -- lock is automatically released */

    Guard guard (agent -> getMonitorLock());

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* make sure that the meeting is connected */

    if (meeting -> status != MEET_COMPLETE) {
	return MEET_CANT_RECEIVE;
    }

	/* do the selection */

    do {

	if ((rc = io -> simpleSelect_noLock (*meeting, meetingStatus, delay)) != AGENT_OK) {

	    if (rc == AGENT_TIMEOUT) {
	 	rc = MEET_TIMEOUT;
	    } else {
		rc = MEET_LOST_BACKGROUND;
	    }

	    return (rc);
        }

    } while (meetingStatus == MEET_COMPLETE);

	/* see what the status is */

    if (meetingStatus == MEET_MESSAGE_READY) {
	return MEET_NEXT_MESSAGE;
    } else if (meetingStatus != MEET_FILE_READY) { 
	return MEET_LOST_CONNECTION;
    }

	/* receive the file */

    if ((message = message_receive (meeting -> sockfd, messagesAcrossMeetings)) == NULL) {

	newStatus = MEET_BROKEN;

    } else if (message -> getFlag() != MEETING_FILE) {

	newStatus = MEET_BROKEN;

    } else {

	char *bp;
	int nwritten;
	int nrequested;
	UINT_32 totalBytes = message -> getLong (0); 

	    /* assume success */

	newStatus = MEET_COMPLETE;
	meeting -> fileReady = e_FALSE;

	    /* create the buffer */

	char *buffer = new char[COPY_BUFFER_SIZE];
	SmartCharPtr bufferPtr (buffer);

	    /* copy the file off the socket */ 

	bytes = 0;

	while ((bytes < totalBytes) && (newStatus == MEET_COMPLETE)) {

	    int nread;
	    int toRead;

		/* read a buffer of data from the socket */
	  
	    nrequested = (int) (totalBytes - bytes);

	    if (nrequested > COPY_BUFFER_SIZE) {
		nrequested = COPY_BUFFER_SIZE;
	    }

	    if ((tcpip_blockingReadn (meeting -> sockfd, buffer, nrequested, nread) != e_TCPIP_OK) 
		|| (nread < nrequested)) {
		newStatus = MEET_BROKEN;
		break;
	    }

		/* write the buffer of data to the file */

	    bp     = buffer;
	    toRead = nrequested;

	    while (1) {

		if (callbacks.write (toRead, bp, nwritten) != Connection::e_Ok) {
		    newStatus = MEET_BROKEN;
		    break;
		}

		if (nwritten < toRead) {
		    toRead -= nwritten;
		    bp += nwritten;
		    continue;
		}

		break;
	    }

	    if (newStatus != MEET_COMPLETE) {
		break;
	    }

		/* increment the byte count */

	    bytes += (UINT_32) nrequested;
	}

	    /* check if there is another message right behind the file */
	    /* and turn back on "read-ready" checking for the meeting  */
	    /* socket                                                  */

#ifdef FIX_NOW
	    /* move the changeFlags call into a MEETING method */
#endif

	if (newStatus == MEET_COMPLETE) {

	    if (meeting -> handler != NULL) {
		meeting -> handler -> changeFlags (SigioTrap::e_READ_READY);
	    }
	}
    }

	/* meeting has broken? */

    if (newStatus != MEET_COMPLETE) {
	meeting -> terminateConnection ();
	spliceMeetingFromList (meeting);
	meeting -> status = newStatus;
	spliceMeetingIntoList (meeting);
	return MEET_LOST_CONNECTION;
    }

	/* cleanup */

    DELETE_OBJECT(message);
    return MEET_OK;
}

/* AgentMeetings::receiveMessage

   Purpose: Receive a message from a meeting

     Input: delay    = maximum time to wait for the incoming message
		       (struct timeval)
	
	    handle   = the meeting handle
		       (UINT_32)

    Output: The procedure returns one of the following codes.

	    MEET_OK               = success
	    MEET_NONEXISTENT      = meeting does not exist
	    MEET_CANT_RECEIVE     = meeting is not connected
	    MEET_LOST_CONNECTION  = meeting has broken   
	    MEET_TIMEOUT	  = timeout has expired
	    MEET_NEXT_FILE        = next thing is a file

	    On success

	    code   = the message code 
		     (UINT_32 &)

	    string = the message string (dynamically allocated)
		     (char *&)
*/

int AgentMeetings::receiveMessage (struct timeval delay, UINT_32 handle, UINT_32 &code, char *&string)
{
    int rc;
    HashNode *node;
    MESSAGE *message;
    MEETING *meeting;
    UINT_8 meetingStatus = MEET_NO_CHANGE;

	/* get the IO instance */

    AgentIO *io = agent -> get_io();
    assert (io != NULL);

	/* assume no message */

    string = NULL;

	/* mutual exclusion -- lock is automatically released */

    Guard guard (agent -> getMonitorLock());

	/* assertions on the parameters */
	/* find the meeting */

    if ((node = meetingTable.lookup (handle, NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* make sure that the meeting is connected */

    if (meeting -> status != MEET_COMPLETE) {
	return MEET_CANT_RECEIVE;
    }

	/* do the selection */

    do {

	if ((rc = io -> simpleSelect_noLock (*meeting, meetingStatus, delay)) != MEET_OK) {

	    if (rc == AGENT_TIMEOUT) {
	 	rc = MEET_TIMEOUT;
	    } else {
		rc = MEET_LOST_BACKGROUND;
	    }

	    return (rc);
        }

    } while (meetingStatus == MEET_COMPLETE);

	/* see what the status is */

    if (meetingStatus == MEET_FILE_READY) {
	return MEET_NEXT_FILE;
    } else if (meetingStatus != MEET_MESSAGE_READY) {
	return MEET_LOST_CONNECTION;
    }

	/* assume success */

    meeting -> messageReady = e_FALSE;

 	/* receive the message */

    if ((message = message_receive (meeting -> sockfd, messagesAcrossMeetings)) == NULL) {
	meeting -> terminateConnection ();
	spliceMeetingFromList (meeting);
	meeting -> status = MEET_BROKEN;
	spliceMeetingIntoList (meeting);
	return MEET_LOST_CONNECTION;
    }

	/* check if there is another message right behind it */ 
	/* and turn back on "read-ready" checking for the    */
	/* meeting socket                                    */

#ifdef FIX_NOW
	/* move the changeFlags call into a MEETING method */
#endif

    if (meeting -> handler != NULL) {
	meeting -> handler -> changeFlags (SigioTrap::e_READ_READY);
    }

	/* done */

    code   = message -> getLong (0); 
    string = message -> getString (1);    // don't delete 
    DELETE_OBJECT(message);
    return MEET_OK; 
}

/* AgentMeetings::sendFile

   Purpose: Send a file along a meeting

     Input: handle    = the meeting handle
		        (UINT_32)

	    callbacks = file access callbacks
			(FileCallbacks &)

	    bytes     = number of bytes to send from file
			(UINT_32)

    Output: The procedure returns one of the following codes.

	    MEET_OK               = success
	    MEET_NONEXISTENT      = meeting does not exist
	    MEET_CANT_SEND        = meeting is not connected
	    MEET_LOST_CONNECTION  = meeting has broken   
	    MEET_CANT_READ        = unable to
*/

int AgentMeetings::sendFile (UINT_32 handle, FileCallbacks &callbacks, UINT_32 bytes)
{
    int nwritten;
    UINT_8 newStatus;
    HashNode *node;
    MEETING *meeting;
    char flag = MEETING_FILE; 

	/* mutual exclusion -- lock is automatically released */

    Guard guard (agent -> getMonitorLock());

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* make sure that the meeting is connected */

    if (meeting -> status != MEET_COMPLETE) {
	return MEET_CANT_SEND;
    }

	/* construct and send the message and then the file */

    MESSAGE message (messagesAcrossMeetings.messages[flag]);
    message.addLong (0, bytes);

    if ((tcpip_blockingWriten (meeting -> sockfd, &flag, 1, nwritten) != e_TCPIP_OK) || (nwritten != 1)) {

	newStatus = MEET_BROKEN;

    } else if (message_send (meeting -> sockfd, message) != MESSAGE_OK) {

	newStatus = MEET_BROKEN;

    } else {

	UINT_32 totalSent = 0;

	    /* assume success */

	newStatus = MEET_COMPLETE;

	    /* create the buffer */

	char *buffer = new char[COPY_BUFFER_SIZE];
	SmartCharPtr bufferPtr (buffer);
	
	    /* copy the file onto the socket */

	while ((totalSent < bytes) && (newStatus == MEET_COMPLETE)) {
	
	    int nread;
	    int nwritten;

		/* read a buffer of data from the file */
	  
	    int nrequested = (int) (bytes - totalSent);

	    if (nrequested > COPY_BUFFER_SIZE) {
		nrequested = COPY_BUFFER_SIZE;
	    }

	    char *bp    = buffer;
	    int toWrite = nrequested;

	    while (1) {

		if (callbacks.read (toWrite, bp, nread) != Connection::e_Ok) {
		    newStatus = MEET_BROKEN;
		    break;
		}

		if (nread < toWrite) {
		    toWrite -= nread;
		    bp += nread;
		    continue;
		}

		break;
	    }

	    if (newStatus != MEET_COMPLETE) {
		break;
	    }

		/* write out the buffer */

	    if ((tcpip_blockingWriten (meeting -> sockfd, buffer, nrequested, nwritten) != e_TCPIP_OK) 
		|| (nwritten < nread)) {
	        newStatus = MEET_BROKEN;
		break;
	    } 

		/* increment the byte count */

	    totalSent += (UINT_32) nrequested;
	}
    }

	/* meeting has broken? */

    if (newStatus != MEET_COMPLETE) {
	meeting -> terminateConnection ();
	spliceMeetingFromList (meeting);
	meeting -> status = newStatus;
	spliceMeetingIntoList (meeting);
	return MEET_LOST_CONNECTION;
    }

    return MEET_OK;
}

/* AgentMeetings::sendMessage

   Purpose: Send a message along a meeting

     Input: handle = the meeting handle
		     (UINT_32)

	    code   = the message code
		     (UINT_32)

	    string = the message string
		     (char *)

    Output: The procedure returns one of the following codes.

	    MEET_OK               = success
	    MEET_NONEXISTENT      = meeting does not exist
	    MEET_CANT_SEND        = meeting is not connected
	    MEET_LOST_CONNECTION  = meeting has broken   
*/

int AgentMeetings::sendMessage (UINT_32 handle, UINT_32 code, char *string)
{
    int nwritten;
    UINT_8 new_status;
    HashNode *node;
    MEETING *meeting;
    char flag = MEETING_MESSAGE; 

	/* assertions on the parameters */

    assert (string != NULL);

	/* mutual exclusion -- lock is automatically released */

    Guard guard (agent -> getMonitorLock());

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* make sure that the meeting is connected */

    if (meeting -> status != MEET_COMPLETE) {
	return MEET_CANT_SEND;
    }

	/* construct and send the message */

    MESSAGE message (messagesAcrossMeetings.messages[flag]);
    message.addLong (0, code);
    message.addString (1, string);

    if ((tcpip_blockingWriten (meeting -> sockfd, &flag, 1, nwritten) != e_TCPIP_OK) || (nwritten != 1)) {
	new_status = MEET_BROKEN;
    } else if (message_send (meeting -> sockfd, message) != MESSAGE_OK) {
	new_status = MEET_BROKEN;
    } else {
	new_status = MEET_COMPLETE;
    }

	/* meeting has failed? */

    if (new_status != MEET_COMPLETE) {
	meeting -> terminateConnection ();
	spliceMeetingFromList (meeting);
	meeting -> status = new_status;
	spliceMeetingIntoList (meeting);
	return MEET_LOST_CONNECTION;
    }

    return MEET_OK;
}

/* AgentMeetings::failMeeting

   Purpose: Fail a meeting

     Input: handle = the meeting handle
		     (UITN_32)

    Output: The procedure returns one of the following error codes.

	    MEET_OK          = success
	    MEET_NONEXISTENT = no meeting with the specified handle 
*/

int AgentMeetings::failMeeting (UINT_32 handle)
{
    HashNode *node;
    MEETING *meeting;

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, (char *) NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* turn off the background handler */

    meeting -> terminateConnection ();

	/* move the meeting into the failed list */

    spliceMeetingFromList (meeting);
    meeting -> status = MEET_FAILED;
    spliceMeetingIntoList (meeting);
    return (MEET_OK);
}

/* AgentMeetings::acceptMatchingMeeting

   Purpose: If an agent A has requested a meeting with us and we are now
	    requesting a meeting with A, then just accept the requested
	    meeting rather than sending a new request.

     Input: id = id of the agent with whom we are requesting a meeting
		 (const AgentId &)

    Output: The procedure returns one of the following error codes.

	    MEET_OK	      = success
            MEET_NONEXISTENT  = there is no matching meeting
	    MEET_CANT_ACCEPT  = meeting is not in the "requested" state
	    MEET_SYSTEM_ERROR = couldn't create necessary sockets

	    On success

	    port         = selected TCP/IP port 
		           (UINT_32 &)

	    handle       = local handle of the meeting
			   (UINT_32 &)

	    remoteHandle = remote handle for this meeting
			   (UINT_32 &) 
*/

int AgentMeetings::acceptMeetingInternal (MEETING *mptr, UINT_32 &port)
{
    int rc;
    int sockfd;
    int chosenPort;

	/* first setup the socket for acceptance */

#ifndef MEETING_PERFORMANCE_TEST

    if (tcpip_setup (TCPIP_ANY_PORT, sockfd, chosenPort) != e_TCPIP_OK) {

	rc = MEET_SYSTEM_ERROR;

#else

    char temp[24];
    int scratch;
    static int nextChosenPort = 0;

    chosenPort = nextChosenPort++;

    if (chosenPort == TCPIP_ANY_PORT) {
	chosenPort = nextChosenPort++;
    }

    strcpy (temp, "/tmp/testsocket");

#ifdef UNIQUE_UNIX_PORTS

    scratch = chosenPort;
 
    for (int i = 0; i < 5; ++i) {
	temp[i+15] = (scratch % 10) + '0';
	scratch = scratch / 10;
    }

    temp[20] = '\0';

#endif

    if (unix_setup (temp, sockfd) != e_TCPIP_OK) {

	rc = MEET_SYSTEM_ERROR;

#endif

    } else {

	    /* everything is okay */

	rc = MEET_OK;
	port = (UINT_32) chosenPort;

	    /* move the meeting into the appropriate list */

	spliceMeetingFromList (mptr);
	mptr -> status = MEET_ACCEPTING;
	spliceMeetingIntoList (mptr);

	    /* turn on the background handler */

	mptr -> startConnection (sockfd, SigioTrap::e_READ_READY);
    }

    return (rc);
}

int AgentMeetings::acceptMatchingMeeting (const AgentId &id, UINT_32 &handle, UINT_32 &remoteHandle, UINT_32 &port)
{
    MEETING *mptr;
    int rc = MEET_NONEXISTENT;

	/* lock for mutual exclusion */

    Guard guard (agent -> getMonitorLock());

	/* find a matching meeting */

    for (mptr = meetingLists[MEET_REQUESTED]; mptr != NULL; mptr = mptr -> next) {

	    /* need the same IP addresses */

	if (mptr -> id.getServerIp() != id.getServerIp()) {
	    continue;
	}

	    /* if names are specified, we need the same names */
	    /* otherwise we need the same id's                */

	if ((mptr -> id.getName().length() != 0) && (id.getName().length() != 0)) {

	    if (mptr -> id.getName() != id.getName()) {
		continue;
	    }
	 
	} else {

	    if (mptr -> id.getId() != id.getId()) {
		continue;
	    }
	}

	    /* meetings match -- accept the meeting */

	if ((rc = acceptMeetingInternal (mptr, port)) == MEET_OK) {
	    handle = mptr -> localHandle;
	    remoteHandle = mptr -> remoteHandle;
	}
	
	break;
    }

	/* done */

    return (rc); 
}

/* AgentMeetings::acceptMeeting

   Purpose: Accept a meeting

     Input: handle = the meeting handle
		     (UINT_32)

    Output: The procedure returns one of the following error codes.

	    MEET_OK	      = success
            MEET_NONEXISTENT  = meeting does not exist
	    MEET_CANT_ACCEPT  = meeting is not in the "requested" state
	    MEET_SYSTEM_ERROR = couldn't create necessary sockets

	    On success

	    id           = id of the agent on the other end
		           (struct AgentId &)

	    port         = selected TCP/IP port 
		           (UINT_32 &)

	    remoteHandle = remote handle for this meeting
			   (UINT_32 &) 
*/

int AgentMeetings::acceptMeeting (UINT_32 handle, UINT_32 &remoteHandle, AgentId &id, UINT_32 &port)
{ 
    int rc;
    HashNode *node;
    MEETING *meeting;

	/* lock for mutual exclusion */

    Guard guard (agent -> getMonitorLock());

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, (char *) NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* make sure that the meeting is in the proper state */

    if (meeting -> status != MEET_REQUESTED) {
	return MEET_CANT_ACCEPT;
    }

	/* do the actual acceptance and pass stuff back */

    if ((rc = acceptMeetingInternal (meeting, port)) == MEET_OK) {
	id = meeting -> id;
	remoteHandle = meeting -> remoteHandle;
    }

    return (rc);
}

/* AgentMeetings::rejectMeeting

   Purpose: Reject a meeting

     Input: handle = the meeting handle
		     (UINT_32)

    Output: The procedure returns one of the following error codes.

	    MEET_OK	     = success
            MEET_NONEXISTENT = meeting does not exist
	    MEET_CANT_REJECT = meeting is not in the "requested" state

	    On success

	    id           = id of the agent on the other end
		           (struct AgentId &)

	    remoteHandle = remote handle for this meeting
			   (UINT_32 &) 
*/

int AgentMeetings::rejectMeeting (UINT_32 handle, UINT_32 &remoteHandle, AgentId &id)
{
    HashNode *node;
    MEETING *meeting;

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, (char *) NULL)) == NULL) {
	return MEET_NONEXISTENT;
    }

    meeting = (MEETING *) node -> data;

	/* make sure that the meeting is in the proper state */

    if (meeting -> status != MEET_REQUESTED) {
	return MEET_CANT_REJECT;
    }

	/* move the meeting into the appropriate queue location */

    spliceMeetingFromList (meeting);
    meeting -> status = MEET_REJECTED;
    spliceMeetingIntoList (meeting);

	/* pass stuff back */

    id           = meeting -> id;
    remoteHandle = meeting -> remoteHandle;
    return (MEET_OK);
}

/* MEEETINGS::handleIncomingMeeting

   Purpose: Handle an incoming meeting request, acceptance or rejection 

     Input: meeting = the incoming meeting request
		      (struct INCOMING_MEETING *)

    Output: The procedure handles the incoming meeting request.
*/

void AgentMeetings::handleIncomingMeeting (INCOMING_MEETING *meeting)
{
	/* assertions on the parameters */

    assert (meeting != NULL);

	/* lock -- automatically released */

    Guard guard (agent -> getMonitorLock());

	/* take the appropriate action */

    switch (meeting -> status) {

	case ACT_REQUEST:

	    handleIncomingRequest (meeting);
	    break;

	case ACT_ACCEPT:

	    handleIncomingAcceptance (meeting);
	    break;

	case ACT_REJECT:

	    handleIncomingRejection (meeting);
	    break;

	default:

	    /* just ignore the request -- this represents the philosophy */
	    /* of doing no harm but not attempting to recover (at this   */
	    /* level) from every possible protocol error                 */ 

	    DELETE_OBJECT(meeting);
    }
}

/* AgentMeetings::handleIncomingAcceptance

   Purpose: Handle an incoming meeting acceptance

     Input: meeting = the incoming meeting acceptance
		      (struct INCOMING_MEETING *)

    Output: The procedure handles the incoming meeting acceptance.
*/

void AgentMeetings::handleIncomingAcceptance (INCOMING_MEETING *meeting)
{
    int sockfd;
    HashNode *node;
    MEETING *localMeeting;
    int finalStatus;

	/* assertions on the parameters */

    assert (meeting != NULL);
    assert (meeting -> status == ACT_ACCEPT);

	/* we should have both a local and  remote handle */

    if ((meeting -> remoteHandle == UNDEF_MEETING_HANDLE) ||
	(meeting -> localHandle  == UNDEF_MEETING_HANDLE)) {

	/* here we just ignore the request -- this is consistent with  */
	/* a philosophy of doing no harm but not attempting to recover */
	/* (at this level) from every possible protocol error          */

	DELETE_OBJECT(meeting);
	return;
    }

	/* make sure we have a valid IP address and port -- again */
	/* we just ignore the request if we do not                */

    if ((meeting -> port -> ip == UNKNOWN_IP) ||
	(meeting -> port -> port == TCPIP_ANY_PORT)) {
	DELETE_OBJECT(meeting);
	return;
    }

	/* find the meeting -- again we just ignore the request if  */
	/* there is no meeting with the given local handle *or*     */
	/* the meeting is not in the expected state MEET_REQUESTING */

    if ((node = meetingTable.lookup (meeting -> localHandle, NULL)) == NULL) {
	DELETE_OBJECT(meeting);
	return;
    }

    localMeeting = (MEETING *) node -> data;

    if (localMeeting -> status != MEET_REQUESTING) {
	DELETE_OBJECT(meeting);
	return;
    }

	/* make sure that the sender of the acceptance is who we expect --       */
	/* if this check succeeds, due to additional checks on the sender id     */ 
	/* performed by the agent server, we know ... (1) if the meeting         */
	/* was requested with an agent on the SAME machine, it                   */
	/* has been accepted by that specific agent, and (2) if the meeting      */
	/* was requested with an agent on a DIFFERENT machine, it                */
	/* has at least been accepted by some nonlocal agent.   We do not yet    */
	/* perform any other checks at this level.  However, once the meeting    */
	/* moves into the accepting state, the agent itself can perform any      */
	/* desired checks on the security information associated with the        */
	/* acceptor.                                                             */

#ifdef FIX_LATER
	/* Along with adding session keys to inter-machine meetings, we need     */
	/* to add nonces to both the request and acceptance (with the acceptance */
	/* nonce calculated in some way from the request nonce), so that we can  */
	/* tell that the acceptance is coming from the same agent that received  */
        /* the request.  Right now meetings that cross machine boundaries are    */
	/* simply not secure.                                                    */
#endif

    if ((localMeeting -> id.getServerIp() != meeting -> id -> getServerIp()) ||
	((localMeeting -> id.getId() != meeting -> id -> getId()) &&
	 (localMeeting -> id.getName() != meeting -> id -> getName()))) {
	DELETE_OBJECT(meeting);
	return;
    }

	/* now here we go -- first fill in the newly available remote port */
	/* and security information                                        */

    localMeeting -> port     = *(meeting -> port);
    localMeeting -> security = *(meeting -> security);
    DELETE_OBJECT(meeting);

	/* change the meeting status to MEET_CONNECTING */

    spliceMeetingFromList (localMeeting);
    localMeeting -> status = MEET_CONNECTING;
    spliceMeetingIntoList (localMeeting);

	/* now start the connection process */

#ifndef MEETING_PERFORMANCE_TEST 

    if (tcpip_socket(sockfd) != e_TCPIP_OK) {

	finalStatus = MEET_FAILED;

    } else {

	TcpipErrors tcpipRc;

	if ((tcpipRc = tcpip_async_connect (sockfd, localMeeting -> port)) == e_TCPIP_OK) {
	    finalStatus = MEET_COMPLETE;
	} else if (tcpipRc != e_TCPIP_PENDING) {
	    finalStatus = MEET_FAILED;
	} else {
	    finalStatus = MEET_CONNECTING;
	}
    }

#else

    if (unix_socket(sockfd) != e_TCPIP_OK) {

	finalStatus = MEET_FAILED;

    } else {

	int scratch;
	char temp[24];
	TcpipErrors tcpipRc;

	strcpy (temp, "/tmp/testsocket");

#ifdef UNIQUE_UNIX_PORTS

	scratch = localMeeting -> port.port;
 
	for (int i = 0; i < 5; ++i) {
	    temp[i+15] = (scratch % 10) + '0';
	    scratch = scratch / 10;
	}

	temp[20] = '\0';

#endif

	if ((tcpipRc = unix_async_connect (sockfd, temp)) == e_TCPIP_OK) {
	    finalStatus = MEET_COMPLETE;
	} else if (tcpipRc != e_TCPIP_PENDING) {
	    finalStatus = MEET_FAILED;
	} else {
	    finalStatus = MEET_CONNECTING;
	}
    }

#endif

	/* switch the meeting to MEET_FAILED or MEET_COMPLETE if necessary */

    if (finalStatus != MEET_CONNECTING) {
	spliceMeetingFromList (localMeeting);
	localMeeting -> status = finalStatus;
	spliceMeetingIntoList (localMeeting);
    }

	/*  turn on the background handler if needed */

    if (finalStatus == MEET_CONNECTING) {
	localMeeting -> startConnection (sockfd, SigioTrap::e_READ_READY | SigioTrap::e_WRITE_READY);
    } else if (finalStatus == MEET_COMPLETE) {
	localMeeting -> startConnection (sockfd, SigioTrap::e_READ_READY);
    }
}

/* AgentMeetings::handleIncomingRejection

   Purpose: Handle an incoming meeting rejection 

     Input: meeting = the incoming meeting rejection 
		      (struct INCOMING_MEETING *)

    Output: The procedure handles the incoming meeting rejection.
*/

void AgentMeetings::handleIncomingRejection (INCOMING_MEETING *meeting)
{
    HashNode *node;
    MEETING *localMeeting;

	/* assertions on the parameters */

    assert (meeting != NULL);
    assert (meeting -> status == ACT_REJECT); 

	/* we should have both a local and  remote handle */

    if ((meeting -> remoteHandle == UNDEF_MEETING_HANDLE) ||
	(meeting -> localHandle  == UNDEF_MEETING_HANDLE)) {

	/* here we just ignore the request -- this is consistent with  */
	/* a philosophy of doing no harm but not attempting to recover */
	/* (at this level) from every possible protocol error          */

	DELETE_OBJECT(meeting);
	return;
    }

	/* find the meeting -- again we just ignore the request if  */
	/* there is no meeting with the given local handle *or*     */
	/* the meeting is not in the expected state MEET_REQUESTING */

    if ((node = meetingTable.lookup (meeting -> localHandle, NULL)) == NULL) {
	DELETE_OBJECT(meeting);
	return;
    }

    localMeeting = (MEETING *) node -> data;

    if (localMeeting -> status != MEET_REQUESTING) {
	DELETE_OBJECT(meeting);
	return;
    }

	/* make sure that the sender of the rejection is who we expect --        */
	/* if this check succeeds, due to additional checks on the sender id     */ 
	/* performed by the agent server, we know ... (1) if the meeting         */
	/* was requested with an agent on the SAME machine, it has been          */
	/* rejected by that specific agent, and (2) if the meeting               */
	/* was requested with an agent on a DIFFERENT machine, it has            */
	/* at least been accepted by some nonlocal agent.   We do not yet        */
	/* perform any other checks at this level.  However, once the meeting    */
	/* moves into the rejected state, the agent itself can perform any       */
	/* desired checks on the security information associated with the        */
	/* rejector.                                                             */

#ifdef FIX_LATER
	/* Along with adding session keys to inter-machine meetings, we need     */
	/* to add nonces to both the request and rejection (with the rejection   */
	/* nonce calculated in some way from the request nonce), so that we can  */
	/* tell that the rejection is coming from the same agent that received   */
        /* the request.  Right now meetings that cross machine boundaries are    */
	/* simply not secure.                                                    */
#endif

    if ((localMeeting -> id.getServerIp() != meeting -> id -> getServerIp()) ||
	((localMeeting -> id.getId() != meeting -> id -> getId()) &&
	 (localMeeting -> id.getName() != meeting -> id -> getName()))) {
	DELETE_OBJECT(meeting);
	return;
    }

	/* now fill in the newly available security information and */
	/* switch the meeting status to MEET_REJECTED               */

    localMeeting -> security = *(meeting -> security);
    spliceMeetingFromList (localMeeting);
    localMeeting -> status = MEET_REJECTED;
    spliceMeetingIntoList (localMeeting);
    DELETE_OBJECT(meeting);
}

/* AgentMeetings::handleIncomingRequest

   Purpose: Handle an incoming meeting request

     Input: meeting = the incoming meeting request
		      (struct INCOMING_MEETING *)

    Output: The procedure handles the incoming meeting request.
*/

void AgentMeetings::handleIncomingRequest (INCOMING_MEETING *meeting)
{
    int there;
    MEETING *mptr;
    HashNode *node;

	/* assertions on the parameters */

    assert (meeting != NULL);
    assert (meeting -> status == ACT_REQUEST); 

	/* we might have too many meetings */

    if (meetingCount >= (UINT_32) MAX_32) {

	/* here we just ignore the request -- this is consistent with  */
	/* a philosophy of doing no harm but not attempting to recover */
	/* (at this level) from every possible error -- this will      */
	/* essentially never happen anyways                            */

	DELETE_OBJECT(meeting);
	return;
    }

	/* we should have a remote handle */

    if (meeting -> remoteHandle == UNDEF_MEETING_HANDLE) {

	/* here we just ignore the request -- this is consistent with  */
	/* a philosophy of doing no harm but not attempting to recover */
	/* (at this level) from every possible protocol error          */

	DELETE_OBJECT(meeting);
	return;
    }

	/* we should not have a local handle if this is truly a request */

    if (meeting -> localHandle != UNDEF_MEETING_HANDLE) {

	/* here we just ignore the request -- this is consistent with  */
	/* a philosophy of doing no harm but not attempting to recover */
	/* (at this level) from every possible protocol error          */

	DELETE_OBJECT(meeting);
	return;
    }

	/*
	 * If this agent has sent a meeting request to the agent that is now
	 * requesting its own meeting in turn, we just link up the two 
	 * requests.
	 *
	 * I. If we have the smaller IP or equal IP's and a smaller numeric id,
	 *    we update the recipient and security information and move the 
         *    meeting to "requested".
	 *
	 * II. Otherwise do nothing.  The other agent will be sending an
	 *     ACT_ACCEPT if it accepts the meeting.
	 */
     
#ifdef FIX_LATER
	/*
	 * As we acceptance and rejection, we need inter-machine security
	 * when matching an incoming request against a request that the agent
	 * previously made itself.  As before, however, remember that the agent
	 * can perform any desired checks against the other agent's security
	 * vector once the meeting goes into the accepting, rejected or
	 * complete state.
	 */
#endif

    for (mptr = meetingLists[MEET_REQUESTING]; mptr != NULL; mptr = mptr -> next) {

	    /* need the same IP addresses */

	if (mptr -> id.getServerIp() != meeting -> id -> getServerIp()) {
	    continue;
	}

	    /* if names are specified, we need the same names */
	    /* otherwise we need the same id's                */

	if ((mptr -> id.getName().length() != 0) && (meeting -> id -> getName().length() != 0)) {

	    if (mptr -> id.getName() != meeting -> id -> getName()) {
		continue;
	    }
	 
	} else {

	    if (mptr -> id.getId() != meeting -> id -> getId()) {
		continue;
	    }
	}

	    /* the meetings match so decide whether we need to do anything */

	LOCATION *location = agent -> get_location ();
	assert (location != NULL);
	const AgentId &localId = location -> getLocal ();

	    /* must convert the IP addresses to host order before comparing -- */
	    /* otherwise bad things can happen if the meeting is between one   */
	    /* machine where host- and network-order and the same and a second */
	    /* machine where host- and network-order are not the same          */

	UINT_32 localHip  = ntohl (localId.getServerIp());
	UINT_32 remoteHip = ntohl (meeting -> id -> getServerIp());

	if ((localHip < remoteHip) ||
	    ((localHip == remoteHip) &&
	     (localId.getId() < meeting -> id -> getId()))) {
	    mptr -> id           = *(meeting -> id);
	    mptr -> security     = *(meeting -> security);
	    mptr -> remoteHandle = meeting -> remoteHandle;
	    spliceMeetingFromList (mptr);
	    mptr -> status       = MEET_REQUESTED;
	    spliceMeetingIntoList (mptr);
	}

	DELETE_OBJECT(meeting);
	return;
    }

	/* get the meeting into the appropriate structure */

    mptr = new MEETING (*meeting, this);

	/* generate a handle */

    there = 1;

    while (there) {

	node = meetingTable.add (nextHandle, (char *) NULL, mptr, there);

	if (!there) {
	    mptr -> localHandle = nextHandle; 
	    meetingCount += 1;
	}

	nextHandle = (nextHandle % MAX_U32) + 1;
    }

	/* splice the meeting into the appropriate status list */

    spliceMeetingIntoList (mptr);

	/* new handle callback */

    new_handle (mptr -> localHandle);

	/* now update the INCOMING_MEETING structure and hand it back */
	/* to MASKS for inclusion into the appropriate queue          */
   
    meeting -> status      = mptr -> status;
    meeting -> localHandle = mptr -> localHandle;
 
    AgentMasks *masks = agent -> get_masks();
    assert (masks != NULL);
    masks -> addMeetingToQueue (meeting);
}

/* AgentMeetings::lookupMeeting

   Purpose: Get information about a meeting

     Input: handle = the meeting handle
		     (UINT_32)

    Output: The procedure returns NULL if there is no meeting with the
	    specified handle.  Otherwise the procedure returns a pointer to
	    a dynamically allocated MEETING_INFO structure that contains
	    the meeting information.
*/

MEETING_INFO *AgentMeetings::lookupMeeting (UINT_32 handle)
{
    HashNode *node;
    MEETING *meeting;

	/* mutual exclusion */

    Guard guard (agent -> getMonitorLock());

	/* find the meeting */

    if ((node = meetingTable.lookup (handle, (char *) NULL)) == NULL) {
	return ((MEETING_INFO *) NULL); 
    }

    meeting = (MEETING *) node -> data;
    return (new MEETING_INFO (*meeting));
}

MEETING *AgentMeetings::lookup_noLock (UINT_32 handle)
{
    HashNode *node;
    
	/* find the meeting */

    if ((node = meetingTable.lookup (handle, (char *) NULL)) == NULL) {
	return ((MEETING *) NULL); 
    }

    return ((MEETING *) node -> data);
}
