/* Agent Tcl
   Bob Gray
   4 April 1996

   genMeeting.h

   This file defines the data structures that store information about
   meetings.

   Copyright (c) 1995-1998, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _GEN_MEETING_H
#define _GEN_MEETING_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _AGENT_AGENT_ID_H
#include "agentAgentId.h"	// AgentId
#endif
#ifndef _AGENT_SECURE_H
#include "agentSecure.h"	// BASE_SECURITY
#endif
#ifndef _MESG_PORT_H
#include "mesgPort.h"		// RemotePort
#endif
#ifndef _SUPP_HASH_H
#include "suppHash.h"		// Hash
#endif
#ifndef _PLAT_EXCLUSION_H
#include "platExclusion.h"	// SigioTrap
#endif
#ifndef _PLAT_SIGIO_H
#include "platSigio.h"		// SigioTrap
#endif
#ifndef _PLAT_TIMERS_H
#include "platTimers.h"		// TimerTrap
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"		// UINT_32
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"		// BOOLEAN
#endif

    /* forward declarations */

class FileCallbacks;
class INCOMING_MEETING;
class RESTRICT;

    /* meeting status */

const int MEET_REQUESTING		= 0;
const int MEET_ACCEPTING		= 1;
const int MEET_CONNECTING		= 2;
const int MEET_REQUESTED		= 3;
const int MEET_COMPLETE			= 4;
const int MEET_REJECTED			= 5;
const int MEET_FAILED			= 6;
const int MEET_TERMINATED		= 7;
const int MEET_BROKEN		        = 8; 
const int MEET_REJECTING		= 9;
const int NUM_MEET_STATES		= 10;

    /* additional status codes for the select operation */

const int MEET_DOES_NOT_EXIST		= 0x10;
const int MEET_MESSAGE_READY		= 0x11;
const int MEET_FILE_READY		= 0x12;
const int MEET_NO_CHANGE		= 0x13;

    /* meeting codes */

const int MEET_OK			= 0;
const int MEET_TOO_MANY			= 1;
const int MEET_NONEXISTENT		= 2;
const int MEET_CANT_ACCEPT		= 3; 
const int MEET_SYSTEM_ERROR		= 4;
const int MEET_CANT_REJECT		= 5;
const int MEET_TIMEOUT			= 6;
const int MEET_LOST_BACKGROUND		= 7;
const int MEET_CANT_SEND		= 8;
const int MEET_LOST_CONNECTION		= 9;
const int MEET_CANT_RECEIVE		= 10;
const int MEET_NEXT_FILE		= 11;
const int MEET_CANT_READ		= 12;
const int MEET_NEXT_MESSAGE		= 13;
const int MEET_CANT_WRITE		= 14;

    /* meeting actions */

const UINT_8 ACT_REQUEST		= 0;
const UINT_8 ACT_ACCEPT			= 1;
const UINT_8 ACT_REJECT			= 2;

    /* an undefined handle */

const UINT_32 UNDEF_MEETING_HANDLE	= 0x0;

    /* meeting information */

struct MEETING_INFO
{
    int status;                /* status of the meeting                    */
    int lastStatus;	       /* last reported status                     */
    BOOLEAN messageReady;      /* message is waiting to be read            */
    BOOLEAN fileReady;         /* file is waiting to be read               */
    AgentId id;                /* agent on the other end                   */
    BASE_SECURITY security;    /* security information for that agent      */
    RemotePort port;          /* TCP/IP port associated with this meeting */
    UINT_32 localHandle;       /* this agent's handle for the meeting      */
    UINT_32 remoteHandle;      /* remote agent's handle for the meeting    */

	/* constructors */

    MEETING_INFO (const AgentId &p_id);
    MEETING_INFO (const INCOMING_MEETING &meeting);
    MEETING_INFO (const MEETING_INFO& meeting);
};

    /* a meeting */

class MEETING: public MEETING_INFO, public SigioTrap
{
    friend class AgentMeetings;
    friend class AgentMeetingsIterator;

private:

	/* 
         * linked list of meetings
	 */

    MEETING *previous;    // previous meeting with the *same* status
    MEETING *next;        // next meeting with the *same* status  

	/* 
	 * socket, SIGIO handler and last set of flags that the SIGIO 
	 * handler reported via the sigioAlarm method
	 */

    int sockfd;
    int lastSigioFlags;
    SigioHandler *handler;

	/* 
	 * back pointer into the meetings structure 
	 */

    AgentMeetings *meetings;

	/* 
	 * socket is ready for reading or writing
	 */

    void sigioAlarm (int fd, int flags);

	/* 
	 * constructor and destructor
 	 */

    MEETING (const AgentId &id, AgentMeetings *meetings);
    MEETING (const INCOMING_MEETING &meeting, AgentMeetings *meetings);
    virtual ~MEETING ();

	/*
	 * associate a socket with the meeting  
	 */

    void startConnection (int sockfd, int flags);
    void turnBackgroundOn (int flags);
    void resetBackground (int flags);
    void turnBackgroundOff (void);
    void terminateConnection (void);

	/*
	 * detect and handle a change in meeting status  
	 */

    void handleStatusChange (void);

public:

	/*
	 * get the socket file descriptor
	 */

    int getSockfd (void) const {
	return (sockfd);
    }
};

    /* all meetings */

class AgentMeetings
{
    friend class AGENT;
    friend class AgentMasks;
    friend class MEETING;
    friend class AgentMeetingsIterator;

private:

	/* agent associated with this AentMeetings instance  */

    AGENT *agent;
   
	/* hash table of meetings and the next meeting handle */

    UINT_32 meetingCount;
    UINT_32 nextHandle;
    Hash meetingTable;

	/* meetings indexed by their current status */

    MEETING *meetingLists[NUM_MEET_STATES];

	/* splice a meeting into or out of the appropriate list */

    void spliceMeetingIntoList (MEETING *meeting);
    void spliceMeetingFromList (MEETING *meeting);

	/* handle incoming request, acceptances and rejections */

    void handleIncomingRequest (INCOMING_MEETING *meeting);
    void handleIncomingAcceptance (INCOMING_MEETING *meeting);
    void handleIncomingRejection (INCOMING_MEETING *meeting);

	/* internal versions (do the actual work) */ 

    void killMeetingInternal (MEETING *meeting);
    int acceptMeetingInternal (MEETING *meeting, UINT_32 &port);

	/* make a new meeting (called from AGENT) */

    int makeNewMeeting (const AgentId &id, UINT_32 &handle);

	/* accept and reject meetings (called from AGENT)  */

    int acceptMeeting 
	(UINT_32 handle, UINT_32 &remoteHandle, AgentId &id, UINT_32 &port);

    int rejectMeeting 
	(UINT_32 handle, UINT_32 &remoteHandle, AgentId &id);

    int acceptMatchingMeeting
	(const AgentId &id, UINT_32 &handle, UINT_32 &remoteHandle, 
	 UINT_32 &port);

	/* send and receive messages and files (called from AGENT) */

    int sendMessage 
	(UINT_32 handle, UINT_32 code, char *string);

    int receiveMessage 
	(struct timeval delay, UINT_32 handle, UINT_32 &code, char *&string);

    int sendFile
	(UINT_32 handle, FileCallbacks &callbacks, UINT_32 bytes);

    int receiveFile
	(struct timeval delay, UINT_32 handle, FileCallbacks &callbacks, UINT_32 &bytes);

	/* kill a meeting (called from AGENT) */

    int killMeeting (UINT_32 handle);

	/* kill all the meetings (called from AGENT) */

    void killAllMeetings (void);

	/* fail a meeting (called from AGENT) */

    int failMeeting (UINT_32 handle);

	/* handle an incoming meeting request (called from Masks) */

    void handleIncomingMeeting (INCOMING_MEETING *meeting);

protected:

	/* callbacks for handle creation and deletion */

    virtual void new_handle (UINT_32 handle) = 0;
    virtual void remove_handle (UINT_32 handle) = 0;

	/* constructor and destructor */

    AgentMeetings (AGENT *agent);
    virtual ~AgentMeetings ();

public:

	/* 
	 * get the number of meetings
	 */

    UINT_32 getCount (void) const {
	return (meetingCount);
    };

	/* 
	 * lookup a meeting
	 */

    MEETING *lookup_noLock (UINT_32 handle);

    MEETING_INFO * lookupMeeting (UINT_32 handle);
};

    // iterator that iterates through all existing meetings

class AgentMeetingsIterator
{
    int m_which;
    MEETING *m_nextMeeting;
    AgentMeetings &m_meetings;

public:

    AgentMeetingsIterator (AgentMeetings &meetings, int which):
	m_which (which),
	m_nextMeeting (NULL),
	m_meetings (meetings)
    {
	assert ((which >= 0) && (which < NUM_MEET_STATES));
	m_nextMeeting = m_meetings.meetingLists[which];
    }

    void reset (int which)
    {
	assert ((which >= 0) && (which < NUM_MEET_STATES));
	m_which = which;
	m_nextMeeting = m_meetings.meetingLists[which];
    }

    MEETING *next (void)
    {
	MEETING *meeting = m_nextMeeting;

	if (meeting != NULL) {
	    m_nextMeeting = meeting -> next;
	}

	return (meeting);
    }
};

#endif   /* GEN_MEETING */
