/* Agent Tcl
   Bob Gray
   27 July 1998

   genMesgDriver.cc

   This file implements the MessageDriver class, which provides routines for
   sending and receiving messages across a connection.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include "platPorting.h"	// platform-specific definitions
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "platSizes.h"		// UINT_32
#include "suppDBuffer.h"	// DynamicBuffer
#include "mesgConnection.h"	// Connection
#include "genEncrypt.h"		// secureForServer
#include "genMesgDriver.h"	// MessageDriver
#include "genMessage.h"		// MESSAGE
#include "genTransmit.h"	// will go away

/* MessageDriver::connectionCodeToMessageCode

   Purpose: Convert a connection error code into a message error code
*/

int MessageDriver::connectionCodeToMessageCode (Connection::ConnectionErrors connectionRc)
{
    int messageRc;

    switch (connectionRc) {

	case Connection::e_Ok:

	    messageRc = MESSAGE_OK;
	    break;

	case Connection::e_LostConnection:

	    messageRc = MESSAGE_LOST_CONNECTION;
	    break;

	case Connection::e_SystemError:

	    messageRc = MESSAGE_SYS_ERROR;
	    break;

	case Connection::e_Timeout:

	    messageRc = MESSAGE_TIMEOUT;
	    break;

	case Connection::e_NotFound:

	    messageRc = MESSAGE_NOT_FOUND;
	    break;

	case Connection::e_NoServer:

	    messageRc = MESSAGE_NO_SERVER;
	    break;

	case Connection::e_Pending:

	    messageRc = MESSAGE_PENDING;
	    break;

	case Connection::e_CantRead:

	    messageRc = MESSAGE_CANT_READ;
	    break;

	case Connection::e_CantWrite:

	    messageRc = MESSAGE_CANT_WRITE;
	    break;

	default:

	    abort_with_message ("unknown Connection error");
    }

    return (messageRc);
}

/* MessageDriver::unsecureReceive

   Purpose: Receive an unencrypted, unsigned message from a connection
 
     Input: connection = the connection
                         (Connection &)

	    messageSet = set of messages that should contain the incoming
                         message
                         (const MESSAGE_SET &)

	    stop       = timeout or NULL if there is no timeout
			 (struct timeval *)

    Output: The procedure returns MESSAGE_LOST_CONNECTION if the connection
            broke before the message could be received, MESSAGE_TIMEOUT if
	    the timeout expired before the message could be received, 
	    MESSAGE_INVALID if the incoming message is invalid (e.g., a
	    negative size), MESSAGE_SYS_ERROR on system error (e.g., unable
	    to allocate enough memory for the incoming message), and MESSAGE_OK
	    on success.  On success, the "receivedMessage" parameter 
	    (MESSAGE *&) is set to a dynamically allocated MESSAGE instance 
	    that contains the received message.
*/

int MessageDriver::unsecureReceive (MESSAGE *&receivedMessage, Connection &connection, const MESSAGE_SET &messageSet, struct timeval *stop)
{
    int rc;
    int nread;
    INT_32 size = 0;
    char *buffer; 
    UINT_32 netSize;
    INT_32 fullSize; 
    Connection::ConnectionErrors connectionRc;

	// set receivedMessage to NULL in case we fail to receive the message
  
    receivedMessage = NULL;

	// get the size of the message -- keep looping if we get a size of 0,
	// since we use "empty" messages as application-level acknowledgements
	// that can be ignored on the client side

    do {

	if ((connectionRc = connection.read (sizeof(INT_32), (char *) &netSize, nread, stop)) == Connection::e_Timeout) {

	    rc = MESSAGE_TIMEOUT;

	} else if (connectionRc != Connection::e_Ok) {

	    rc = MESSAGE_LOST_CONNECTION;

	} else {

	    size = ntohl (netSize);

	    if (size < 0) {
	        rc = MESSAGE_INVALID;
	    } else {
	        rc = MESSAGE_OK;
	    }
	}

    } while ((rc == MESSAGE_OK) && (size == 0));

	// bail if we did not get the size

    if (rc != MESSAGE_OK) {
	return (rc);
    }

	// add the size field to the message length

    fullSize = size + sizeof(UINT_32);

	// allocate an empty MESSAGE

    if ((receivedMessage = new MESSAGE (fullSize)) == NULL) {
	return (MESSAGE_SYS_ERROR);
    }

	// get the message buffer

    buffer = receivedMessage -> getBuffer ();

	// read the message contents off the connection and delinearize
	// the contents

    memcpy (buffer, (char *) &netSize, sizeof(INT_32));

    if ((connectionRc = connection.read (size, buffer + sizeof(UINT_32), nread, stop)) == Connection::e_Timeout) {

	rc = MESSAGE_TIMEOUT;

    } else if (connectionRc != Connection::e_Ok) {
	
	rc = MESSAGE_LOST_CONNECTION;

    } else if (receivedMessage -> delinearize (messageSet) < 0) {

	rc = MESSAGE_INVALID;

    } else {

	rc = MESSAGE_OK;
    }

    if (rc != MESSAGE_OK) {
	DELETE_OBJECT(receivedMessage);
    }

    return (rc);
}


/* MessageDriver::unsecureSend

   Purpose: Send an unencrypted, unsigned message across a connection
  
     Input: connection = the connection
                         (Connection &)

            message    = the message
                         (MESSAGE &)

	    stop       = timeout or NULL if there is no timeout
			 (struct timeval *)

    Output: The procedure returns MESSAGE_OK if the message is sent 
            successfully, MESSAGE_SYS_ERROR if the message could not be
            linearized (which indicates a lack of memory, etc.), MESSAGE_TIMEOUT
	    if the timeout expired before the message could be written in its
	    entirety, and MESSAGE_LOST_CONNECTION if the connection breaks 
	    before the message could be written in its entirety.
*/

int MessageDriver::unsecureSend (Connection &connection, MESSAGE &message, struct timeval *stop)
{
    int rc;
    UINT_32 size;
    char *buffer;
    int nwritten;
    Connection::ConnectionErrors connectionRc;

	// linearize the message and write the message onto the connection

    if (message.linearize() < 0) {

	rc = MESSAGE_SYS_ERROR;

    } else {

	size    = message.getSize();
	buffer  = message.getBuffer();

	if ((connectionRc = connection.write (size, buffer, nwritten, stop)) == Connection::e_Timeout) {
	    rc = MESSAGE_TIMEOUT;
	} else if (connectionRc != Connection::e_Ok) {
	    rc = MESSAGE_LOST_CONNECTION;
	} else {
	    rc = MESSAGE_OK;
	}
    }

    return (rc);
}

/* MessageDriver::sendFromBuffer

   Purpose: Send the contents of a buffer across a connection

     Input: connection = the connection
                         (Connection &)

            buffer     = the buffer
                         (DynamicBuffer &)

    Output: The procedure returns MESSAGE_OK if the buffer contents are
            sent successfully, and MESSAGE_LOST_CONNECTION if the connection
            breaks before the buffer contents could be written in their
            entirety.
*/

int MessageDriver::sendFromBuffer (Connection &connection, DynamicBuffer &buffer)
{
    int code;
    INT_32 size;
    int nwritten;
    UINT_32 netSize;

	// write the buffer contents onto the connection

    size    = buffer.size();
    netSize = htonl (size);

    if (connection.write (sizeof(UINT_32), (char *) &netSize, nwritten) != Connection::e_Ok) {
	code = MESSAGE_LOST_CONNECTION;
    } else if (connection.write (size, buffer.value(), nwritten) != Connection::e_Ok) {
	code = MESSAGE_LOST_CONNECTION;
    } else {
	code = MESSAGE_OK;
    }

    return (code);
}

/* MessageDriver::receiveIntoBuffer

   Purpose: Receive a message from a connection and put it into a buffer

     Input: connection = the connection
                         (Connection &)

    Output: On success the procedure loads the incoming message into the
            buffer (DynamicBuffer &buffer) and returns MESSAGE_OK.  On
	    failure the procedure returns MESSAGE_LOST_CONNECTION (if the
            connection broke before the message was received in its entirety)
	    or MESSAGE_SYS_ERROR (if there was not enough free memory to
	    allocate a buffer for the incoming message).
	    
*/

int MessageDriver::receiveIntoBuffer (Connection &connection, DynamicBuffer &buffer)
{
    char *charBuffer;
    UINT_32 size;
    UINT_32 netSize;
    int nread;

	// read the message into the buffer

    if (connection.read (sizeof(UINT_32), (char *) &netSize, nread) != Connection::e_Ok) {
	return MESSAGE_LOST_CONNECTION;
    }

    size = ntohl (netSize);

    if ((charBuffer = new char [size]) == NULL) {
	return MESSAGE_SYS_ERROR;
    }

    if (connection.read (size, charBuffer, nread) != Connection::e_Ok) {
	DELETE_ARRAY_OBJECT(charBuffer);
	return MESSAGE_LOST_CONNECTION;
    }

    DynamicBuffer tempBuffer (size, charBuffer, BufferOwnership::e_TAKE);
    buffer = tempBuffer;
    return MESSAGE_OK;
}

/* MessageDriver::secureSend

   Purpose: Send a signed and/or encrypted message across a connection

     Input: connection = the connection
                         (Connection &)

            machine    = machine on the other end of the connection
                         (const MachineId *)

#ifdef FIX_LATER
	    machine is needed to construct the keyname of the recipient server.
	    This keyname should be constructed at a higher level and provided
	    as input to this routine.
#endif

	    security   = security description of the message sender
                         (const BASE_SECURITY &)

#ifdef FIX_LATER
	    We should not pass the entire security structure into this routine.
	    Only a few of its components are needed here (e.g., the keyname of 
	    the sender).
#endif

	    message    = the message
                         (MESSAGE &) 

	    stop       = timeout or NULL if there is no timeout
		  	 (struct timeval *)

    Output: The procedure returns MESSAGE_OK if the message is sent
            successfully, MESSAGE_ENCRYPT if the message could not be
            encrypted (e.g., a bad keyname), MESSAGE_SYS_ERROR if a system
            error occurs during transmission or encryption, and 
	    MESSAGE_LOST_CONNECTION if the connection breaks before the
            message could be sent in its entirety, 
*/

int MessageDriver::secureSend (Connection &connection, const MachineId &machine, const BASE_SECURITY &security, MESSAGE &message, struct timeval *stop)
{
    int code;
    MESSAGE *securedMessage = NULL;

	// encrypt the message if necessary

    if ((code = secureForServer (machine, security, message, securedMessage)) != MESSAGE_OK) {
	DELETE_OBJECT(securedMessage);
	return (code);
    }

	// send the message(s)

    code = MessageDriver::unsecureSend (connection, *securedMessage, stop);

    if (code == MESSAGE_OK) {
	if  (securedMessage -> getFlag() == PROT_KEY) {
	    code = MessageDriver::unsecureSend (connection, message, stop);
	}
    }

	// done

    DELETE_OBJECT(securedMessage);
    return (code);
}
