/* Agent Tcl
   Bob Gray
   9 February 1995

   genMessage.cc

   This file implements the functions associated with messages.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "agentAgentId.h"	// AgentId
#include "agentPermit.h"	// Permit

#include "agentId.h"
#include "agentPermit.h"
#include "genMessage.h"
#include "genSecurity.h"
#include "genUtility.h"
#include "truefalse.h"

  /* messages going up the socket */

static MESSAGE_TYPE messagesToServerTypes[NUM_REQUESTS] = 
{
  {REQ_BEGIN    , 3, {MESG_TYPE_STR,       /* actual machine name           */
                      MESG_TYPE_IP,        /* actual machine IP address     */
		      MESG_TYPE_LONG,      /* process id                    */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, 
  {REQ_NAME     , 2, {MESG_TYPE_LONG,      /* local id                      */
                      MESG_TYPE_STR,       /* requested name                */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_END      , 2, {MESG_TYPE_STR,       /* local name                    */
		      MESG_TYPE_LONG,      /* local id                      */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, 
  {REQ_FORCE    , 2, {MESG_TYPE_STR,       /* local name                    */
		      MESG_TYPE_LONG,      /* local id                      */
	 	      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}, 
  {REQ_SUBMIT   , 9, {MESG_TYPE_BYTE,      /* SUB_SCRIPT, SUB_FORK, SUB_JUMP */
	 	      MESG_TYPE_ID,        /* root agent id                  */
		      MESG_TYPE_ID,	   /* root home agent id             */
		      MESG_TYPE_ID,        /* local agent id                 */
		      MESG_TYPE_ID,	   /* local home agent id            */
		      MESG_TYPE_STR,	   /* language name                  */
                      MESG_TYPE_BINARY,    /* script or state                */
		      MESG_TYPE_BINARY,	   /* restrictions   		     */
		      MESG_TYPE_BINARY,	   /* masks                          */
		      0, 0, 0, 0}},
  {REQ_MESSAGE  , 5, {MESG_TYPE_ID,        /* local agent id                */
                      MESG_TYPE_STR,       /* destination name              */
                      MESG_TYPE_LONG,      /* destination numeric id        */
                      MESG_TYPE_LONG,      /* message code                  */
                      MESG_TYPE_STR,       /* message string                */
		      0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_MEETING  , 9, {MESG_TYPE_ID,        /* local agent id                */
                      MESG_TYPE_STR,       /* destination name              */
                      MESG_TYPE_LONG,      /* destination id                */
                      MESG_TYPE_BYTE,      /* ACT_ACCEPT, ACT_REFUSE ...    */
		      MESG_TYPE_LONG,      /* local handle                  */
		      MESG_TYPE_LONG,      /* remote handle                 */
		      MESG_TYPE_STR,       /* actual machine                */
		      MESG_TYPE_IP,        /* actual IP address             */
		      MESG_TYPE_LONG,      /* port number                   */
		      0, 0, 0, 0}},
  {REQ_EVENT    , 5, {MESG_TYPE_ID,        /* local agent id                */
                      MESG_TYPE_STR,       /* destination name              */
                      MESG_TYPE_LONG,      /* destination id                */
                      MESG_TYPE_STR,       /* event tag                     */
                      MESG_TYPE_STR,       /* event string                  */
		      0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_GET      , 2, {MESG_TYPE_LONG,      /* local id                      */
		      MESG_TYPE_LONG,	   /* local pid			   */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_INFO     , 3, {MESG_TYPE_BYTE,      /* INFO_ALL_IDS ...              */
                      MESG_TYPE_STR,       /* local name                    */
                      MESG_TYPE_LONG,      /* local id                      */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_STATUS   , 5, {MESG_TYPE_BYTE,      /* STATUS_EXISTS 		   */
		      MESG_TYPE_BYTE,	   /* STATUS_EVENT, STATUS_NO_EVENT */
		      MESG_TYPE_STR,	   /* agent name                    */
		      MESG_TYPE_LONG,      /* agent id			   */
		      MESG_TYPE_LONG,      /* requestor agent id            */
		      0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_IP       , 0, {0, 
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_AUTH     , 1, {MESG_TYPE_LONG,	   /* agent id                      */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_SECURITY , 1, {MESG_TYPE_SECURE,	   /* owner keyname                 */
		      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {REQ_ERROR    , 5, {MESG_TYPE_ID,        /* local agent id                */
                      MESG_TYPE_STR,       /* destination name              */
                      MESG_TYPE_LONG,      /* destination numeric id        */
                      MESG_TYPE_LONG,      /* message code                  */
                      MESG_TYPE_STR,       /* message string                */
		      0, 0, 0, 0, 0, 0, 0, 0}}
};

  /* messages going from the server to an agent */

static MESSAGE_TYPE messagesFromServerTypes[NUM_RESPONSES] =
{
  {RESP_IP     , 2, {MESG_TYPE_STR,	  /* server                        */
		     MESG_TYPE_IP,	  /* IP address                    */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_ID     , 1, {MESG_TYPE_ID,        /* server                        */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_OK     , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_ERROR  , 1, {MESG_TYPE_BYTE,      /* server error code             */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_MESSAGE, 5, {MESG_TYPE_LONG,      /* number of items at server     */
		     MESG_TYPE_ID,        /* source agent id               */
		     MESG_TYPE_SECURE,	  /* security information          */
                     MESG_TYPE_LONG,      /* message code                  */
                     MESG_TYPE_STR,       /* message string                */
		     0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_MEETING, 9, {MESG_TYPE_LONG,      /* number of items at server     */
		     MESG_TYPE_ID,        /* source agent id               */
		     MESG_TYPE_SECURE,	  /* security information	   */
                     MESG_TYPE_BYTE,      /* ACT_REQUEST, ACT_ACCEPT, etc. */
		     MESG_TYPE_LONG,      /* local handle                  */
	             MESG_TYPE_LONG,      /* remote handle                 */
                     MESG_TYPE_STR,       /* actual machine                */
                     MESG_TYPE_IP,        /* actual IP address             */ 
                     MESG_TYPE_LONG,      /* port number                   */
		     0, 0, 0, 0}},
  {RESP_EVENT  , 5, {MESG_TYPE_LONG,      /* number of items at server     */
		     MESG_TYPE_ID,        /* source agent id               */
		     MESG_TYPE_SECURE,    /* security information          */
                     MESG_TYPE_STR,       /* event tag                     */
                     MESG_TYPE_STR,       /* event string                  */
		     0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_NONE   , 0, {0,
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_IDS    , 1, {MESG_TYPE_IDS,       /* ID_LIST                        */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_NAMES  , 1, {MESG_TYPE_NAMES,     /* NAME_LIST                      */ 
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
  {RESP_INFO   , 9, {MESG_TYPE_STR,       /* actual server                  */
                     MESG_TYPE_IP,        /* actual server IP               */
		     MESG_TYPE_ID,	  /* 4-element agent id             */
		     MESG_TYPE_SECURE,    /* security information           */
                     MESG_TYPE_LONG,      /* process id (agent interpreter) */
                     MESG_TYPE_LONG,      /* process id (agent handler)     */
                     MESG_TYPE_LONG,      /* messages in queue              */
                     MESG_TYPE_LONG,      /* meetings in queue              */
                     MESG_TYPE_LONG,      /* events in queue                */
		     0, 0, 0, 0}},
  {RESP_BEGIN  , 2, {MESG_TYPE_ID,        /* server                         */
		     MESG_TYPE_BINARY,    /* 64 random bytes                */
		     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
}; 

    /* messages going to an interpreter */

static MESSAGE_TYPE messagesToInterpreterTypes[NUM_INTERP] = 
{
    {INTERP_SUBMIT,13, {MESG_TYPE_BYTE,       /* SUB_SCRIPT, SUB_FORK ...    */
		        MESG_TYPE_ID,         /* root agent id               */
			MESG_TYPE_ID,         /* root home agent id          */
		        MESG_TYPE_ID,	      /* local home agent id         */
		        MESG_TYPE_ID,         /* local agent id              */
		        MESG_TYPE_BINARY,     /* the script or state         */
			MESG_TYPE_SECURE_ALL, /* security information        */
			MESG_TYPE_BINARY,     /* 64 random bytes             */
			MESG_TYPE_BINARY,     /* restrictions                */
			MESG_TYPE_BINARY,     /* masks                       */
			MESG_TYPE_BYTE,       /* trusted-machine flag        */
			MESG_TYPE_PERMIT,     /* starting permit             */
			MESG_TYPE_STR	      /* agent temp directory        */
		        }}
};

static MESSAGE_TYPE messagesAcrossMeetingsTypes[NUM_MEETING] =
{
    {MEETING_MESSAGE, 2, {MESG_TYPE_LONG,         /* message code       */
			  MESG_TYPE_STR,	  /* message string     */
			  0, 
			  0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {MEETING_FILE   , 1, {MESG_TYPE_LONG,         /* file length        */
			  0,
		          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {MEETING_DONE   , 0, {0,
		          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}}
};

MESSAGE_SET messagesToServer	   = {NUM_REQUESTS , messagesToServerTypes};
MESSAGE_SET messagesFromServer	   = {NUM_RESPONSES, messagesFromServerTypes};
MESSAGE_SET messagesToInterpreter  = {NUM_INTERP   , messagesToInterpreterTypes};
MESSAGE_SET messagesAcrossMeetings = {NUM_MEETING  , messagesAcrossMeetingsTypes};

/* MESSAGE::~MESSAGE

   Purpose: This procedure is the destructor for class MESSAGE.
*/

MESSAGE::~MESSAGE ()
{
    for (int i = 0; i < n; i++) {

	if (elements[i].dealloc) {

	    switch (types[i]) {

		case MESG_TYPE_ID:

		    DELETE_OBJECT_WITH_CAST(AgentId*,elements[i].string);
		    break;

		case MESG_TYPE_IDS:

		    DELETE_OBJECT_WITH_CAST(ID_LIST*,elements[i].string);
		    break;

		case MESG_TYPE_NAMES:

		    DELETE_OBJECT_WITH_CAST(NAME_LIST*,elements[i].string);
		    break;

		case MESG_TYPE_PERMIT:

		    DELETE_OBJECT_WITH_CAST(Permit*,elements[i].string);
		    break;

		case MESG_TYPE_SECURE:
		case MESG_TYPE_SECURE_ALL:

		    DELETE_OBJECT_WITH_CAST(BASE_SECURITY*,elements[i].string);
		    break;

		default:

		    DELETE_ARRAY_OBJECT(elements[i].string);
	    }
	}
    }

    if (buffer != staticBuffer) {
	DELETE_ARRAY_OBJECT(buffer);
    }
}
 
/* MESSAGE::MESSAGE

   Purpose: This procedure is a constructor for class MESSAGE.

     Input: type = type of the message elements
		   (const MESSAGE_TYPE &)
*/

MESSAGE::MESSAGE (const MESSAGE_TYPE &type):
	buffer	   ((char *) NULL),
	flag	   (type.flag),
	n 	   (type.n)
{
    for (int i = 0; i < n; i++) {
	types[i]             = type.types[i];
	elements[i].dealloc  = e_FALSE;
	elements[i].string   = (char *) NULL;
	elements[i].number   = 0;
    }
}

/* MESSAGE::MESGAE

   Purpose: This procedure is a constructor for class MESSAGE.

     Input: size = required buffer size 
		   (UINT_32)
*/

MESSAGE::MESSAGE (UINT_32 size):
    bufferSize	(size),
    buffer	((char *) NULL),
    flag	(0),
    n		(0)
{
    if (size < STATIC_BUFFER_SIZE) {
	buffer = staticBuffer;
    } else {
	buffer = new char [size];
    }
}

/* MESSAGE::delinearize

   Purpose: This procedure delinearizes the buffer.

     Input: set = message set that should contain the incoming message 
		  (const MESSAGE_SET &)

    Output: The procedure returns -1 if the buffer is not formatted properly.
	    Otherwise the procedure delinearizes the buffer and returns 0.
*/

int MESSAGE::delinearize (const MESSAGE_SET &set)
{
    int i;
    UINT_32 j;
    ID_LIST *idList;
    NAME_LIST *nameList;
    char *string;
    char *bp;
    char *end_bp;
    UINT_32 length;
    UINT_8 integer8;
    UINT_16 integer16;
    UINT_32 integer32;
    AgentId *id;
    Permit *permit;
    BASE_SECURITY *security;

	/* assertions */

    assert (bufferSize > 0);
    assert (buffer != NULL);

	/* start and end of the relevant data in the buffer */ 

    bp     = buffer + sizeof(UINT_32);
    end_bp = buffer + bufferSize;

	/* get the flag */

    flag = *bp++;

	/* fill in the types */

    if (flag > set.n) {
	return (-1);
    }

    n = set.messages[flag].n;

    for (i = 0; i < n; ++i) {
	types[i]             = set.messages[flag].types[i];
	elements[i].dealloc  = e_FALSE;
	elements[i].string   = (char *) NULL;
	elements[i].number   = 0;
    }

	/* delinearize */

    for (i = 0; i < n; i++) {

	switch (types[i]) {

	    case MESG_TYPE_NAMES:

		if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
		    return (-1);
		}

		nameList = new NAME_LIST (length);
		elements[i].string  = (char *) nameList;
		elements[i].dealloc = e_TRUE;  

		for (j = 0; j < length; j++) {

		    if ((bp = breakout_string (bp, end_bp, &string)) == NULL) {
			return (-1);
		    }

		    nameList -> fill (j, string);
		    DELETE_ARRAY_OBJECT(string);
		}

		break;

	    case MESG_TYPE_IDS:

		if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
		    return (-1);
		}

		if (bp + length * sizeof(UINT_32) > end_bp) {
		    return (-1);
		}

		idList = new ID_LIST (length);
		elements[i].string  = (char *) idList;
		elements[i].dealloc = e_TRUE;  

		for (j = 0; j < length; j++) {
		    bp = fast_src_memcpy ((char *) &integer32, bp, sizeof(UINT_32));
		    idList -> ids[j] = ntohl (integer32);
		}

		break;

	    case MESG_TYPE_BYTE:

		if ((bp = breakout_byte (bp, end_bp, integer8)) == NULL) {
		    return (-1);
		}
 
		elements[i].number = integer8; 
		break;

	    case MESG_TYPE_ID:

		{
		    id = new AgentId ();
		    elements[i].string = (char *) id;
		    elements[i].dealloc = e_TRUE;

		    AgentIdUnpacker unpacker (*id);

		    if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
		        return (-1);
		    }

		    break;
		}

	    case MESG_TYPE_PERMIT:

		{
		    permit = new Permit ();
	            elements[i].string = (char *) permit;
		    elements[i].dealloc = e_TRUE;

		    PermitUnpacker unpacker (*permit);

		    if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
		        return (-1);
		    }

		    break;
		}

	    case MESG_TYPE_SECURE_ALL:

		security = new BASE_SECURITY ();
		elements[i].string = (char *) security;
		elements[i].dealloc = e_TRUE;

		if ((bp = security -> completeDelinearize (bp, end_bp)) == NULL) {
		    return (-1);
		}

		break;

	    case MESG_TYPE_SECURE:

		security = new BASE_SECURITY ();
		elements[i].string = (char *) security;
		elements[i].dealloc = e_TRUE;

		if ((bp = security -> idDelinearize (bp, end_bp)) == NULL) {
		    return (-1);
		}

		break;

	    case MESG_TYPE_SHORT:

		if (bp + sizeof(UINT_16) > end_bp) {
		    return (-1);
		}

		bp = fast_src_memcpy ((char *) &integer16, bp, sizeof(UINT_16));
		elements[i].number = ntohs (integer16);
		break;

	    case MESG_TYPE_STR:

		if ((bp = breakout_string (bp, end_bp, &elements[i].string)) == NULL) {
		    return (-1);
		}

		elements[i].dealloc = e_TRUE;
		break;

	    case MESG_TYPE_BINARY:

		    /* get the length */

		if ((bp = breakout_long (bp, end_bp, length)) == NULL) {
		    return (-1);
		}

		if (bp + length > end_bp) {
		    return (-1);
		}

		    /* create the buffer */

		elements[i].number  = length;
		elements[i].string  = new char [length]; 
		elements[i].dealloc = e_TRUE;

		    /* copy the data */

		bp = fast_src_memcpy (elements[i].string, bp, length);
		break;

	    case MESG_TYPE_LONG:
        
		if ((bp = breakout_long (bp, end_bp, elements[i].number)) == NULL) {
		    return (-1);
		}

		break;

	    case MESG_TYPE_IP:

		if ((bp = breakout_ip (bp, end_bp, elements[i].number)) == NULL) {
		    return (-1);
		}

		break;
	}
    }

    return (0); 
}

/* MESSAGE::linearize

   Purpose: Linearize the message

     Input: None

    Output: The procedure returns -1 if it is unable to linearize the message.
	    Otherwise the procedure returns 0.
*/

struct TCPIP_VALUE
{
    UINT_8  net_byte;
    UINT_16 net_short;
    UINT_32 net_long;
    UINT_32 host_long;
};

int MESSAGE::linearize (void)
{
    char *bp;                      
    register int i;
    register int j;
    ID_LIST *idList;
    UINT_32 net_long;
    NAME_LIST *nameList;
    UINT_32 netSize;
    BASE_SECURITY *security;
    TCPIP_VALUE values[MAX_ELEMENTS];    /* buffer for converted elements */

	/* have we already done this? */

    if (buffer != NULL) {
	return (0);
    }

	/* nothing yet */ 

    bufferSize = 0;

	/* determine the size of the message */

    bufferSize += sizeof(UINT_8);

    for (i = 0; i < n; i++) {

	switch (types[i]) {

	    case MESG_TYPE_BYTE:

		values[i].net_byte = elements[i].number;
		bufferSize += sizeof (UINT_8);
		break;

	    case MESG_TYPE_SHORT:

	  	values[i].net_short = htons (elements[i].number);
		bufferSize += sizeof (UINT_16);
		break;

	    case MESG_TYPE_LONG:

		values[i].net_long = htonl (elements[i].number);
		bufferSize += sizeof (UINT_32);
		break;

	    case MESG_TYPE_ID:

		{
		    AgentIdPacker packer (* ((AgentId *) elements[i].string));
		    bufferSize += packer.getPackedSize();
		    break;
		}


	    case MESG_TYPE_PERMIT:

		{
	            PermitPacker packer (* ((Permit *) elements[i].string));
		    bufferSize += packer.getPackedSize();
		    break;
		}

	    case MESG_TYPE_SECURE_ALL:

		security = (BASE_SECURITY *) elements[i].string;
		bufferSize += security -> getCompleteLinearSize ();
		break;

	    case MESG_TYPE_SECURE:

		security = (BASE_SECURITY *) elements[i].string;
		bufferSize += security -> getIdLinearSize ();
		break;

	    case MESG_TYPE_IP:

		values[i].net_long = elements[i].number;
		bufferSize += sizeof (UINT_32);
		break;

	    case MESG_TYPE_NAMES:

		nameList = (NAME_LIST *) elements[i].string;
		values[i].host_long = nameList -> count;
		values[i].net_long  = htonl (values[i].host_long);

		bufferSize += (nameList -> count) * sizeof(UINT_32) + sizeof(UINT_32);

		for (j = 0; j < nameList -> count; j++) {
		    bufferSize += nameList -> lengths[j];
		}

		break;

	    case MESG_TYPE_IDS:

		idList = (ID_LIST *) elements[i].string;
		values[i].host_long = idList -> count;
		values[i].net_long  = htonl (values[i].host_long);
		bufferSize += idList -> count * sizeof(UINT_32) + sizeof(UINT_32);
		break;

	    case MESG_TYPE_BINARY:

		values[i].host_long = elements[i].number; 
		values[i].net_long  = htonl (values[i].host_long);
		bufferSize += values[i].host_long + sizeof(UINT_32);
		break;

	    case MESG_TYPE_STR:

		if (elements[i].string == NULL) {
	
		    values[i].host_long = 0;
		    values[i].net_long  = 0;
		    bufferSize += sizeof (UINT_32);

		} else {

		    values[i].host_long = strlen(elements[i].string) + 1;
		    values[i].net_long  = htonl (values[i].host_long);
		    bufferSize += values[i].host_long;
		    bufferSize += sizeof (UINT_32);
		}

		break;
	}
    }

	/* space for the size field */

    netSize = htonl (bufferSize);
    bufferSize += sizeof(UINT_32);

	/* use the static buffer if it is large enough; otherwise allocate */
	/* a dynamic buffer                                                */

    if (bufferSize <= STATIC_BUFFER_SIZE) {

	buffer = staticBuffer;

    } else {

	if ((buffer = new char [bufferSize]) == NULL) {
	    return (-1);
	}
    }

    bp = buffer;

	/* put in the size and flag */

    bp = fast_memcpy (bp, (char *) &netSize, sizeof(UINT_32));
    bp = fast_memcpy (bp, (char *) &flag, sizeof(UINT_8));

	/* put in each component */

    for (i = 0; i < n; i++) {

	switch (types[i]) {

	    case MESG_TYPE_BYTE:

		bp = fast_memcpy (bp, (char *) &values[i].net_byte, sizeof(UINT_8));
		break;

	    case MESG_TYPE_SHORT:

		bp = fast_memcpy (bp, (char *) &values[i].net_short, sizeof(UINT_16));
		break;

	    case MESG_TYPE_STR:

		bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));
		bp = fast_memcpy (bp, elements[i].string, values[i].host_long);
		break;

	    case MESG_TYPE_ID:

		{
		    AgentIdPacker packer (* ((AgentId *) elements[i].string));
		    bp = packer.pack (bp);
		    break;
		}
		
	    case MESG_TYPE_PERMIT:

		{
		    PermitPacker packer (* ((Permit *) elements[i].string));
		    bp = packer.pack (bp);
		    break;
		}

	    case MESG_TYPE_SECURE_ALL:

	  	security = (BASE_SECURITY *) elements[i].string;
		bp = security -> completeLinearize (bp);
		break;

	    case MESG_TYPE_SECURE:

	  	security = (BASE_SECURITY *) elements[i].string;
		bp = security -> idLinearize (bp);
		break;
	
	    case MESG_TYPE_LONG:

		bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));
		break;

	    case MESG_TYPE_IP:

		bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));
		break;

	    case MESG_TYPE_NAMES:

		bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));

		nameList = (NAME_LIST *) elements[i].string;

		for (j = 0; j < nameList -> count; j++) {
		    net_long = htonl (nameList -> lengths[j]);
		    bp = fast_memcpy (bp, (char *) &net_long, sizeof(UINT_32));
		    bp = fast_memcpy (bp, nameList -> names[j], nameList -> lengths[j]);
		}

		break;

	    case MESG_TYPE_IDS:

		bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));

		idList = (ID_LIST *) elements[i].string;

		for (j = 0; j < idList -> count; j++) {
		    net_long = htonl (idList -> ids[j]);
		    bp = fast_memcpy (bp, (char *) &net_long, sizeof(UINT_32));
		}

		break;

	    case MESG_TYPE_BINARY:

		bp = fast_memcpy (bp, (char *) &values[i].net_long, sizeof(UINT_32));
		bp = fast_memcpy (bp, elements[i].string, values[i].host_long); 
		break;
	}
    }

	/* done */

    return (0);
}

/* requestFlagToString

   Purpose: Convert a request flag to a descriptive string

     Input: flag = the request flag
		   (UINT_8)

    Output: The procedure returns a pointer to a statically allocated
	    string that describes the request flag.
*/

static char *requestStrings[NUM_REQUESTS] = 
{
	"REQ_BEGIN",
	"REQ_NAME",
	"REQ_END",
	"REQ_FORCE",
	"REQ_SUBMIT",
	"REQ_MESSAGE",
	"REQ_MEETING",
	"REQ_EVENT",
	"REQ_GET",
	"REQ_INFO",
	"REQ_STATUS",
	"REQ_IP",
	"REQ_AUTH",	
	"REQ_SECURITY"
};

char *requestFlagToString (UINT_8 flag)
{
	assert (flag < NUM_REQUESTS);
	return (requestStrings[flag]);
}
