/* Agent Tcl
   Bob Gray
   9 Februrary 1995

   genMessage.h

   This file defines the data structures and flags associated with messages.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _GEN_MESSAGE_H
#define _GEN_MESSAGE_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#include "agentId.h"
#include "agentSecure.h"
#include "genUtility.h"

   /* forward declarations */

class AgentId;
class MESSAGE_SET;
class Permit;

    /* maximum number of elements in a message */

const UINT_8 MAX_ELEMENTS = 13;

    /* element types */

const UINT_8 MESG_TYPE_STR		= 0;
const UINT_8 MESG_TYPE_LONG		= 1;
const UINT_8 MESG_TYPE_BYTE		= 2;
const UINT_8 MESG_TYPE_SHORT		= 3;
const UINT_8 MESG_TYPE_IP		= 4;
const UINT_8 MESG_TYPE_BINARY 		= 5;
const UINT_8 MESG_TYPE_IDS    		= 6;
const UINT_8 MESG_TYPE_NAMES		= 7;
const UINT_8 MESG_TYPE_ID  		= 8;
const UINT_8 MESG_TYPE_SECURE		= 9;
const UINT_8 MESG_TYPE_SECURE_ALL	= 10;
const UINT_8 MESG_TYPE_PERMIT		= 11;

    /* request flags */

const UINT_8 REQ_BEGIN		= 0;
const UINT_8 REQ_NAME		= 1;
const UINT_8 REQ_END		= 2;
const UINT_8 REQ_FORCE		= 3;
const UINT_8 REQ_SUBMIT 	= 4;
const UINT_8 REQ_MESSAGE	= 5;
const UINT_8 REQ_MEETING	= 6;
const UINT_8 REQ_EVENT		= 7;
const UINT_8 REQ_GET		= 8;
const UINT_8 REQ_INFO		= 9;
const UINT_8 REQ_STATUS		= 10;
const UINT_8 REQ_IP		= 11;
const UINT_8 REQ_AUTH		= 12;
const UINT_8 REQ_SECURITY	= 13;
const UINT_8 REQ_ERROR		= 14;
const UINT_8 NUM_REQUESTS	= 15;

    /* convert a request flag to a string */

char *requestFlagToString 
	(UINT_8 flag);

    /* response flags */

const UINT_8 RESP_IP		= 0;
const UINT_8 RESP_ID		= 1;
const UINT_8 RESP_OK		= 2;
const UINT_8 RESP_ERROR		= 3;
const UINT_8 RESP_MESSAGE	= 4;
const UINT_8 RESP_MEETING	= 5;
const UINT_8 RESP_EVENT		= 6;
const UINT_8 RESP_NONE		= 7;
const UINT_8 RESP_IDS		= 8;
const UINT_8 RESP_NAMES		= 9;
const UINT_8 RESP_INFO		= 10;
const UINT_8 RESP_BEGIN		= 11;
const UINT_8 NUM_RESPONSES	= 12;

    /* interpreter message flags */

const UINT_8 INTERP_SUBMIT	= 0;
const UINT_8 NUM_INTERP		= 1;

    /* meeting messages */

const UINT_8 MEETING_MESSAGE	= 0;
const UINT_8 MEETING_FILE	= 1;
const UINT_8 MEETING_DONE	= 2;
const UINT_8 NUM_MEETING	= 3;

    /* status types */

const UINT_8 STATUS_EXISTS	= 0;

    /* status requests can generate an event at a later time if desired */

const UINT_8 STATUS_EVENT	= 0;
const UINT_8 STATUS_NO_EVENT	= 1;

    /* information that can be requested from the server */

const UINT_8 INFO_ALL_IDS	= 0;
const UINT_8 INFO_ALL_NAMES	= 1;
const UINT_8 INFO_ID		= 2;

    /* flags to describe whether or not the agent came from a trusted machine */

enum TrustedFlags {
    e_MACHINE_UNTRUSTED			= 0x0,
    e_MACHINE_TRUSTED_BUT_UNVERIFIED	= 0x1,
    e_MACHINE_TRUSTED			= 0x2
};

    /* structure that contains a message type */

struct MESSAGE_TYPE
{
    UINT_8 flag;                  /* message flag       */
    int n;                        /* number of elements */
    UINT_8 types[MAX_ELEMENTS];   /* types of elements  */
};

    /* strcuture that contains a message element */

struct MESSAGE_ELEMENT
{
    BOOLEAN dealloc;      /* TRUE if the element should be deallocated */
    UINT_32 number;       /* value is here if the element is numeric   */
    char *string;         /* value is here if the element is a string  */
};

    /* structure that contains a message */

class MESSAGE 
{
	    /* static buffer size */

	enum {
	    STATIC_BUFFER_SIZE = 1024
	};

	    /* message components and buffer */

	UINT_32 bufferSize;			 /* message buffer size    */
	char *buffer;                            /* dynamic message buffer */
	char staticBuffer[STATIC_BUFFER_SIZE];	 /* static message buffer  */
	UINT_8 flag;                             /* message flag           */ 
	int n;                                   /* number of elements     */
	UINT_8 types[MAX_ELEMENTS];              /* types of the elements  */
	MESSAGE_ELEMENT elements[MAX_ELEMENTS];  /* the elements           */

    public:

	    /* constructor and destructor */

	MESSAGE (UINT_32 size);
	MESSAGE (const MESSAGE_TYPE &type);
       ~MESSAGE ();

	    /* linearize and delinearize */

	int linearize (void);
	int delinearize (const MESSAGE_SET &set);

	    /* assemble the message */

	void addString (int i, char *string, BOOLEAN dealloc = e_FALSE) {
	    assert (types[i] == MESG_TYPE_STR);
	    elements[i].string = string;
	    elements[i].dealloc = dealloc;
	}

	void addByte (int i, UINT_8 number) {
	    assert (types[i] == MESG_TYPE_BYTE);
	    elements[i].number = number;
	}

	void addShort (int i, UINT_16 number) {
	    assert (types[i] == MESG_TYPE_SHORT);
	    elements[i].number = number;
	}

	void addLong (int i, UINT_32 number) {
	    assert (types[i] == MESG_TYPE_LONG);
	    elements[i].number = number;
	}

	void addIp (int i, UINT_32 ip) {
	    assert (types[i] == MESG_TYPE_IP);
	    elements[i].number = ip;
	}

	void addBinary (int i, UINT_32 length, char *p_buffer, BOOLEAN dealloc = e_FALSE) {
	    assert (types[i] == MESG_TYPE_BINARY);
	    elements[i].number = length;
	    elements[i].string = p_buffer;
	    elements[i].dealloc = dealloc;
	}

	void addId (int i, AgentId *id, BOOLEAN dealloc = e_FALSE) {
	    assert (types[i] == MESG_TYPE_ID);
	    elements[i].string = (char *) id;
	    elements[i].dealloc = dealloc;
	}

	void addIds (int i, ID_LIST *ids, BOOLEAN dealloc = e_FALSE) {
	    assert (types[i] == MESG_TYPE_IDS);
	    elements[i].string = (char *) ids;
	    elements[i].dealloc = dealloc;
	}

	void addNames (int i, NAME_LIST *names, BOOLEAN dealloc = e_FALSE) {
	    assert (types[i] == MESG_TYPE_NAMES);
	    elements[i].string = (char *) names;
	    elements[i].dealloc = dealloc;
	}

	void addSecurity (int i, BASE_SECURITY *security, BOOLEAN dealloc = e_FALSE) {
	    assert ((types[i] == MESG_TYPE_SECURE) || (types[i] == MESG_TYPE_SECURE_ALL));
	    elements[i].string = (char *) security;
	    elements[i].dealloc = dealloc;
	}

	void addPermit (int i, Permit *permit, BOOLEAN dealloc = e_FALSE) {
	    assert (types[i] == MESG_TYPE_PERMIT);
	    elements[i].string = (char *) permit; 
	    elements[i].dealloc = dealloc;
	}

	    /* get the flag, size and buffer */

	UINT_8 getFlag (void) {
	    return (flag);
	}

	UINT_32 getSize (void) {
	    return (bufferSize);
	}

	char *getBuffer (void) {
	    return (buffer);
	}

	    /* get the pieces */

	char *getString (int i) {
	    assert (types[i] == MESG_TYPE_STR);
	    char *string = elements[i].string;
	    elements[i].string = NULL;
	    return (string);
	}

	UINT_8 getByte (int i) {
	    assert (types[i] == MESG_TYPE_BYTE);
	    return ((UINT_8) elements[i].number);
	}

	UINT_16 getShort (int i) {
	    assert (types[i] == MESG_TYPE_SHORT);
	    return ((UINT_16) elements[i].number);
	}

	UINT_32 getLong (int i) {
	    assert (types[i] == MESG_TYPE_LONG);
	    return ((UINT_32) elements[i].number);
	}

	UINT_32 getIp (int i) {
	    assert (types[i] == MESG_TYPE_IP);
	    return ((UINT_32) elements[i].number);
	}

	AgentId *getId (int i) {
	    assert (types[i] ==	MESG_TYPE_ID);
	    AgentId *id = (AgentId *) elements[i].string;
	    elements[i].string = NULL;
	    return (id);
	}
 
	BASE_SECURITY *getSecurity (int i) {
	    assert ((types[i] == MESG_TYPE_SECURE) || (types[i] == MESG_TYPE_SECURE_ALL));
	    BASE_SECURITY *security = (BASE_SECURITY *) elements[i].string;
	    elements[i].string = NULL;
	    return (security);
	}

	char *getBinary (int i, UINT_32 &length) {

	    char *l_buffer;

	    assert (types[i] == MESG_TYPE_BINARY);
 
	    if ((length = elements[i].number) == 0) {
		l_buffer = NULL;
	    } else {
		l_buffer = elements[i].string;
		elements[i].string = NULL; 
	    }

	    return (l_buffer); 
	}

	ID_LIST *getIds (int i) {
	    assert (types[i] == MESG_TYPE_IDS);
	    ID_LIST *list = (ID_LIST *) elements[i].string;
	    elements[i].string = NULL;
	    return (list);
	}

	NAME_LIST *getNames (int i) {
	    assert (types[i] == MESG_TYPE_NAMES);
	    NAME_LIST *list = (NAME_LIST *) elements[i].string;
	    elements[i].string = NULL;
	    return (list);
	}

	Permit *getPermit (int i) {
	    assert (types[i] == MESG_TYPE_PERMIT);
	    Permit *permit = (Permit *) elements[i].string;
	    elements[i].string = NULL;
	    return (permit);
	}
};

    /* structure that contains a SET of message types */

struct MESSAGE_SET
{
    int n;                    /* number of messages in the set */
    MESSAGE_TYPE *messages;   /* the messages                  */
};
    
    /* SETS of message types */

extern MESSAGE_SET messagesToServer;
extern MESSAGE_SET messagesFromServer;
extern MESSAGE_SET messagesToInterpreter;
extern MESSAGE_SET messagesAcrossMeetings;

#endif
