/* Agent Tcl
   Bob Gray
   20 April 1996

   genPGP.cc

   This file implements the routines that are used to access PGP.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "platExclusion.h"
#include "suppStrings.h"
#include "suppDString.h"
#include "genDefaults.h"
#include "genFile.h"
#include "genPipe.h"
#include "genPGP.h"
#include "genSecurity.h"

    /* size of the line buffer for PGP output */

const int PGP_LINE_SIZE = 2048;

    /* exit code that is used to detect "exec" failure */

const int EXEC_FAILURE_CODE = 127;

    /* PGP output string that indicates a keyname */

const char *const PGP_KEY_OUTPUT = "Good signature from user \"";

    /* PGP output string that indicates a need for random bits */

const char *const PGP_BITS_OUTPUT = "We need to generate";

    /* environment variable that contains full pathname of PGP executable */
    /* and the environment variable that contains the full pathname of    */
    /* the PGP key directory                                              */

const char *const PGP_PROG_ENV = "PGPPROG";
const char *const PGP_PATH_ENV = "PGPPATH";

    /* extension for encrypted PGP files */

const char *const PGP_EXTENSION	= "pgp";

    /* location of the PGP executable and the PGP key directory */
    /* and the output from the most recent PGP command          */ 

static BOOLEAN 	      pgpInitializationDone = e_FALSE;
static DynamicString *pgpExecutablePath     = NULL;
static DynamicString *pgpKeyDirectory       = NULL;
static DynamicString *pgpOutput 	    = NULL;

/* PGP_setExecutablePath and PGP_setKeyDirectory

   Purpose: Set the pathname of the PGP executable and the PGP key
  	    directory.
*/

void PGP_setExecutablePath (DynamicString path)
{
    if (pgpExecutablePath == NULL) {
	pgpExecutablePath = new DynamicString ();
    }

    *pgpExecutablePath = path;
}

void PGP_setKeyDirectory (DynamicString path)
{
    char *fullname;

    if (pgpKeyDirectory == NULL) {
	pgpKeyDirectory = new DynamicString ();
    }

    if ((fullname = FileUtility::fileTildeSubst (path.value())) == NULL) {
	DynamicString string;
	string.append ("PGP_setKeyDirectory could not resolve the tilde (~) in filename \"");
	string.append (path);
	string.append ("\"");
	abort_with_message (string.value());
    }

    pgpKeyDirectory -> empty ();
    pgpKeyDirectory -> append ("PGPPATH=");
    pgpKeyDirectory -> append (fullname);
    DELETE_ARRAY_OBJECT(fullname);
}

/* PGP_init

   Purpose: Initialize the PGP subsystem

     Input: None

    Output: The procedure identifies the full path of the pgp executable.
*/

void PGP_init (void)
{
    char *envString;

    if (!pgpInitializationDone) {

	pgpInitializationDone = e_TRUE;

	if (pgpExecutablePath == NULL) {

	    if ((envString = getenv (PGP_PROG_ENV)) == NULL) {
	        PGP_setExecutablePath (PGP_EXECUTABLE);
	    } else {
		PGP_setExecutablePath (envString);
	    }
	}

	if (pgpKeyDirectory == NULL) {

	    if ((envString = getenv (PGP_PATH_ENV)) == NULL) {
		PGP_setKeyDirectory (PGP_KEY_DIRECTORY);
	    } else {
		PGP_setKeyDirectory (envString);
	    }
	}

	pgpOutput = new DynamicString ();
    }
}

/* PGP_extension

   Purpose: Return the extension for PGP encrypted files

     Input: None

    Output: The procedure returns a pointer to a constant string that
	    contains the extension.
*/

const char *PGP_extension (void)
{
    return (PGP_EXTENSION);
}

/* PGP_error

   Purpose: Return the text of the most recent PGP error

     Input: None

    Output: The procedure returns a pointer to a static string that contains
	    the most recent PGP error.
*/

DynamicString PGP_error (void)
{
    if (pgpOutput == NULL) {
	DynamicString emptyString;
	return (emptyString);
    }

    return (*pgpOutput);
}

/* PGP_run

   Purpose: Fork PGP to perform an encryption (or descryption) task 

     Input: argv   = arguments
		     (char **argv)

	    signer = pointer to a char *
		     (char **)

    Output: The procedure returns -1 if PGP fails.  Otherwise the procedure
	    returns 0 for success.  On succes, if signer is not NULL, *signer 
	    is set to NULL if the encrypted data was not signed; otherwise
	    *signer is set to a dynamically allocated string that contains
	    the keyname of the signer.
*/

extern "C" int fdopen_thread (int, int);

static int PGP_run (char **argv, char **signer)
{
    int rc = 0;
    char *sp;
    pid_t pid;
    int count = 0;
    int connector[2];
    char *pgpenviron[2];
    FILE *pgpConnection;
    WAIT_STATUS_TYPE status;
    char buffer[PGP_LINE_SIZE + 1];
    static int keyOutLength = strlen(PGP_KEY_OUTPUT);
    static int bitsOutLength = strlen(PGP_BITS_OUTPUT);

	/* assertions on the parameters */

    assert (argv != NULL);

	/* block signals -- automatically released */

    MonitorLock lock;
    Guard guard (lock);

	/* initialize PGP if necessary */

    if (!pgpInitializationDone) {
	PGP_init ();
    }

    argv[0] = pgpExecutablePath -> value();
    
	/* set up the PGP environment */

    pgpenviron[0] = pgpKeyDirectory -> value();
    pgpenviron[1] = (char *) NULL;

	/* assume that the message is not signed */

    if (signer != NULL) {
	*signer = (char *) NULL;
    }

	/* construct a stream pipe that will connect the parent to PGP */

    if (PipeUtility::createStreamPipe (connector) != PipeUtility::e_OK) {
	return (-1);
    }

    if ((pgpConnection = fdopen (connector[0], "r+")) == NULL) {
	PipeUtility::closePipe (connector[0]);
	PipeUtility::closePipe (connector[1]);
	return (-1);
    }

	/* fork the child */

    if ((pid = fork ()) < 0) {
	fclose (pgpConnection);
	PipeUtility::closePipe (connector[0]);
	PipeUtility::closePipe (connector[1]);
	return (-1);
    }

	/* CHILD executes PGP -- first redirect stdin, stdout and stderr onto */
	/* connector[1] and then call "exec"                                  */

    if (pid == 0) {
	dup2 (connector[1], 0);
	dup2 (connector[1], 1);
	dup2 (connector[1], 2);
	(void) execve (argv[0], argv, pgpenviron);
	exit (EXEC_FAILURE_CODE);
    }				

	/* PARENT reads PGP output off of connector[0] and then waits for */
	/* the exit status -- if desired the loop will strip out the      */
	/* signature verification line                                    */

    PipeUtility::closePipe (connector[1]);
 
    pgpOutput -> empty ();

    while (fgets (buffer, PGP_LINE_SIZE, pgpConnection) != NULL) {

	    // see if the line contains the signer's key

	if (signer != NULL) {
	    if (!strncmp (buffer, PGP_KEY_OUTPUT, keyOutLength)) {
		*signer = strcpyWithAlloc (buffer + keyOutLength);
		(*signer)[strlen(*signer) - 3] = '\0'; 
	    }
	}

	    // add the line to the output buffer

	for (sp = buffer; *sp != '\0'; sp++) {
	    if (*sp == 07) {
		*sp = ' ';
	    }
	}

	pgpOutput -> append ("- ");
        pgpOutput -> append (*buffer != ' ' ? buffer : buffer + 1);

	    // see if the line is asking the user to manually generate random 
	    // bits -- this is an error, since although we can automatically
	    // generate random bits, we can not pass these bits on to the 
	    // separate PGP executable (without resorting to master and slave 
	    // tty's, which is too much work, mainly because we are eventually 
	    // going to abandon PGP anyways).

	if (!strncmp (buffer, PGP_BITS_OUTPUT, bitsOutLength)) {

		// special error message

	    pgpOutput -> empty ();
	    pgpOutput -> append ("* FATAL ERROR: \n");
	    pgpOutput -> append ("*\n");
	    pgpOutput -> append ("* PGP is asking for the manual generation of random bits.  Agent Tcl cannot\n");
	    pgpOutput -> append ("* handle this.  Please use the \"genrand\" utility to generate the file\n");
	    pgpOutput -> append ("* \"randseed.bin\" in directory \"");
	    pgpOutput -> append (pgpKeyDirectory -> value());
	    pgpOutput -> append ("\".\n");

		// kill the PGP child process
 
	    kill (pid, SIGQUIT);
	    rc = -1;
	    break;
	}

	    // we now have one more line

	count += 1;
    }

	// wait for the PGP process to finish

    while ((waitpid (pid, (int *) &status, 0) < 0) && ((errno == EINTR) || (errno == EAGAIN))) { 
	// empty
    }

	// weird test since PGP returns an exit code of 1 even when 
	// decryption succeeds  				    

    if (rc == 0) {

	if (!WIFEXITED(status)) {

	    rc = -1;

	} else if (WEXITSTATUS(status) == 0) {

	    rc = 0;

	} else if ((WEXITSTATUS(status) != EXEC_FAILURE_CODE) && (count == 0)) {

	    rc = 0;

	} else {
	
	    if (WEXITSTATUS(status) == EXEC_FAILURE_CODE) {
		pgpOutput -> empty ();
		pgpOutput -> append ("* FATAL ERROR: \n");
		pgpOutput -> append ("*\n");
		pgpOutput -> append ("* unable to exec the PGP program \"");
		pgpOutput -> append (*pgpExecutablePath);
		pgpOutput -> append ("\"\n");
	    }

	    rc = -1;
	}
    }

    fclose (pgpConnection);
    PipeUtility::closePipe (connector[0]);
    return (rc);
}

/* PGP_EncryptAndSign

   Purpose: Encrypt and sign a file using PGP

     Input: plaintextFilename = file that contains the plaintext
				(char *)

	    encryptedFilename = file that will contain the encrypted text
			        (char *)

	    recipientKey      = PGP keyname of the recipient   
				(char *)

            sender	      = PGP keyname and pass phrase of the sender
			        (class SEC_IDENTITY &)

    Output: The procedure returns -1 on error.  Otherwise the procedure returns
	    0 and the encrypted text is placed in the specified file.
*/

const int ES_argc = 16;

const int ES_index_plaintext	= 7;
const int ES_index_recipient	= 8;
const int ES_index_signer	= 10;
const int ES_index_phrase	= 12;
const int ES_index_encrypted	= 14;

static char *const ES_argv[ES_argc] = {
	(char *) NULL,		/* full PGP path      */
	"+batchmode",
	"+verbose=0",
	"+force",
	"+nomanual=on",
	"+compress=on",
	"-es",
	(char *) NULL,		/* plaintext filename */
	(char *) NULL,		/* recipient keyname  */
	"-u",			
	(char *) NULL, 		/* signer keyname     */
	"-z",
	(char *) NULL,		/* signer pass phrase */
	"-o",
	(char *) NULL,		/* encrypted filename */
	(char *) NULL		/* END OF COMMAND     */
};

int PGP_EncryptAndSign (char *plaintextFilename, char *encryptedFilename, char *recipientKey, SEC_IDENTITY &sender)
{
    int i;
    char *argv[ES_argc];

	/* assertions on the parameters */

    assert (plaintextFilename != NULL);
    assert (encryptedFilename != NULL);
    assert (recipientKey != NULL);
    assert (!(sender.m_keyname.isEmpty()));
    assert (!(sender.m_phrase.isEmpty()));

	/* fill in the command */

    for (i = 0; i < ES_argc; i++) {
	argv[i] = ES_argv[i];
    }

    argv[ES_index_plaintext] = plaintextFilename;
    argv[ES_index_recipient] = recipientKey;
    argv[ES_index_signer   ] = sender.m_keyname.value();
    argv[ES_index_phrase   ] = sender.m_phrase.value();
    argv[ES_index_encrypted] = encryptedFilename;

	/* run PGP */

    if (PGP_run (argv, (char **) NULL) < 0) {
	return (-1);
    }

    return (0);
}

/* PGP_Sign

   Purpose: Sign a file using PGP

     Input: plaintextFilename = file that contains the plaintext
				(char *)

	    encryptedFilename = file that will contain the encrypted text
			        (char *)

            sender	      = PGP keyname and pass phrase of the sender
			        (class SEC_IDENTITY &)

    Output: The procedure returns -1 on error.  Otherwise the procedure returns
	    0 and the encrypted text is placed in the specified file.
*/

const int S_argc = 16;

const int S_index_plaintext	= 7;
const int S_index_signer	= 9;
const int S_index_phrase	= 11;
const int S_index_encrypted	= 13;

static char *const S_argv[S_argc] = {
	(char *) NULL,          /* full PGP path      */	
	"+batchmode",
	"+verbose=0",
	"+force",
	"+nomanual=on",
	"+compress=on",
	"-s",
	(char *) NULL,		/* plaintext filename */
	"-u",			
	(char *) NULL, 		/* signer keyname     */
	"-z",
	(char *) NULL,		/* signer pass phrase */
	"-o",
	(char *) NULL,		/* encrypted filename */
	(char *) NULL		/* END OF COMMAND     */
};

int PGP_Sign (char *plaintextFilename, char *encryptedFilename, SEC_IDENTITY &sender)
{
    int i;
    char *argv[S_argc];

	/* assertions on the parameters */

    assert (plaintextFilename != NULL);
    assert (encryptedFilename != NULL);
    assert (!(sender.m_keyname.isEmpty()));
    assert (!(sender.m_phrase.isEmpty()));

	/* fill in the command */

    for (i = 0; i < S_argc; i++) {
	argv[i] = S_argv[i];
    }

    argv[S_index_plaintext] = plaintextFilename;
    argv[S_index_signer   ] = sender.m_keyname.value();
    argv[S_index_phrase   ] = sender.m_phrase.value();
    argv[S_index_encrypted] = encryptedFilename;

	/* run PGP */

    if (PGP_run (argv, (char **) NULL) < 0) {
	return (-1);
    }

    return (0);
}

/* PGP_Encrypt

   Purpose: Encrypt a file using PGP

     Input: plaintextFilename = file that contains the plaintext
				(char *)

	    encryptedFilename = file that will contain the encrypted text
			        (char *)

	    recipientKey      = PGP keyname of the recipient   
				(char *)

    Output: The procedure returns -1 on error.  Otherwise the procedure returns
	    0 and the encrypted text is placed in the specified file.
*/

const int E_argc = 12;

const int E_index_plaintext	= 7;
const int E_index_recipient	= 8;
const int E_index_encrypted	= 10;

static char *const E_argv[E_argc] = {
	(char *) NULL,	  	/* full PGP path      */
	"+batchmode",
	"+verbose=0",
	"+force",
	"+nomanual=on",
	"+compress=on",
	"-e",
	(char *) NULL,		/* plaintext filename */
	(char *) NULL,		/* recipient keyname  */
	"-o",
	(char *) NULL,		/* encrypted filename */
	(char *) NULL		/* END OF COMMAND     */
};

int PGP_Encrypt (char *plaintextFilename, char *encryptedFilename, char *recipientKey)
{
    int i;
    char *argv[E_argc];

	/* assertions on the parameters */

    assert (plaintextFilename != NULL);
    assert (encryptedFilename != NULL);
    assert (recipientKey != NULL);

	/* fill in the command */

    for (i = 0; i < E_argc; i++) {
	argv[i] = E_argv[i];
    }

    argv[E_index_plaintext] = plaintextFilename;
    argv[E_index_recipient] = recipientKey;
    argv[E_index_encrypted] = encryptedFilename;

	/* run PGP */

    if (PGP_run (argv, (char **) NULL) < 0) {
	return (-1);
    }

    return (0);
}

/* PGP_Decrypt

   Purpose: Decrypt a *non-signed* file using PGP

     Input: plaintextFilename = file that will contain the plaintext
				(char *)

	    encryptedFilename = file that contains the encrypted text 
			        (char *)

            receiver	      = PGP keyname and pass phrase of the sender
			        (class SEC_IDENTITY &)

    Output: The procedure returns -1 on error.  Otherwise the procedure returns
	    0 and the decrypted text is placed in the specified file.
*/

const int D_argc = 13;

const int D_index_encrypted	= 5;
const int D_index_decrypter	= 7;
const int D_index_phrase	= 9;
const int D_index_plaintext	= 11;

static char *const D_argv[D_argc] = {
	(char *) NULL,		/* full PGP path         */
	"+batchmode",
	"+verbose=0",
	"+force",
	"+nomanual=on",
	(char *) NULL,		/* encrypted filename    */
	"-u",			
	(char *) NULL, 		/* decrypter keyname     */
	"-z",
	(char *) NULL,		/* decrypter pass phrase */
	"-o",
	(char *) NULL,		/* plaintext filename    */ 
	(char *) NULL		/* END OF COMMAND        */
};

static int pgp_decrypt (char *encryptedFilename, char *plaintextFilename, SEC_IDENTITY &receiver, char **signer)
{
    int i;
    char *argv[D_argc];

	/* assertions on the parameters */

    assert (plaintextFilename != NULL);
    assert (encryptedFilename != NULL);
    assert (!(receiver.m_keyname.isEmpty()));
    assert (!(receiver.m_phrase.isEmpty()));

	/* fill in the command */

    for (i = 0; i < D_argc; i++) {
	argv[i] = D_argv[i];
    }

    argv[D_index_encrypted] = encryptedFilename;
    argv[D_index_decrypter] = receiver.m_keyname.value();
    argv[D_index_phrase   ] = receiver.m_phrase.value();
    argv[D_index_plaintext] = plaintextFilename;

	/* run PGP */

    if (PGP_run (argv, signer) < 0) {
	return (-1);
    }

    return (0);
}

int PGP_Decrypt (char *encryptedFilename, char *plaintextFilename, SEC_IDENTITY &receiver)
{
    return (pgp_decrypt (encryptedFilename, plaintextFilename, receiver, (char **) NULL));
};

/* PGP_DecryptSigned

   Purpose: Decrypt a *signed* file using PGP

     Input: plaintextFilename = file that will contain the plaintext
				(char *)

	    encryptedFilename = file that contains the encrypted text 
			        (char *)

            receiver	      = PGP keyname and pass phrase of the sender
			        (class SEC_IDENTITY &)

    Output: The procedure returns NULL on error.  Otherwise the procedure
	    returns a pointer to a dyamically-allocated string that contains
	    the keyname of the signer. 
*/

char *PGP_DecryptSigned (char *encryptedFilename, char *plaintextFilename, SEC_IDENTITY &receiver)
{
    char *signer = NULL;

    if (pgp_decrypt (encryptedFilename, plaintextFilename, receiver, &signer) < 0) {
	DELETE_OBJECT(signer);
	return ((char *) NULL);
    }

    return (signer);
}
