/* Agent Tcl
   Bob Gray
   10 January 1996

   genRestrict.cc

   This file implements the permit routines.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "platMath.h"
#include "platSystem.h"
#include "platTimers.h"
#include "platTimeval.h"
#include "agentId.h"
#include "genRestrict.h"
#include "genServerInterface.h"
#include "genUtility.h"
#include "truefalse.h"

    // initial number of slots in the permit stack

const int DEFAULT_PERMIT_SLOTS	= 8;

/*
 * UNIMPLEMENTED default constructors and assignment operators
 */

RESTRICT::RESTRICT (const RESTRICT&) 
{
    abort_with_message ("copy constructor for RESTRICT is not supported!");
}

RESTRICT& RESTRICT::operator = (const RESTRICT&) 
{
    abort_with_message ("assignment operator for RESTRICT is not supported!");
    return (*this);
}

RestrictPacker::RestrictPacker (const RestrictPacker&packer):
    m_restrict (packer.m_restrict)
{
    abort_with_message ("copy constructor for RestrictPacker is not supported!");
}

RestrictPacker& RestrictPacker::operator = (const RestrictPacker&) 
{
    abort_with_message ("assignment operator for RestrictPacker is not supported!");
    return (*this);
}

RestrictUnpacker::RestrictUnpacker (const RestrictUnpacker& unpacker):
    m_restrict (unpacker.m_restrict)
{
    abort_with_message ("copy constructor for RestrictUnpacker is not supported!");
}

RestrictUnpacker& RestrictUnpacker::operator = (const RestrictUnpacker&) 
{
    abort_with_message ("assignment operator for RestrictUnpacker is not supported!");
    return (*this);
}

/*
 * STATIC support routines
 */

/* expandPermitArray

   Purpose: Expand an array of PERMIT_SET instances

     Input: array = array of PERMIT_SET instances
		    (PERMIT_SET *)

	    size  = size of the array
		    (UINT_32)

    Output: The procedure doubles the size of the array, setting newSize
            (UINT_32) to the new size, and returning a pointer (PERMIT_SET *)
	    to the new array.
*/

static PERMIT_SET *expandPermitArray (PERMIT_SET *array, UINT_32 size, UINT_32 &newSize)
{
    PERMIT_SET *newArray;

	/*
	 * calculate the new size
	 */

    newSize = size << 1;

	/*
	 * create the new array and copy in the existing permits
	 */

    newArray = new PERMIT_SET [newSize];

    for (UINT_32 i = 0; i < size; ++i) {
	newArray[i] = array[i];
    }

	/*
	 * done
	 */

    return (newArray);	
}

/* PERMIT_SET::PERMIT_SET and PERMIT_SET::operator=

   Purpose: Constructors and assignment operator

     Input: set = the permit set to copy
		  (const PERMIT_SET &)
*/

PERMIT_SET::PERMIT_SET (void):
    trap (NULL),
    duration (),
    threshold (),
    minimum (),
    restrict (NULL)
{
    // empty
}
  
PERMIT_SET::PERMIT_SET (const PERMIT_SET &set):
    trap (set.trap),
    duration (set.duration),
    threshold (set.threshold),
    minimum (set.minimum),
    restrict (set.restrict)
{
    // empty
}

PERMIT_SET& PERMIT_SET::operator= (const PERMIT_SET &set)
{
    trap      = set.trap;
    duration  = set.duration;
    threshold = set.threshold;
    minimum   = set.minimum;
    restrict  = set.restrict;
    return (*this);
}
  
/* PERMIT_SET::thresholdInit

   Purpose: Initialize PERMIT_SET with a known duration and threshold permit
*/

void PERMIT_SET::thresholdInit (int level, RESTRICT *newRestrict, const Permit &newDuration, const Permit &newThreshold, TimerTrap *newTrap)
{
	/* set the duration permit */

    restrict  = newRestrict;
    duration  = newDuration;
    threshold = newThreshold;
    trap      = newTrap;

	/* finish off the initialization */

    finishInit (level);
}

/* PERMIT_SET::durationInit

   Purpose: Initialize PERMIT_SET with a known duration permit
*/

void PERMIT_SET::durationInit (int level, RESTRICT *newRestrict, const Permit &newDuration, TimerTrap *newTrap)
{
	/* set the duration permit */

    restrict  = newRestrict;
    duration  = newDuration;
    threshold = Permit (); 
    trap      = newTrap;
 
	/* set the threshold permit */

    if (duration.haveWallLimit()) {
	threshold.setWallLimit (
	    TimevalUtil::addTimevals (
		SystemUtil::getCurrentWall (), duration.getWallLimit()
	    )
	);
    }

    if (duration.haveCpuLimit()) {
	threshold.setCpuLimit (
	    TimevalUtil::addTimevals (
		SystemUtil::getCurrentCpu (), duration.getCpuLimit()
	    )
	);
    }

    if (duration.haveJumpLimit()) {
	threshold.setJumpLimit (
	    MathUtil::addWithCeiling (
		restrict -> getTotalJumps(), duration.getJumpLimit()
	    )
	);
    }

    if (duration.haveChildrenLimit()) {
	threshold.setChildrenLimit (
	    MathUtil::addWithCeiling (
		restrict -> getTotalChildren(), duration.getChildrenLimit()
	    )
	);
    }

    if (duration.haveDepthLimit()) {
	threshold.setDepthLimit (
	    MathUtil::addWithCeiling (
		restrict -> getChildDepth(), duration.getDepthLimit()
	    )
	);
    }

	/* finish off the initilization */

    finishInit (level);
}

void PERMIT_SET::finishInit (int level)
{
	/* determine the current minimum threshold */

    if (level != 0) {

	minimum.setToMinimum (threshold, restrict -> permitSet[level - 1].minimum);

    } else {

	minimum = threshold;
    }
}   

/* RESTRICT::wallAlarm and RESTRICT::cpuAlarm
 
   Purpose: Invoked when a global or user-defined wall-time or cpu-time limit
            has expired
*/

void RESTRICT::wallAlarm (void)
{
	/* check the global permit and abort on violation */

    checkGlobalPermit();

	/* check the user-defined permits as soon as possible */

    pendingPermitViolation ();
}

void RESTRICT::cpuAlarm (void)
{
	/* check the global permit and abort on violation */

    checkGlobalPermit();

	/* check the user-defined permits as soon as possible */

    pendingPermitViolation ();
}

/* RESTRICT::resetWallTime

   Purpose: Reset the agent's starting wall time to the current time
*/

void RESTRICT::resetWallTime (void) 
{
    startTimeval = SystemUtil::getCurrentWall();
    setGroupStartTimeval (startTimeval);
}

/* RESTRICT::RESTRICT

   Purpose: This procedure is the constructor for class RESTRICT.

      Input: p_agent = agent associated with this RESTRICT structure
		       (class AGENT *)
*/

RESTRICT::RESTRICT (AGENT *p_agent):
    agent (p_agent),
    startTimeval (SystemUtil::getCurrentWall()),
    totalJumps (0),
    totalChildren (0),
    childDepth (0),
    permitLevel (0),
    permitSlots (DEFAULT_PERMIT_SLOTS)
{
	/* base and user-specified permits */

    permitSet = new PERMIT_SET [permitSlots];

	/* resource usage at the time that the agent entered the current     */
	/* TrustedMachine group -- this defaults to the current wall time    */
	/* and to zero jumps, children, depth and CPU time which corresponds */
	/* to a NEW agent that has just come into existence                  */

    struct timeval nulltime;
    nulltime.tv_sec  = 0;
    nulltime.tv_usec = 0;
    setGroupStartTimeval (startTimeval);
    setGroupStartJumps (0);
    setGroupStartChildren (0);
    setGroupStartDepth (0);
    setGroupStartCpu (nulltime);
}

/* RESTRICT::~RESTRICT

   Purpose: This procedure is the destructor for class RESTRICT.
*/

RESTRICT::~RESTRICT ()
{
    timers.removeAllAttachedToTrap (this);
    DELETE_ARRAY_OBJECT(permitSet);
}

/* RESTRICT::checkPermitsInt

   Purpose: See if a permit violation has occurred

     Input: level = permit level at which to start checking
		    (int) 

    Output: The procedure returns a flag set that indicates which permits
	    have been violated.  The flag set is 0 if no permits have been
	    violated.  The PERMIT_WALL bit is set if the wall-time permit has
	    been violated.  The PERMIT_CPU bit is set if the cpu-time bit has 
	    been violated.  We do not check for PERMIT_JUMPS since the jump
	    will never happen if the permit does not allow it.  We do not 
            check for PERMIT_CHILDREN or PERMIT_DEPTH since the child creation
            will never happen if the permit does not allow it.  
*/

int RESTRICT::checkPermitsInt (int level)
{
    int flags = 0;
    struct timeval wall;
    struct timeval cpu;

	/* no permit means no violation */

    if (level < 0) {
	return 0;
    }

	/* check the wall time */

    wall = SystemUtil::getCurrentWall ();

    if (permitSet[level].minimum.haveWallLimit()) {
	if (TimevalUtil::compareTimevals (wall, permitSet[level].minimum.getWallLimit()) >= 0) {
	    flags |= Permit::e_PERMIT_WALL;
	}
    }

	/* check the cpu time */

    cpu  = SystemUtil::getCurrentCpu ();

    if (permitSet[level].minimum.haveCpuLimit()) {
	if (TimevalUtil::compareTimevals (cpu, permitSet[level].minimum.getCpuLimit()) >= 0) {
	    flags |= Permit::e_PERMIT_CPU;
	}
    }

    return (flags);
}

/* RESTRICT::checkPermits

   Purpose: Check for any permit violation

     Input: None

    Output: The procedure returns a flag set that indicates which permits have
	    been violated.  The flag set is 0 if no permits have been violated.
	    The PERMIT_WALL bit is set if the wall-time permit has been
	    violated.  The PERMIT_CPU bit is set if the cpu-time limit has
	    been violated.  We do not check for PERMIT_JUMPS since the jump
	    will never happen if the permit does not allow it.  We do not
	    check for PERMIT_CHILDREN or PERMIT_DEPTH since the child creation
	    will never happen if the permit does not allow it.  
*/

int RESTRICT::checkPermits (void)
{
    return (checkPermitsInt (permitLevel));
}

/* RESTRICT::checkGlobalPermit

   Purpose: Check for a violation of the *global* permit

     Input: None

    Output: The procedure *exits* the agent if a violation has occurred.
*/

void RESTRICT::checkGlobalPermit (void)
{
    int flags;

	/* see which global permits have been violated */

    if ((flags = checkPermitsInt (0)) > 0) {

	fprintf (stderr, "global permit violation: ");

	if (flags & Permit::e_PERMIT_WALL) {
	    fprintf (stderr, "wall-time\n");
	    exit (ServerInterface::e_GLOBAL_WALL_VIOLATION);
	} else if (flags & Permit::e_PERMIT_CPU) {
	    fprintf (stderr, "cpu-time\n");
	    exit (ServerInterface::e_GLOBAL_CPU_VIOLATION);
	} else if (flags & Permit::e_PERMIT_CHILDREN) {
	    fprintf (stderr, "children\n");
	    exit (ServerInterface::e_GLOBAL_CHILDREN_VIOLATION);
	} else if (flags & Permit::e_PERMIT_DEPTH) {
	    fprintf (stderr, "depth\n");
	    exit (ServerInterface::e_GLOBAL_DEPTH_VIOLATION);
	} else if (flags & Permit::e_PERMIT_JUMPS) {
	    fprintf (stderr, "jumps\n");
	    exit (ServerInterface::e_GLOBAL_JUMPS_VIOLATION);
	}

	abort_with_message ("should never get here");   // should never get here
	exit (1);
    }
}

/* RESTRICT::checkCurrentPermit

   Purpose: Check for a violation of the *most recent* permit

     Input: None

    Output: The procedure returns a flag set that indicates which permits
	    have been violated.  The flag set is 0 if no permits have been
	    violated or if the permit violation is due to an older restriction.
	    The PERMIT_WALL bit is set if the most recent wall-time permit has
	    been violated.  The PERMIT_CPU bit is set if the most recent
	    cpu-time permit has been violated.  We do not check for
	    PERMIT_JUMPS since the jump will never happen if the permit does
	    not allow it.  We do not check for PERMIT_CHILDREN or
            PERMIT_DEPTH since the child creation will never happen if the 
	    permit does not allow it.
*/

int RESTRICT::checkCurrentPermit (void)
{
    int flags = 0;
    int nextFlags;

	/* If there is a permit violation, then report it only if the  */
	/* permit violation is occurring at the current level -- i.e., */
	/* only if the permit violation is not occurring at a higher   */
	/* level                                                       */

    flags = checkPermitsInt (permitLevel);

    if (flags != 0) {

	nextFlags = checkPermitsInt (permitLevel - 1);

	if (nextFlags != 0) {
	    flags = 0;
	}
    }

    return (flags);
}

/* RESTRICT::remove

   Purpose: Remove the most recent permit 

     Input: None

    Output: None
*/

void RESTRICT::removePermit (void) 
{
    assert (permitLevel > 0);

    PERMIT_SET& currentSet = permitSet[permitLevel];

    if (currentSet.duration.haveWallLimit() && (currentSet.trap != NULL)) {
	timers.remove (currentSet.minimum.getWallLimit(), currentSet.trap);
    }

    if ((currentSet.duration.haveCpuLimit()) && (currentSet.trap != NULL)) {
	timers.removeCpu (currentSet.minimum.getWallLimit(), currentSet.trap);
    }

    permitLevel -= 1;
}


/* RESTRICT::addTrap 

   Purpose: Add a new timer trap to the current permit set

     Input: trap = the timer trap
		   (class TimerTrap *)

    Output: The timer trap is added to the current list of permits.
*/

void RESTRICT::addTrap (TimerTrap *trap)
{
    Permit permit;

	/* assertions on the parameters */

    assert (trap != NULL);

	/* add the dummy permit to the list of nested permits */

    permitLevel += 1;

    if (permitLevel >= permitSlots) {
	PERMIT_SET *newSet = expandPermitArray (permitSet, permitSlots, permitSlots);
	DELETE_ARRAY_OBJECT(permitSet);
 	permitSet = newSet;
    }	

    permitSet[permitLevel].durationInit (permitLevel, this, permit, trap);

	/* set the wall timer if necessary */

    if (permitSet[permitLevel].minimum.haveWallLimit()) {
        timers.add (permitSet[permitLevel].minimum.getWallLimit(), trap);
    }

	/* set the cpu timer if necessary */

    if (permitSet[permitLevel].minimum.haveCpuLimit()) {
	timers.addCpu (permitSet[permitLevel].minimum.getCpuLimit(), trap);
    }
}

/* RESTRICT::getBase

   Purpose: Get the current base permit

     Input: None

    Output: The procedure returns the current base permit.
*/

Permit RESTRICT::getBase (void) const  
{
    return (permitSet[0].duration);
}

/* RESTRICT::addBase

   Purpose: Add a new base permit

     Input: permit = the base permit
		     (class Permit &)

   	    trap   = TimerTrap instance if we need an asynchronous interrupt
		     and NULL otherwise
		     (class TimerTrap *)

    Output: The procedure adds the new base permit.
*/ 

void RESTRICT::addBase (const Permit &permit)
{
	/* get the global permits and remember the trap */

    PERMIT_SET &globalPermits = permitSet[0];
    globalPermits.trap = this;

	/* merge in the new duration permits */

    globalPermits.duration.merge (permit);

	/* calculate the new threshold permits */

    globalPermits.threshold = globalPermits.duration;

    if (globalPermits.threshold.haveWallLimit()) {
        globalPermits.threshold.setWallLimit (
	    TimevalUtil::addTimevals (
		globalPermits.threshold.getWallLimit(), startTimeval
	    )
	);
    }

	/* new minimum permits are just the new threshold permits */ 

    globalPermits.minimum = globalPermits.threshold;

	/* add the time handlers */

    if (globalPermits.minimum.haveWallLimit()) {
	timers.add (globalPermits.minimum.getWallLimit(), this);
    }

    if (globalPermits.minimum.haveCpuLimit()) {
	timers.addCpu (globalPermits.minimum.getCpuLimit(), this);
    }

	/* now recalculate the minimums all the way down -- note that we   */
	/* don't actually have to set any more timers since if one of the  */
	/* minimums changes, the only possible new value is the value from */
	/* the base permit, for which we have already set a timer above    */

    for (UINT_32 i = 1; i <= permitLevel; ++i) {
	permitSet[i].minimum.setToMinimum (permitSet[i].threshold, permitSet[i-1].minimum);
    }
}

/* RESTRICT::addPermit

   Purpose: Add a new permit 

     Input: permit = the permit
		     (class Permit &)

	    trap    = TimerTrap instance if we need an asynchonous interrupt
		      and NULL otherwise
		      (class TimerTrap *)
			
    Output: The procedure adds the new permit.
*/

void RESTRICT::add (Permit &duration, Permit &thresh, TimerTrap *trap)
{
	/* add the permit to the list of nested permits */

    permitLevel += 1;

    if (permitLevel >= permitSlots) {
	PERMIT_SET *newSet = expandPermitArray (permitSet, permitSlots, permitSlots);
	DELETE_ARRAY_OBJECT(permitSet);
 	permitSet = newSet;
    }

    permitSet[permitLevel].thresholdInit (permitLevel, this, duration, thresh, trap);

	/* set timers if needed */

    if (trap != NULL) {

	if (duration.haveWallLimit()) {
	    timers.add (permitSet[permitLevel].minimum.getWallLimit(), trap);
	}

	if (duration.haveCpuLimit()) {
	    timers.addCpu (permitSet[permitLevel].minimum.getCpuLimit(), trap);
	}
    }
}

void RESTRICT::addPermit (Permit &permit, TimerTrap *trap)
{
	/* add the permit to the list of nested permits */

    permitLevel += 1;

    if (permitLevel >= permitSlots) {
	PERMIT_SET *newSet = expandPermitArray (permitSet, permitSlots, permitSlots);
	DELETE_ARRAY_OBJECT(permitSet);
 	permitSet = newSet;
    }

    permitSet[permitLevel].durationInit (permitLevel, this, permit, trap);

	/* set timers if needed */

    if (trap != NULL) {

	if (permit.haveWallLimit()) {
	    timers.add (permitSet[permitLevel].minimum.getWallLimit(), trap);
	}

	if (permit.haveCpuLimit()) {
	    timers.addCpu (permitSet[permitLevel].minimum.getCpuLimit(), trap);
	}
    }
}

/* RESTRICT::getPermits

   Purpose: Get the permits at a particular level

     Input: level = desired level
		    (int)

    Output: The procedure returns a pointer to a dynamically allocated
            PERMIT_SET structure that contains the permits from the desired
	    level.
*/

PERMIT_SET *RESTRICT::getPermits (UINT_32 level)
{
    assert (level <= permitLevel);
    return (new PERMIT_SET (permitSet[level]));
}

/* RestrictPacker::getPackedSize

   Purpose: Determine the number of bytes needed to represent the restrictions
	    as a byte stream 
*/

UINT_32 RestrictPacker::getPackedSize (void)
{
    UINT_32 size = 0; 

	// add in the resource usages so far

    if ((m_type == ServerInterface::e_SUB_JUMP) |
	(m_type == ServerInterface::e_SUB_AGLET_JUMP)) {

	struct timeval dummy;
	TimevalPacker tpacker (dummy);

	size += sizeof(UINT_32);		// parent-child depth
	size += tpacker.getPackedSize();	// starting wall time
	size += tpacker.getPackedSize();	// elapsed CPU time
	size += sizeof(UINT_32);		// jumps so far
	size += sizeof(UINT_32);		// children so far

	PermitPacker packer (m_restrict.groupStart);
	size += packer.getPackedSize ();

    } else {

	size += sizeof(UINT_32);	// parent-child depth
	size += sizeof(UINT_32);	// parent-child depth (within the group)
    }

	// if we're Aglet-jumping or jumping, add in size of base permit; 
	// if we are forking, submitting or Aglet-submitting, add in just the
	// base depth limit
    
    if ((m_type == ServerInterface::e_SUB_AGLET_JUMP) ||
	(m_type == ServerInterface::e_SUB_JUMP)) {
      
	PermitPacker dpacker (m_restrict.permitSet[0].duration);
	size += dpacker.getPackedSize ();

    } else {

	Permit permit;	
	Permit basePermit = m_restrict.getBase();

	if (basePermit.haveDepthLimit()) {
	    permit.setDepthLimit (basePermit.getDepthLimit());
	}

	PermitPacker dpacker (permit);
	size += dpacker.getPackedSize();
    } 

	// if we are jumping or forking, add in the user-defined permits

    if ((m_type == ServerInterface::e_SUB_FORK) ||
	(m_type == ServerInterface::e_SUB_JUMP)) {
 
	size += sizeof (UINT_32);	// permit count

	for (UINT_32 i = 1; i <= m_restrict.permitLevel; ++i) {
	    if (m_restrict.permitSet[i].trap == &m_restrict) {
		PermitPacker dpacker (m_restrict.permitSet[i].duration);
		size += dpacker.getPackedSize ();
		PermitPacker tpacker (m_restrict.permitSet[i].threshold);
		size += tpacker.getPackedSize ();
	    }
	}
    }
	
    return (size);
}

/* RestrictPacker::pack

   Purpose: Pack the restrictions into a byte stram

     Input: buffer = character buffer in which to place the byte stream
		     (char *)

    Output: The procedure returns a char * that points into the buffer
            immediately after the newly added bytes.
*/

char *RestrictPacker::pack (char *bp)
{
    UINT_32 netLong;

	// capture resource usages so far

    if ((m_type == ServerInterface::e_SUB_JUMP) ||
	(m_type == ServerInterface::e_SUB_AGLET_JUMP)) {

        netLong = htonl (m_restrict.childDepth);
        bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	TimevalPacker wpacker (m_restrict.startTimeval);
	bp = wpacker.pack (bp);

	struct timeval elapsedCpuTimeval = SystemUtil::getCurrentCpu ();
	TimevalPacker cpacker (elapsedCpuTimeval);
	bp = cpacker.pack (bp);

	netLong = htonl (m_restrict.totalJumps);
	bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	netLong = htonl (m_restrict.totalChildren);
	bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	PermitPacker packer (m_restrict.groupStart);
	bp = packer.pack (bp);
	
    } else {

        netLong = htonl (m_restrict.childDepth);
        bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	netLong = htonl (m_restrict.getGroupStartDepth());
        bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

#ifdef TAKE_OUT

	Permit permit;
	Permit basePermit = m_restrict.getBase();

	if (basePermit.haveDepthLimit()) {
	    permit.setDepthLimit (basePermit.getDepthLimit());
	}

	PermitPacker packer (permit);
	bp = packer.pack (bp);

#endif
    }

	// if we're Aglet-jumping or jumping, capture base permit; if we are 
	// submitting, Aglet-submitting or forking, capture just the depth
	// part of the base permit

    if ((m_type == ServerInterface::e_SUB_AGLET_JUMP) || 
	(m_type == ServerInterface::e_SUB_JUMP)) {
     
	PermitPacker dpacker (m_restrict.permitSet[0].duration);
	bp = dpacker.pack (bp);

    } else {

	Permit permit;
	Permit basePermit = m_restrict.getBase();

	if (basePermit.haveDepthLimit()) {
	    permit.setDepthLimit (basePermit.getDepthLimit());
	}

	PermitPacker packer (permit);
	bp = packer.pack (bp);
    }

	// if we're jumping or forking, capture the user-defined permits

    if ((m_type == ServerInterface::e_SUB_JUMP) || 
	(m_type == ServerInterface::e_SUB_FORK)) {

	UINT_32 i;
	UINT_32 count = 0;

	for (i = 1; i <= m_restrict.permitLevel; ++i) {
	    if (m_restrict.permitSet[i].trap == &m_restrict) {
		count += 1;
	    }
	}
	
	netLong = htonl (count);
	bp = fast_memcpy (bp, (char *) &netLong, sizeof(UINT_32));

	for (i = 1; i <= m_restrict.permitLevel; ++i) {
	    if (m_restrict.permitSet[i].trap == &m_restrict) {
		PermitPacker dpacker (m_restrict.permitSet[i].duration);
		bp = dpacker.pack (bp);
		PermitPacker tpacker (m_restrict.permitSet[i].threshold);
		bp = tpacker.pack (bp);
	    }
	}
    }

    return (bp);
}

/* RestrictUnpacker::unpack

   Purpose: Unpack restrictions from a byte stream

     Input: bp     = start of the buffer containing the restrictions
		     (char *)

	    end_bp = end of the buffer containing the restrictions
		     (char *)

    Output: The procedure returns NULL on error.  Otherwise the procedure
	    returns a pointer to the buffer position immediately after the
	    restrictions.
*/

char *RestrictUnpacker::unpack (char *bp, char *end_bp)
{
    UINT_32 hostLong;
    
	// break out the resource usages so far

    if ((m_type == ServerInterface::e_SUB_JUMP) ||
	(m_type == ServerInterface::e_SUB_AGLET_JUMP)) {

	    // unpack the current resource usages
	    // child depth

	UINT_32 childDepth;

	if ((bp = breakout_long (bp, end_bp, childDepth)) == NULL) {
	    return (NULL);
	}

	    // starting wall time

	struct timeval startWallTimeval;
	TimevalUnpacker wunpacker (startWallTimeval);

	if ((bp = wunpacker.unpack (bp, end_bp)) == NULL) {
	    return (NULL);
	}

	    // elapsed CPU time

	struct timeval elapsedCpuTimeval;
	TimevalUnpacker cunpacker (elapsedCpuTimeval);

	if ((bp = cunpacker.unpack (bp, end_bp)) == NULL) {
	    return (NULL);
	}

	    // jumps so far

	UINT_32 totalJumps;

	if ((bp = breakout_long (bp, end_bp, totalJumps)) == NULL) {
	    return (NULL);
	}

	    // children so far

	UINT_32 totalChildren;

	if ((bp = breakout_long (bp, end_bp, totalChildren)) == NULL) {
	    return (NULL);
	}

	    // unpack the group entry permit and make sure that all the
	    // components are present

	Permit group;
	PermitUnpacker unpacker (group);

	if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
	    return (NULL);
	}

	if (!group.haveAll()) {
	    return (NULL);
	}

	    // sanity check the resource usage so far, e.g., the time at
	    // at which the agent began its life should be less than (or equal
	    // to) the time at which it entered the current machine group,
	    // which in turn should be less than (or equal to) the current time
	
	    // sanity check the wall time - If the starting wall time or the
	    // group entry time is bigger than the current time, we reset the 
	    // start/entry time to the current time, rather than aborting the 
	    // agent.  We do this because the machine clocks might not be 
	    // synchronized, so a mismatch between the start/entry times and 
	    // the current times does *not* indicate a malformed state image.

	struct timeval currentWall = SystemUtil::getCurrentWall();

	if (!(TimevalUtil::compareTimevals (startWallTimeval, group.getWallLimit()) <= 0)) {
	    return (NULL);
	}

	if (!(TimevalUtil::compareTimevals (startWallTimeval, currentWall) <= 0)) {
	    startWallTimeval = currentWall;
	}

	if (!(TimevalUtil::compareTimevals (group.getWallLimit(), currentWall) <= 0)) {
	    group.setWallLimit (currentWall);
	}

	    // sanity check the CPU time

	if (!(TimevalUtil::compareTimevals (group.getCpuLimit(), elapsedCpuTimeval) <= 0)) {
	    return (NULL);
	}

	    // sanity check the parent-child depth

	if (!(group.getDepthLimit() <= childDepth)) {
	    return (NULL);
	}

	    // sanity check the number of jumps

	if (!(group.getJumpLimit() <= totalJumps)) {
	    return (NULL);
	}

	    // sanity check the number of children

	if (!(group.getChildrenLimit() <= totalChildren)) {
	    return (NULL);
	}

	    // remember the resource usage so far

	m_restrict.totalChildren       = totalChildren;
	m_restrict.totalJumps          = totalJumps;
	m_restrict.childDepth          = childDepth;
	m_restrict.setStartTimeval     (startWallTimeval);
	SystemUtil::setElapsedCpu      (elapsedCpuTimeval);
	m_restrict.setGroupStartPermit (group);

    } else {

	if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	    return (NULL);
	}

	m_restrict.childDepth = hostLong;

	if ((bp = breakout_long (bp, end_bp, hostLong)) == NULL) {
	    return (NULL);
	}

	m_restrict.setGroupStartDepth (hostLong);
    }

	// if we're Aglet jumping or jumping, break out the base permit;
	// if we're submitting, Aglet-submitting or forking, break out just
	// the depth piece of the base permit

    if ((m_type == ServerInterface::e_SUB_AGLET_JUMP) || 
	(m_type == ServerInterface::e_SUB_JUMP)) {

	Permit duration;
	PermitUnpacker dunpacker (duration);
      
	if ((bp = dunpacker.unpack (bp, end_bp)) == NULL) {
	    return (NULL);
	}
     
	m_restrict.RESTRICT::addBase (duration);

    } else {

	Permit permit;
	PermitUnpacker unpacker (permit);

	if ((bp = unpacker.unpack (bp, end_bp)) == NULL) {
	    return (NULL);
	}

	if (permit.haveWallLimit() || permit.haveCpuLimit() || permit.haveChildrenLimit() || permit.haveJumpLimit()) {
	    return (NULL);
	}

	m_restrict.RESTRICT::addBase (permit);
    }

	// break out the user-specified permits if we're jumping or forking

    if ((m_type == ServerInterface::e_SUB_JUMP) || 
	(m_type == ServerInterface::e_SUB_FORK)) {

	UINT_32 i = 0;
	UINT_32 count = 0;
	
	if ((bp = breakout_long (bp, end_bp, count)) == NULL) {
	    return NULL;
	}

	for (i = 0; i < count; i++) {

	    Permit duration;
	    Permit threshold;

	    PermitUnpacker dunpacker (duration);
	    PermitUnpacker tunpacker (threshold);

	    if ((bp = dunpacker.unpack (bp, end_bp)) == NULL) {
		return (NULL);
	    }

	    if ((bp = tunpacker.unpack (bp, end_bp)) == NULL) {
		return (NULL);
	    }

	    m_restrict.RESTRICT::add (duration, threshold, &m_restrict);
	}
    }

    return (bp);
}
