/* Agent Tcl
   Bob Gray
   10 January 1996

   genRestrict.h

   This file defines the permit routines and data structures. 

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _GEN_RESTRICT
#define _GEN_RESTRICT

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _AGENT_PERMIT_H
#include "agentPermit.h"		// Permit
#endif
#ifndef _PLAT_MATH
#include "platMath.h"			// MathUtil
#endif
#ifndef _PLAT_TIMERS_H
#include "platTimers.h"			// TimerTrap
#endif
#ifndef _GEN_SERVER_FLAGS_H
#include "genServerInterface.h"		// SubmissionTypes
#endif

/* ----------------------------------------------------------------------- */

#include "platPorting.h"
#include "agentId.h"
#include "genUtility.h"
#include "platSystem.h"
#include "platTimers.h"
#include "platTimeval.h"

const int INFINITY_SECONDS = 31557600;

    /* a set of permits */

class RESTRICT;

struct PERMIT_SET
{
private:

	/* initialization helper */

    void finishInit (int level);

public:

	/* the trap associated with this restriction */

    TimerTrap *trap;

	/* amount, threshold and minimum threshold */

    Permit duration;
    Permit threshold;
    Permit minimum;

	/* backpointer to the RESTRICT class */

    RESTRICT *restrict;

	/* constructors and destructor */

    PERMIT_SET (void);
    PERMIT_SET (const PERMIT_SET &set);
   ~PERMIT_SET () {};

	/* assignment operator */

    PERMIT_SET& operator= (const PERMIT_SET &set);

	/* finish off initialization */

    void thresholdInit
	(int level, RESTRICT *restrict, const Permit &duration, const Permit &thresh, TimerTrap *trap);

    void durationInit
	(int level, RESTRICT *restrict, const Permit &duration, TimerTrap *trap);
}; 

    /* class RESTRICT keeps track of the active "restrict" commands */

class RESTRICT: public TimerTrap
{
    friend class AGENT;
    friend class PERMIT_SET;
    friend class RestrictPacker;
    friend class RestrictUnpacker;

private:

	/*
	 * unimplemented
	 */

    RESTRICT (const RESTRICT &);
    RESTRICT & RESTRICT::operator= (const RESTRICT&);

	/*
	 * wall-time limit or cpu-time limit for a base or user-defined
	 * permit has expired
	 */

    void wallAlarm (void);
    void cpuAlarm  (void);

	/* 
	 * reset the starting wall time (called from AGENT)
	 */

    void resetWallTime (void);

protected:

	/* agent associated with this RESTRICT structure */

    AGENT *agent;

#ifdef FIX_LATER
	/* the following four variables plus CPU time should possibly */
	/* be in a Permit instance				      */
#endif

	/* timeval at which the agent started execution */

    struct timeval startTimeval;

	/* total number of migrations that the agent has performed so far */

    UINT_32 totalJumps;

	/* total number of children that the agent has created so far */

    UINT_32 totalChildren;

	/* depth of this agent within the parent-child hierarchy */

    UINT_32 childDepth;
 
	/* resource usages at the time that the agent started executing */
	/* in the current set of TrustedMachine's -- we keep track of   */
	/* these so that we can enforce restrictions on total resource  */
	/* usage within the group                                       */

    Permit groupStart;
 
	/* timers for wall and cpu time */

    Timers timers;

	/* current set of permits and total number of slots in the set */

    UINT_32 permitLevel;
    UINT_32 permitSlots;
    PERMIT_SET *permitSet;

	/* check for a permit violation */

    int checkPermitsInt (int level);

	/* add a new permit (with a known threshold) */

    void add (Permit &duration, Permit &thresh, TimerTrap *trap);

	/* constructor and destructor */

    RESTRICT (AGENT *agent);
    virtual ~RESTRICT (); 

public:

	/*
	 * get and set the group-start resource usages
	 */

    Permit getGroupStartPermit (void) {
	return (groupStart);
    }

    void setGroupStartPermit (const Permit& permit) {
	groupStart = permit;
    }

	/* 
         * get and set the time at which the agent started (overall and 
	 * in the current TrustedMachine group)
	 */

    struct timeval getStartTimeval (void) {
	return (startTimeval);
    } 

    void setStartTimeval (struct timeval start) {

	struct timeval current = SystemUtil::getCurrentWall ();

	if (TimevalUtil::compareTimevals (current, start) < 0) {
	    startTimeval = current;
	} else {
	    startTimeval = start;
	}
    } 

    struct timeval getGroupStartTimeval (void) {
	return (groupStart.getWallLimit());
    }

    void setGroupStartTimeval (struct timeval start) {
	groupStart.setWallLimit (start);
    }

	/*
	 * get, set and increment the number of jumps so far AND get and set 
	 * the number of jumps at the time that the agent entered the current 
	 * TrustedMachine group
	 */

    UINT_32 getTotalJumps (void) {
	return (totalJumps);
    }

    void setTotalJumps (UINT_32 p_jumps) {
	totalJumps = p_jumps;
    }

    UINT_32 incrementTotalJumps (void) {
	totalJumps = MathUtil::addWithCeiling (totalJumps, (UINT_32) 1);
	return (totalJumps);
    }

    UINT_32 getGroupStartJumps (void) {
	return (groupStart.getJumpLimit());
    }

    void setGroupStartJumps (UINT_32 p_jumps) {
	groupStart.setJumpLimit(p_jumps);
    }

	/* 
	 * get, set and increment the number of children created so far AND
	 * get and set the number of children that had been created at the time
	 * that the agent entered the current TrustedMachine group
	 */

    UINT_32 getTotalChildren (void) {
	return (totalChildren);
    }

    void setTotalChildren (UINT_32 p_children) {
	totalChildren = p_children;
    }

    UINT_32 incrementTotalChildren (void) {
	totalChildren = MathUtil::addWithCeiling (totalChildren, (UINT_32) 1);
	return (totalChildren);
    }

    UINT_32 getGroupStartChildren (void) {
	return (groupStart.getChildrenLimit());
    }

    void setGroupStartChildren (UINT_32 p_children) {
	groupStart.setChildrenLimit(p_children);
    }

	/* 
	 * get, set and increment the depth of this agent in the parent-child
	 * hierarchy AND get and set the depth of the parent agent that first
	 * entered the current TrustedMachine group
	 */

    UINT_32 getChildDepth (void) {
	return (childDepth);
    }

    void setChildDepth (UINT_32 p_depth) {
	childDepth = p_depth;
    }

    UINT_32 incrementChildDepth (void) {
	childDepth = MathUtil::addWithCeiling (childDepth, (UINT_32) 1);
	return (childDepth);
    }

    UINT_32 getGroupStartDepth (void) {
	return (groupStart.getDepthLimit());
    }

    void setGroupStartDepth (UINT_32 p_depth) {
	groupStart.setDepthLimit(p_depth);
    }

	/*
	 * get and set the amount of CPU time that the agent had used at
	 * the time that it entered the current TrustedMachine group
	 */

    struct timeval getGroupStartCpu (void) {
	return (groupStart.getCpuLimit());
    }

    void setGroupStartCpu (struct timeval start) {
	groupStart.setCpuLimit(start);
    }

	/* 
	 * get the associated AGENT instance
	 */

    AGENT *getAgent (void) { 
	return (agent);
    }

	/* 
	 * get the number of permits
	 */

    UINT_32 getLevel (void) {
	return (permitLevel);
    }  

	/* 
	 * get the permits at the given level 
	 */

    PERMIT_SET *getPermits (UINT_32 level);

	/* 
	 * get the current wall time limit 
	 */

    int getWallLimit (struct timeval &wall) { 

	int code;

	if (!(permitSet[permitLevel].minimum.haveWallLimit())) {
	    code = -1;
	} else {
	    wall = permitSet[permitLevel].minimum.getWallLimit ();
	    code = 0;
	}

	return (code);
    } 

	/* 
	 * get the current CPU time limit
	 */

    int getCpuLimit (struct timeval &cpu) { 

	int code;

	if (!(permitSet[permitLevel].minimum.haveCpuLimit())) {
	    code = -1;
	} else {
	    cpu = permitSet[permitLevel].minimum.getCpuLimit ();
	    code = 0;
	}

	return (code);
    } 

	/* 
	 * see if we can jump to a new machine
	 */

    BOOLEAN isJumpAllowed (void) {

	BOOLEAN allowed;

	if (!(permitSet[permitLevel].minimum.haveJumpLimit())) {

	    allowed = e_TRUE;

	} else {

	    allowed = (totalJumps < permitSet[permitLevel].minimum.getJumpLimit()) ? e_TRUE : e_FALSE;
	}

	return (allowed);
    }

	/* 
	 * see if we can create a child agent
	 */

    BOOLEAN isChildAllowed (void) {

	BOOLEAN childAllowed = e_FALSE;
	BOOLEAN depthAllowed = e_FALSE;

	if (!(permitSet[permitLevel].minimum.haveChildrenLimit())) {
	    childAllowed = e_TRUE;
	} else if (totalChildren < permitSet[permitLevel].minimum.getChildrenLimit()) {
	    childAllowed = e_TRUE;
	}


	if (!(permitSet[permitLevel].minimum.haveDepthLimit())) {
	    depthAllowed = e_TRUE;
	} else if (childDepth < permitSet[permitLevel].minimum.getDepthLimit()) {
	    depthAllowed = e_TRUE;
	} 

	return ((childAllowed && depthAllowed) ? e_TRUE : e_FALSE);
    }

	/* 
	 * check for permit violations at the current level, at the global
	 * level or at all levels
	 */

    int  checkCurrentPermit (void);
    void checkGlobalPermit (void);
    int  checkPermits (void);

	/* 
	 * add a new permit or add a new handler to an existing permit
	 */

    void addTrap (TimerTrap *trap);

    void addPermit (Permit &permit, TimerTrap *trap);

    virtual void addUserPermit (Permit &permit) {
	addPermit (permit, this);
    }

	/* 
	 * remove the most recent permit 
	 */

    void removePermit (void);

    virtual void removeUserPermit (void) {
	removePermit ();
    }

	/*
	 * change the base permit or get the current base permit
	 */

    void addBase (const Permit &permit);
    Permit getBase (void) const;

	/*
	 * callback when there is a potential user-defined permit violation 
	 */

    virtual void pendingPermitViolation (void) = 0;
};

/*
 * Iterators that pack and unpack the current restrictions -- we use iterators 
 * since eventually we want to be able to pack the restrictions in fixed-size 
 * chunks so that we can overlap the packing and transmission process
 */

class RestrictPacker
{
    RESTRICT &m_restrict;	               
	// restrictions that we are packing into a byte stream
    ServerInterface::SubmissionTypes m_type;   
	// e_SUB_JUMP, e_SUB_SCRIPT, e_SUB_FORK, e_AGLET_JUMP or e_AGLET_SUBMIT

	// unimplemented

    RestrictPacker (const RestrictPacker&);
    RestrictPacker& operator= (const RestrictPacker&);

public:

	// constructor and destructor

    RestrictPacker (RESTRICT &restrict, ServerInterface::SubmissionTypes type):
	m_restrict (restrict),
	m_type (type)
    {
	// empty
    }
	
   ~RestrictPacker() {
	// empty
    }

	// get the packed size of the permit 

    UINT_32 getPackedSize (void);

	// pack the permit into the buffer 

    char *pack (char *buffer);
};

class RestrictUnpacker
{
    RESTRICT &m_restrict;	          
	// restrictions that we are unpacking from a byte stream
    ServerInterface::SubmissionTypes m_type;
	// e_SUB_JUMP, e_SUB_SCRIPT, e_SUB_FORK, e_AGLET_JUMP or e_AGLET_SUBMIT

	// unimplemented

    RestrictUnpacker (const RestrictUnpacker&);
    RestrictUnpacker& operator= (const RestrictUnpacker&);

public:

	// constructor and destructor

    RestrictUnpacker (RESTRICT &restrict, ServerInterface::SubmissionTypes type):
	m_restrict (restrict),
	m_type (type)
    {
	// empty
    }

   ~RestrictUnpacker () {
	// empty
    }

	// unpack from a buffer

    char *unpack (char *buffer, char *endBuffer);
};


#endif
