/* Agent Tcl
   Bob Gray
   17 April 1996 
   
   genSecurity.cc
  
   This file implements the data structures that keep track of the current
   security parameters.

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "suppStrings.h"
#include "mesgMachId.h"        	// MachineId
#include "suppDString.h"			// DynamicString
#include "genServerInterface.h"		// ServerInterface::SubmissionTypes


/* ---------------------------------------------- */

#include "genAgent.h"
#include "genDefaults.h"
#include "genEncrypt.h"
#include "genFile.h"
#include "genManager.h"
#include "genSecurity.h"

/* SECURITY::SECURITY

   Purpose: This procedure is the constructor for class SECURITY.

     Input: agent = agent associated with this instance
		    (class AGENT *)
*/

SECURITY::SECURITY (AGENT *p_agent):

	agent		(p_agent),
	directory	((char *) NULL),
	available	(e_FALSE),
	availableCode	(SECURITY_NOT_INITIALIZED)

{
	/* empty */
}

/* SECURITY::startingBootstrap

   Purpose: Bootstrap the security system for agents that are *starting* on
	    the local machine -- e.g., access the local filesystem in order
	    to determine the identity of the agent's owner as well as the
	    relevant encryption keys

     Input: None

    Output: The procedure returns one of the following codes.

	    SECURITY_OK    = security has been bootstrapped
	

      Note: This routine -- along with all the security-related data
	    structures -- remain somewhat PGP dependent.  This will change
	    hopefully.
*/

static int getIdentityInfo (DynamicString filename, char *buffer, int size)
{
    int last;
    FILE *fp;

	/* assertions on the parameters */

    assert (buffer != NULL);
    assert (size > 0);

	/* open the file and read the line */

    if ((fp = fopen (filename.value(), "r")) == NULL) {
	return -1;
    }

    if (fgets (buffer, size, fp) == NULL) {
	fclose (fp);
	return -1;
    }

    fclose (fp);

	/* remove the trailing newline */

    last = strlen(buffer) - 1;

    if (buffer[last] == '\n') {
	buffer[last] = '\0';
    }

    return 0;
}

int SECURITY::startingBootstrap (void) 
{
    const char *dir;
    char *temp;
    char *phrase;
    char *keyname;
    DynamicString filename;
#define BUFFER_SIZE 512
    char buffer[BUFFER_SIZE + 1];

	/* get the name of the directory that contains security data */

    if ((temp = getenv (AGENT_DIRECTORY_ENV)) != NULL) {
	dir = temp;
    } else {
	dir = DEFAULT_AGENT_DIRECTORY;
    }    

	/* expand the directory name -- i.e., replace any tilde (~) characters */

    if ((directory = FileUtility::fileTildeSubst (dir)) == NULL) {
	directory  = strcpyWithAlloc (dir);
	availableCode = SECURITY_NO_DIRECTORY;
	return (availableCode);
    } 

	/* read the user identification */

    if ((keyname = getenv (AGENT_KEY_ENV)) != NULL) {

	m_owner.m_keyname = keyname;

	    /*
	     * unset the environment variable once we read it (so that an
	     * attacker can not obtain the key just by examining the 
	     * environment)
	     */
	     
	unsetenv ((char *) AGENT_KEY_ENV);
 
    } else {

	filename.empty ();
	filename.append (directory);
	filename.append ("/keyname");

	FileUtility::FileInformation stats;

	if (FileUtility::getFileStats (filename.value(), stats) != FileUtility::e_OK) {
	    availableCode = SECURITY_NO_ID;
	    return (availableCode);
	}

	if (FileUtility::isGroupAccessible(stats.permissions) || 
	    FileUtility::isOtherAccessible(stats.permissions)) {
	    fprintf (stderr, "\nFATAL ERROR: keyname file \"%s\" should be owner-readable/writeable ONLY: please check the contents of the file and change the permissions\n\n", filename.value());
	    exit (1);
	}

	if (getIdentityInfo (filename, buffer, BUFFER_SIZE) < 0) {
	    fprintf (stderr, "\nFATAL ERROR: keyname file \"%s\" exists but has incorrect formant: please contact your Agent Tcl administrator\n\n", filename.value());
	    exit (1);
	}

	m_owner.m_keyname = buffer;
    }

	/* read the user pass phrase */

    if ((phrase = getenv (AGENT_PASS_ENV)) != NULL) {

	m_owner.m_phrase = phrase;

	    /*
	     * unset the environment variable once we read it (so that an
	     * attacker can not obtain the pass phrase just by examining the 
	     * environment)
	     */
	     
	unsetenv ((char *) AGENT_PASS_ENV);

    } else {

	filename.empty ();
	filename.append (directory);
	filename.append ("/passphrase");

	FileUtility::FileInformation stats;

	if (FileUtility::getFileStats (filename.value(), stats) != FileUtility::e_OK) {
	    availableCode = SECURITY_NO_PASS;
	    return (availableCode);
	}

	if (FileUtility::isGroupAccessible(stats.permissions) || 
	    FileUtility::isOtherAccessible(stats.permissions)) {
	    fprintf (stderr, "\nFATAL ERROR: pass phrase file \"%s\" should be owner-readable/writeable ONLY: assume that your pass phrase has been compromised and contact your Agent Tcl administrator\n\n", filename.value());
	    exit (1);
	}

	if (getIdentityInfo (filename, buffer, BUFFER_SIZE) < 0) {
	    fprintf (stderr, "\nFATAL ERROR: pass phrase file \"%s\" exists but has incorrect formant: please contact your Agent Tcl administrator\n\n", filename.value());
	    exit (1);
	}

	m_owner.m_phrase = buffer;
    }

    m_owner.m_authenticated = e_TRUE;
    m_ownerSigned = e_TRUE;

	/* success */

    availableCode = SECURITY_OK;
    available = e_TRUE;
    return (availableCode);
} 

/* SECURITY::arrivingBootstrap

   Purpose: Bootstrap the security system for agents that are *arriving*
	    from another machine

     Input: sender = security information sent from the server
		     (const BASE_SECURITY &)

    Output: The procedure returns one of the following codes.

	    SECURITY_OK    = security has been bootstrapped
*/

int SECURITY::arrivingBootstrap (ServerInterface::SubmissionTypes /*type*/, const BASE_SECURITY& sender)
{
    * (BASE_SECURITY *) this = sender;
    availableCode = SECURITY_OK;
    available = e_TRUE;
    return (SECURITY_OK);
}

/* SECURITY::recordServerKeyname

   Purpose: Record the keyname of the agent's new server

     Input: id = new local identification
		 (const AgentId &)

    Output: The procedure records the server keyname and the agent id.
*/

void SECURITY::recordServerKeyname (const AgentId &id)
{
	/* construct the keyname */

    MachineId machineId = MachineId (id.getServerName(), id.getServerIp());
    SmartCharPtr keyname (makeFullServerKeyname (machineId));

	/* remember the machine and the agent id */

    m_machine            = SEC_IDENTITY ((char *) keyname, (char *) NULL, e_FALSE, e_FALSE);
    m_agentAuthenticated = e_TRUE;
    m_localId            = id.getId();
}

/* SECURITY::throwServerKeyname

   Purpose: Discard the server keyname when the agent is no longer registered

     Input: None 

    Output: The procedure discards the server keyname.
*/

void SECURITY::throwServerKeyname (void)
{ 
    m_machine            = SEC_IDENTITY ((char *) NULL, (char *) NULL, e_FALSE, e_FALSE);
    m_agentAuthenticated = e_FALSE;
}

/* SECURITY::setEncryption

   Purpose: Turn on or off encryption
 
     Input: p_encryption = e_TRUE to turn on encryption and e_FALSE to turn off
                           encryption
                           (BOOLEAN)

    Output: When turning on encryption, the procedure returns a security error
            code if the encryption subsystem is not available.  Otherwise the
	    procedure returns SECURITY_OK.  When turning off encryption (which 
	    always succeeds), the procedure returns SECURITY_OK.
*/

int SECURITY::setEncryption (BOOLEAN p_encryption) {

    int code;

    if ((available == e_FALSE) && (p_encryption == e_TRUE)) {
	code = availableCode;
    } else {
	(void) BASE_SECURITY::setEncryption (p_encryption);
	code = SECURITY_OK;
    }

    return (code);
} 

/* SECURITY::setSignatures

   Purpose: Turn on or off signatures
 
     Input: p_signatures = e_TRUE to turn on signatures and e_FALSE to turn off
    			   signatures 
                           (BOOLEAN)

    Output: When turning on signatures, the procedure returns a security error
            code if the encryption subsystem is not available.  Otherwise the
	    procedure returns SECURITY_OK.  When turning off signatures (which 
	    always succeeds), the procedure returns SECURITY_OK.
*/

int SECURITY::setSignatures (BOOLEAN p_signatures) {

    int code;

    if ((available == e_FALSE) && (p_signatures == e_TRUE)) {
	code = availableCode;
    } else {
	(void) BASE_SECURITY::setSignatures (p_signatures);
	code = SECURITY_OK;
    }

    return (code); 
}

/* SECURITY::securityCodeToShortString

   Purpose: Convert a security error code into a short string

     Input: code = the error code
		   (int)

    Output: The procedure returns a DynamicString that contains the short
            string. 
*/

static char *error[] = {
	"OK",
	"NOT_INITIALIZED",
	"NO_DIRECTORY",
	"NO_PASS_PHRASE",
	"NO_IDENTITY"
};

DynamicString SECURITY::securityCodeToShortString (int code)
{
    DynamicString string;

    if ((code >= SECURITY_BEG_GENERAL) && (code <= SECURITY_END_GENERAL)) {
	string.append (error[code - SECURITY_BEG_GENERAL]);
    } else {
	string.append ("UNKNOWN");
    }

    return (string);
}

/* SECURITY::securityCodeToLongString

   Purpose: Construct a security error code into a long human-readable string

     Input: code   = the error code
		     (int)

   Output: The procedure returns a DynamicString that contains the long string. 
*/
 
DynamicString SECURITY::securityCodeToLongString (int code)
{
    DynamicString string;

	/* put the error message into the string */

    switch (code) {

	case SECURITY_OK:
	    break;

	case SECURITY_NOT_INITIALIZED: 
	    string.append ("security not available: security module has not been initialized");
	    break;

	case SECURITY_NO_DIRECTORY:
	    string.append ("security not available: directory \"");
	    string.append (directory);
	    string.append ("\" does not exist");
	    break;

	case SECURITY_NO_PASS:
	    string.append ("security not available: unable to read pass phrase from environment variable \"");
	    string.append (AGENT_PASS_ENV);
	    string.append ("\" or from file \"");
	    string.append (directory);
	    string.append ("/passphrase\"");
	    break;

	case SECURITY_NO_ID:
	    string.append ("security not available: unable to read keyname from environment variable \"");
	    string.append (AGENT_KEY_ENV);
	    string.append ("\" or from file \"");
	    string.append (directory);
	    string.append ("/keyname\"");
	    break;

	default:
	    string.append ("unknown error");
	    break;
    }

    return (string);
}
