/* Agent Tcl
   Bob Gray
   11 July 1995

   transmit.cc

   This file implmenets the routines that send and receive messages.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"	// platform-specific definitiions
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#include "mesgConnection.h"	// Connection

#include "platTimers.h"
#include "mesgTcpip.h"
#include "agentId.h"
#include "genEncrypt.h"
#include "genManager.h"
#include "genSecurity.h"
#include "genTransmit.h"
#include "genUtility.h"
#include "truefalse.h"

/* secureReceive

   Purpose: Receive an incoming message *performing any necessary decryption*

     Input: sockfd   = socket descriptor on which the message is arriving
		       (int)

	    set      = set of messages in which the message should appear
		       (class MESSAGE_SET &)

	    receiver = security characteristics of the receiver
		       (class BASE_SECURITY &)

    Output: The procedure returns one of the following codes.

	    MESSAGE_OK        = success
	    MESSAGE_SYS_ERROR = system error while receiving the message
	    MESSAGE_DECRYPT   = the message could not be decrypted
	    MESSAGE_INVALID   = the message was invalid

	    On success 

	    sender   = id of the encrypter and boolean flags that indicate
		       whether the message was encrypted and/or signed -- i.e,
		       the security characteristics of the sender
		       (class BASE_SECURITY &)

	    message  = pointer to a dynamically allocated MESSAGE structure
		       that contains the incoming message
		       (struct MESSAGE *&)
*/

int secureReceive (int sockfd, const MESSAGE_SET &set, const BASE_SECURITY &receiver, BASE_SECURITY &sender, MESSAGE *&message)
{
    int rc;
    MESSAGE *securedMessage = NULL;

	/* assume that we do not get a message */

    message = NULL;

	/* receive the secured message */

    if ((securedMessage = message_receive (sockfd, protectionMessages)) == NULL) {
	return (MESSAGE_INVALID);
    }

	/* unsecure the message */ 

    if ((rc = unsecure (set, receiver, *securedMessage, sender, message)) == MESSAGE_OK) {

	    /* and then receive the standard message if necessary */

	if (message == NULL) {

	    if ((message = message_receive (sockfd, set)) == NULL) {
		rc = MESSAGE_INVALID;
	    } else {
		rc = MESSAGE_OK;
	    }
	}
    }

    DELETE_OBJECT(securedMessage);
    return (rc);
}

/* secureReceive_noDelayedAcknowledgement

      Note: This procedure is the same as secureReceive except that it
	    sends user-level acknowledgements to avoid the TCP/IP delayed
	    acknowledgement.

#ifdef FIX_LATER
	    Of course this procedure is hack.
#endif
	    
   Purpose: Receive an incoming message *performing any necessary decryption*

     Input: sockfd   = socket descriptor on which the message is arriving
		       (int)

	    set      = set of messages in which the message should appear
		       (class MESSAGE_SET &)

	    receiver = security characteristics of the receiver
		       (class BASE_SECURITY &)

    Output: The procedure returns one of the following codes.

	    MESSAGE_OK        = success
	    MESSAGE_SYS_ERROR = system error while receiving the message
	    MESSAGE_DECRYPT   = the message could not be decrypted
	    MESSAGE_INVALID   = the message was invalid

	    On success 

	    sender   = id of the encrypter and boolean flags that indicate
		       whether the message was encrypted and/or signed -- i.e,
		       the security characteristics of the sender
		       (class BASE_SECURITY &)

	    message  = pointer to a dynamically allocated MESSAGE structure
		       that contains the incoming message
		       (struct MESSAGE *&)
*/

int secureReceive_noDelayedAcknowledgement (int sockfd, const MESSAGE_SET &set, const BASE_SECURITY &receiver, BASE_SECURITY &sender, MESSAGE *&message)
{
    int rc;
    MESSAGE *securedMessage = NULL;

	/* assume that we do not get a message */

    message = NULL;

	/* put the socket into nonblocking mode */

    BlockingMode oldMode;
    tcpip_setBlockingMode (sockfd, e_NonBlocking, oldMode);

	/* receive the secured message */

    if ((securedMessage = message_receive_noDelayedAcknowledgement (sockfd, protectionMessages)) == NULL) {

	rc = MESSAGE_INVALID;

    } else {

	    /* unsecure the message */

        if ((rc = unsecure (set, receiver, *securedMessage, sender, message)) == MESSAGE_OK) {

	        /* and then receive the standard message if necessary */

	    if (message == NULL) {

	        if ((message = message_receive_noDelayedAcknowledgement (sockfd, set)) == NULL) {
		    rc = MESSAGE_INVALID;
	        } else {
		    rc = MESSAGE_OK;
	        }
	    }
	}
    }

	/* cleanup */

    tcpip_setBlockingMode (sockfd, oldMode, oldMode);
    DELETE_OBJECT(securedMessage);
    return (rc);
}

/* message_receive

   Purpose: Receive a message from a file descriptor

     Input: sockfd = socket descriptor
            types  = possible message types

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated MESSAGE structure 
            that contains the transmitted message.   
*/

static void sendUserLevelAcknowledgement (int sockfd, fd_mask *writeSet, int fdSelectIndex, int fdSelectBit)
{
    UINT_32 zeroSize = 0;
    int bytesToGo = sizeof(UINT_32);

	// loop until we write successfully

    while (1) {

	    // write the empty message (i.e., the acknowledgement)

	int nwritten;

	do {
	    nwritten = write (sockfd, (char *) &zeroSize, bytesToGo);
	} while ((nwritten < 0) && (errno == EINTR));

	if (nwritten < 0) {

	    if (errno != EAGAIN) {
		return;
	    }

	} else {

	    bytesToGo -= nwritten;

	    if (bytesToGo == 0) {
		return;
	    }
	}

	    // wait until we can write the rest

	int selectRc;

	do {
	    writeSet[fdSelectIndex] |= fdSelectBit;
	    selectRc = select (sockfd + 1, (SELECT_MASK *) NULL, (SELECT_MASK *) writeSet, (SELECT_MASK *) NULL, NULL);
	} while ((selectRc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	if (selectRc <= 0) {
	    return;
	}
    }
}

MESSAGE *message_receive (int sockfd, const MESSAGE_SET &set)
{
    int nread;
    INT_32 netSize;
    UINT_32 size;
    INT_32 fullSize; 
    char *buffer; 
    MESSAGE *message;

	/* get the size of the message */

    if (tcpip_readLong (sockfd, size) != e_TCPIP_OK) {
	return NULL;
    } 

    if (size == 0) {
	return NULL;
    }

    netSize  = htonl (size);
    fullSize = size + sizeof(UINT_32);

	/* allocate an empty MESSAGE */

    message = new MESSAGE (fullSize);
    buffer  = message -> getBuffer ();

	/* read the whole message */

    memcpy (buffer, (char *) &netSize, sizeof(UINT_32));

    if ((tcpip_blockingReadn (sockfd, buffer + sizeof(UINT_32), size, nread) != e_TCPIP_OK)
	|| ((UINT_32) nread < size)) {
	DELETE_ARRAY_OBJECT(buffer);
	return NULL;
    }

	/* delinearize the message */

    if (message -> delinearize (set) < 0) {
	DELETE_OBJECT(message);
	message = NULL;
    }

	/* done */

    return (message);
}

/* message_receive_noDelayedAcknowledgement

      Note: This procedure is the same as message_receive except that it
	    sends user-level acknowledgements to avoid the TCP/IP delayed
	    acknowledgement.

	    This procedure assumes that the socket is in nonblocking mode.

#ifdef FIX_LATER
	    Of course this procedure is a hack.
#endif
	    
   Purpose: Receive a message from a file descriptor

     Input: sockfd = socket descriptor
            types  = possible message types

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns a pointer to a dynamically allocated MESSAGE structure 
            that contains the transmitted message.   
*/

MESSAGE *message_receive_noDelayedAcknowledgement (int sockfd, const MESSAGE_SET &set)
{
    UINT_32 netSize;
   
	/* assume that we do not receive the message successfully */

    MESSAGE *message        = NULL;
    MESSAGE *scratchMessage = NULL;

	/* set up to read the message size first */

    BOOLEAN haveSize  = e_FALSE;
    char *  bufferPtr = (char *) &netSize;
    int     bytesToGo = sizeof(UINT_32);

	/* set up the select masks */

    fd_mask readSet[MASK_SIZE];
    FDMASK_ZERO (readSet);
    int fdSelectIndex = FDMASK_INDEX (sockfd);
    int fdSelectBit   = FDMASK_BIT   (sockfd);

	/* set up a polling timeout, but set up to use NO timeout first; */
	/* the select call will switch back and forth between NO timeout */
	/* and the polling timeout (to block or poll respectively)       */

    struct timeval zeroTimeout;
    zeroTimeout.tv_sec  = 0;
    zeroTimeout.tv_usec = 0;
    struct timeval *timeoutPtr = NULL;

	/* loop until we have the message */

    while (1) {

	int nread;
	int selectRc;

	    /* do the select */

	do {
	    readSet[fdSelectIndex] |= fdSelectBit;
	    selectRc = select (sockfd + 1, (SELECT_MASK *) readSet, (SELECT_MASK *) NULL, (SELECT_MASK *) NULL, timeoutPtr);
	} while ((selectRc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    /* bail on error */

	if (selectRc < 0) {
	    break;
	}

	    /* if no bytes were ready to be read, and we were in polling  */
	    /* mode, send out our user-level acknowledgement, switch to   */
	    /* blocking mode and loop back around; if no bytes were ready */
	    /* to be read, and we were in blocking mode, just loop back   */
	    /* around                                                     */

	if (selectRc == 0) {

	    if (timeoutPtr == &zeroTimeout) {
		sendUserLevelAcknowledgement (sockfd, readSet, fdSelectIndex, fdSelectBit);
		timeoutPtr = NULL;
	    }

	    continue;
	}

	    /* read whatever is available on the socket */

#ifdef FIX_LATER
	    /* the read (and following code) should be inside its own */
	    /* loop -- right now we are doing extra select calls      */
#endif

	nread = read (sockfd, bufferPtr, bytesToGo);

	    /* bail on any error or EOF -- We can bail on ANY error  */
	    /* since the select call above tells us that there must  */
	    /* be something there to read.  Without the select call, */
	    /* we would need to loop back and try again if we got    */
	    /* EAGAIN on the read.                                   */

	if (nread <= 0) {
	    break;
	}

	    /* otherwise adjust the byte counts */

	bufferPtr += nread;
	bytesToGo -= nread;

	    /* if we now have all the necessary bytes, and we were   */
	    /* reading the size, set up to read the message; if we   */
	    /* now have all the necessary bytes, and we were reading */
	    /* the message buffer, create the MESSAGE instance and   */
	    /* bail                                                  */

	if (bytesToGo == 0) {

	    if (!haveSize) {

	        UINT_32 size = ntohl (netSize); 

	        if (size == 0) {
		    break;
	        }

	        UINT_32 fullSize = size + sizeof(UINT_32);

	        scratchMessage = new MESSAGE (fullSize);
	        bufferPtr = scratchMessage -> getBuffer();
	        memcpy (bufferPtr, (char *) &netSize, sizeof(UINT_32));
	        bufferPtr += sizeof(UINT_32);

	        bytesToGo = size;
		haveSize  = e_TRUE;

	    } else {

		if (scratchMessage -> delinearize (set) < 0) {

		    message        = NULL;
	
		} else {

		    message        = scratchMessage;
		    scratchMessage = NULL;
		}
	
		break;
	    }
	}

	    /* switch to polling mode */

	timeoutPtr = &zeroTimeout;
    }

	/* cleanup and return */

    if (scratchMessage != NULL) {
	DELETE_OBJECT(scratchMessage);
    }

    return (message);
}

/* message_send

     Purpose: Write a message to a file descriptor

       Input: fd      = file descriptor
			(int)

              message = the message
			(class MESSAGE &)

      Output: The procedure returns -1 on error.  Otherwise the procedure
              writes the message to the file descriptor and returns 0.
*/

int message_send (int sockfd, MESSAGE &message)
{
    UINT_32 size;
    char *buffer;
    int nwritten;

	/* linearize the message */

    if (message.linearize () < 0) {
	return MESSAGE_SYS_ERROR;
    }

	/* write out the message */

    size    = message.getSize ();
    buffer  = message.getBuffer ();

    if ((tcpip_blockingWriten (sockfd, buffer, size, nwritten) != e_TCPIP_OK) 
	|| (nwritten < (int) size)) {
	return MESSAGE_LOST_CONNECTION;
    }

    return MESSAGE_OK; 
}
