/* Agent Tcl
   Bob Gray
   25 June 1995

   genUtility.cc

   This file implements various convenience routines.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "suppStrings.h"
#include "genUtility.h"

  /* compare two strings (NULL and the empty string '\0' are considered equal) */

int strcmpNullAndEmpty (const char *s1, const char *s2)
{
    int rc;
    int nulls;

	/* check for nulls and empty strings */

    nulls = (((s1 == NULL) || (*s1 == '\0')) && ((s2 == NULL) || (*s2 == '\0')));

	/* normal string compare */

    if (nulls) {
	rc = 0;
    } else if ((s1 == NULL) || (*s1 == '\0')) {
	rc = -1;
    } else if ((s2 == NULL) || (*s2 == '\0')) {
	rc = 1;
    } else {
	rc = strcmp(s1,s2);
    }

    return (rc);
}

/* breakout_ip

   Purpose: Break a UINT_32 IP address out of a buffer 

     Input: bp     = current position in the buffer
		     (char *)
		
            end_bp = last position in the buffer
		     (char *)

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets value to the value of the UINT_32
            IP address. 
*/

char *breakout_ip (char *bp, char *end_bp, UINT_32 &value)
{
	/* check bounds -- then get the value */

    if (bp + sizeof(UINT_32) > end_bp) {
	bp = NULL;
    } else {
	bp = fast_src_memcpy ((char *) &value, bp, sizeof(UINT_32));
    }

    return (bp);     
}

/* breakout_short

   Purpose: Break a UINT_16 integer out of a buffer

     Input: bp     = current position in the buffer
		     (char *)
		
            end_bp = last position in the buffer
		     (char *)

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets value to the value of the UINT_16
            integer.
*/

char *breakout_short (char *bp, char *end_bp, UINT_16 &value)
{
	/* check bounds -- then get the value */

    if (bp + sizeof(UINT_16) > end_bp) {
	bp = NULL;
    } else {
	bp = fast_src_memcpy ((char *) &value, bp, sizeof(UINT_16));
	value = ntohs (value);
    }

    return (bp);     
}

/* breakout_long

   Purpose: Break a UINT_32 integer out of a buffer 

     Input: bp     = current position in the buffer
		     (char *)
		
            end_bp = last position in the buffer
		     (char *)

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets value to the value of the UINT_32
            integer.
*/

char *breakout_long (char *bp, char *end_bp, UINT_32 &value)
{
	/* check bounds -- then get the value */

    if (bp + sizeof(UINT_32) > end_bp) {
	bp = NULL;
    } else {
	bp = fast_src_memcpy ((char *) &value, bp, sizeof(UINT_32));
	value = ntohl (value);
    }

    return (bp);     
}

/* breakout_string

   Purpose: Break a string out of the a buffer 

     Input: bp     = current position in the buffer
		     (char *)

            end_bp = last position in the buffer
		     (char *)
           
    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets string to a dynamically allocated
	    copy of the string.
*/

char *breakout_string (char *bp, char *end_bp, char **string)
{
    UINT_32 length;   /* length of the string */

    if ((bp = breakout_long (bp, end_bp, length)) != NULL) {

	if (length == 0) {
	    *string = NULL;
	} else if (bp + length > end_bp) {
	    bp = NULL;
	} else {
	    *string = bp;
	    bp += length - 1;
	    *bp = '\0';
	    bp += 1;
	    *string = strcpyWithAlloc (*string);
	}
    }

    return (bp); 
} 

/* breakout_byte

   Purpose: Break a UINT_8 integer out of a buffer 

     Input: bp     = current position in the buffer
		     (char *)
		
            end_bp = last position in the buffer
		     (char *)

    Output: The procedure returns NULL on error.  Otherwise the procedure
            returns the new bp and sets value to the value of the UINT_8
            integer.
*/

char *breakout_byte (char *bp, char *end_bp, UINT_8 &value)
{
	/* check bounds -- then get the value */

    if (bp + sizeof(UINT_8) > end_bp) {
	bp = NULL;
    } else {
	value = * (unsigned char *) bp++;
    }

    return (bp);     
}
