@TK_STARTUP@

# Agent Tcl
# Bob Gray
# 5 April 1995 
#
# This application is used to generate and manage the PGP keys.

# ------------------------------------------------------------------------------
# BEGIN: global variables
# ------------------------------------------------------------------------------

# global variables
#
# windows        = list of mutually exclusive windows
#
# pgpDirectory   = name of the directory that contains the user's
#		   secring.pgp, pubring.pgp, config.txt and language.txt.
#
# agentDirectory = name of the directory that contains the user's
#                  keyname (and possibly passphrase) file
# 
# pgpProgram     = full pathname of the PGP program
#
# passDone       = SET if the user has finished with the pass phrase
#                  entry box and UNSET otherwise
#
# fileDone       = SET if the user has finished with the export
#                  filename entry box and UNSET otherwise

set windows {.servertop .usertop .exporttop .importtop .dirselect .fileselect}
set pgpDirectory @PGP_KEYS@
set pgpProgram @PGP_EXECUTABLE@
set agentDirectory ~/.agenttcl
catch {unset passDone}
catch {unset fileDone}

# ------------------------------------------------------------------------------
# END: global variables
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: report any PGP errors 
# ------------------------------------------------------------------------------

proc pgp_error {result {key {}}} {

    if {[string first "couldn't execute" $result] != -1} {
	tk_dialog .t "No PGP!" "The \"pgp\" executable can not be found.  PGP must be installed and on your current search path." error 0 OK;
    } elseif {[string first "Bad pass phrase" $result] != -1} {
	tk_dialog .t "Invalid pass phrase!" "Invalid pass phrase for \"$key\".  Please enter the pass phrase again!" error 0 OK
    } else {
	option add *pgpt.msg.wrapLength 8i 100
	tk_dialog .pgpt "PGP error!" "Unexpected error while running PGP!\n$result" error 0 OK
    }
}

# ------------------------------------------------------------------------------
# END: report any PGP errors 
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: get the PGP key directory ready
# ------------------------------------------------------------------------------

# tildeSubstitution replaces the tilde at the beginning of a filename.  I 
# assume that I am missing something obvious and that there is a way to do
# this directly with Tcl primitives.  If so, someone please let me know.

proc tildeSubstitution filename {

    global env

    if {[string index $filename 0] == "~"} {

	if {[string index $filename 1] != "/"} {
	    set separator "/"
	} else {
	    set separator ""
	}

	set filename $env(HOME)$separator[string range $filename 1 end]
    }

    return $filename
}

proc readyDirectory {directory {access {}}} { 

    global env

	# directory might already exist

    if {[file exists $directory]} {

	    # make sure that we actually have a directory

	if {![file isdirectory $directory]} {
	    tk_dialog .t "Not a directory!" "\"$directory\" is not a directory!" error 0 OK
	    return -1
	}

	    # make sure that we have complete access to the directory

	if {![file readable $directory] || ![file writable $directory] || ![file executable $directory]} {
	    tk_dialog "No access!" "You do not have read/write/execute access to directory \"$directory\"!" error 0 OK
	    return -1
	}

	    # make sure that no one else has access to the directory

	if {$access != "read-access-okay"} {

	    file stat $directory stat

	    if {$stat(mode) & 077} {
		tk_dialog .t "Too much access!" "Directory \"$directory\" is readable by other users!" error 0 OK
		return -1
	    }
	}

    } else {

	    # create the directory

	if {[catch {

 	    exec mkdir $directory
	    exec chmod 0700 $directory

	} result]} {

	    tk_dialog .t "Can't create!" "Unable to create directory \"$directory\"!" error 0 OK
	    return -1
	}
    }

    return 0
}

proc readyPgpDirectory {newDirectory} {

    global env
    global pgpDirectory

	# do tilde subsitituion

    set newDirectory [tildeSubstitution $newDirectory]

	# create the directory if necessary

    if {[readyDirectory $newDirectory read-access-okay] != "0"} {
	return -1
    }

	# success so set the environment variable PGPPATH and create an empty
	# config.txt file if one is not present

    if {![file exists $newDirectory/config.txt]} {
	set fd [open $newDirectory/config.txt w]
	puts $fd "# dummy configuration file so that pgp will not return errors"
	close $fd
    }

    set env(PGPPATH) $newDirectory
    set pgpDirectory $newDirectory
    return 0
}

# ------------------------------------------------------------------------------
# END: get the PGP key directory ready
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: change the PGP key directory 
# ------------------------------------------------------------------------------

# the dirselect routines were adapted from the file selection dialog in
# "Practical Programming in Tcl and Tk" by Brent Welch.

proc dirselectResources {} {
    # path is used to enter the file name
    option add *Fileselect*path.relief		sunken	startup
    option add *Fileselect*path.background	white	startup
    option add *Fileselect*path.foreground	black	startup
    # Text for the label on pathname entry
    option add *Fileselect*l.text		File:	startup
    # Text for the OK and Cancel buttons
    option add *Fileselect*ok*text		OK	startup
    option add *Fileselect*ok*underline		0	startup
    option add *Fileselect*cancel.text		Cancel	startup
    option add *Fileselect*cancel.underline 	0	startup
    # Size of the listbox
    option add *Fileselect*list.width		20	startup
    option add *Fileselect*list.height		10	startup
}

# dirselect returns the selected pathname, or {}
proc dirselect {{why "File Selection"} {default {}} {mustExist 1} } {
	global dirselect

	set t [toplevel .dirselect -bd 4 -class Fileselect]
	dirselectResources
    
	message $t.msg -aspect 1000 -text $why
	pack $t.msg -side top -fill x
    
	# Create a read-only entry for the durrent directory
	set dirselect(dirEnt) [entry $t.dir -width 15 \
		-relief flat -state disabled]
	pack $t.dir -side top -fill x
    
	# Create an entry for the pathname
	# The value is kept in dirselect(path)
	frame $t.top
	label $t.top.l -padx 0
	set e [entry $t.top.path \
		-textvariable dirselect(path)]
	pack $t.top -side top -fill x
	pack $t.top.l -side left
	pack $t.top.path -side right -fill x -expand true

    
	# Create a listbox to hold the directory contents
	set lb [listbox $t.list \
		-yscrollcommand [list $t.scroll set]]
	scrollbar $t.scroll -command [list $lb yview]

	# Create the OK and Cancel buttons
	# The OK button has a rim to indicate it is the default
	frame $t.buttons -bd 10
	frame $t.buttons.ok -bd 2 -relief sunken
	set ok [button $t.buttons.ok.b \
		-command {dirselectOK select}]
	set can [button $t.buttons.cancel \
		-command dirselectCancel]

	# Pack the list, scrollbar, and button box
	# in a horizontal stack below the upper widgets
	pack $t.list -side left -fill both -expand true
	pack $t.scroll -side left -fill y
	pack $t.buttons -side left -fill both
	pack $t.buttons.ok $t.buttons.cancel \
		-side top -padx 10 -pady 5
	pack $t.buttons.ok.b -padx 4 -pady 4

	dirselectBindings $t $e $lb $ok $can

	# Initialize variables and list the directory
	if {[string length $default] == 0} {
		set dirselect(path) {}
		set dir [pwd]
	} else {
		set dirselect(path) [file tail $default]
		set dir [file dirname $default]
	}
	set dirselect(dir) {}
	set dirselect(done) 0
	set dirselect(mustExist) $mustExist

	# Wait for the listbox to be visible so
	# we can provide feedback during the listing 
	tkwait visibility .dirselect.list
	dirselectList $dir

	tkwait variable dirselect(done)
	destroy $t
	return $dirselect(path)
}
proc dirselectBindings { t e lb ok can } {
	# t - toplevel
	# e - name entry
	# lb - listbox
	# ok - OK button
	# can - Cancel button

	# Elimate the all binding tag because we
	# do our own focus management
	foreach w [list $e $lb $ok $can] {
	    bindtags $w [list $t [winfo class $w] $w]
	}
	# Dialog-global cancel binding
	bind $t <Control-c> dirselectCancel

	# Entry bindings
	bind $e <Return> {dirselectOK select}

	# A single click, or <space>, puts the name in the entry
	# A double-click, or <Return>, selects the name
	bind $lb <space> "dirselectTake $%W ; focus $e"
	bind $lb <Button-1> \
		"dirselectClick %W %y ; focus $e"
	bind $lb <Return> "dirselectTake %W ; dirselectOK traverse"
	bind $lb <Double-Button-1> \
		"dirselectClick %W %y ; dirselectOK traverse"

	# Focus management.  	# <Return> or <space> selects the name.
	bind $e <Tab> "focus $lb ; $lb select set 0"
	bind $lb <Tab> "focus $e"

	# Button focus.  Extract the underlined letter
	# from the button label to use as the focus key.
	foreach but [list $ok $can] {
		set char [string tolower [string index  \
			[$but cget -text] [$but cget -underline]]]
		bind $t <Alt-$char> "focus $but ; break"
	}
	bind $ok <Tab> "focus $can"
	bind $can <Tab> "focus $ok"

	# Set up for type in
	focus $e
}

proc dirselectList { dir {files {}} } {
	global dirselect

	# Update the directory display
	set e $dirselect(dirEnt)
	$e config -state normal
	$e delete 0 end
	$e insert 0 $dir
	$e config -state disabled
	# scroll to view the tail end
	$e xview moveto 1

	.dirselect.list delete 0 end
	set dirselect(dir) $dir
	if ![file isdirectory $dir] {
	    .dirselect.list insert 0 "Bad Directory"
	    return
	}
	.dirselect.list insert 0 Listing...
	update idletasks
	.dirselect.list delete 0
	if {[string length $files] == 0} {
		# List the directory and add an
		# entry for the parent directory
		set files [glob -nocomplain $dirselect(dir)/.* $dirselect(dir)/*]
	}
	# Sort the directories to the front
	set dirs {}
	foreach f [lsort $files] {
		if [file isdirectory $f] {
			.dirselect.list insert end [file tail $f]/
		} 
	}
}
proc dirselectOK {type} {
	global dirselect

	# compute the new path
	set dirselect(path) [string trim $dirselect(path)]
	set path [file join $dirselect(dir) $dirselect(path)]
	if {$path != "/"} {
	    set path [string trimright $path /]
	}

	# just return the path if we hit the OK button or <return> inside the entry box
	if {$type == "select"} {
		if {$dirselect(path) == ""} {
		    return
		}
		set dirselect(path) $path
		set dirselect(done) 1
		return
	}

	# handle the parent directory
	if {$dirselect(path) == "../"} {
		set dirselect(path) {}
		set dirselect(dir) [file dirname $dirselect(dir)]
		dirselectOK traverse
		return
	}

	# handle the current directory
	if {$dirselect(path) == "./"} {
		set dirselect(path) {}
		dirselectOK traverse
		return
	}

	# handle a new directory
	if [file isdirectory $path] {
		set dirselect(path) {}
		dirselectList $path
		return
	}

	# error if we somehow make it here
	return -code error "bug in directory dialog"
}
proc dirselectCancel {} {
	global dirselect
	set dirselect(done) 1
	set dirselect(path) {}
}

proc dirselectClick { lb y } {
	# Take the item the user clicked on
	global dirselect
	set dirselect(path) [$lb get [$lb nearest $y]]
}
proc dirselectTake { lb } {
	# Take the currently selected list item
	global dirselect
	set dirselect(path) [$lb get [$lb curselection]]
}

proc changePgpDirectory {} {

    global env pgpDirectory windows

	# are we allowed to display the window?

    if {![showWindow .dirselect $windows]} {
	return
    }

	# ask the user for the directory

    set newDirectory [dirselect "Select PGP directory" $pgpDirectory 0]

    if {$newDirectory == ""} {
	return
    }

	# get the directory ready

    if {[readyPgpDirectory $newDirectory] != "0"} {
	return
    }

	# update the key window

    updateMainWindow
}

# ------------------------------------------------------------------------------
# END: change the PGP key directory 
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: get a list of the keys in a particular PGP key ring 
# ------------------------------------------------------------------------------

proc makeKeyList {{priFilename {}} {pubFilename {}}} {

    global pgpProgram pgpDirectory

	# use defaults if necessary

    if {$priFilename == ""} {
	set priFilename $pgpDirectory/secring.pgp
    }

    if {$pubFilename == ""} {
	set pubFilename $pgpDirectory/pubring.pgp
    }

	# make sure that we have the public key file at least

    if {![file exists $pubFilename]} {
	return {}
    }

	# get the lists of private and public keys 

    set public [exec $pgpProgram +verbose=0 +nomanual=on +batchmode +force -kv $pubFilename]
    set public [split [string trim $public] "\n"]

    if {[file exists $priFilename]} {
	set private [exec $pgpProgram +verbose=0 +nomanual=on +batchmode +force -kv $priFilename]
	set private [split [string trim $private] "\n"]
    } else {
	set private {}
    }

	# construct the merged list

    set list {}

    foreach pubkey $public {

	if {[lindex $pubkey 0] == "pub"} {

	    set userid [lrange $pubkey 3 end]
	    set found 0

	    foreach prikey $private {
	        if {$userid == [lrange $prikey 3 end]} {
		    set found 1
		    break
		}
	    }

	    if {$found} { 
		lappend list [list $userid private]
	    } else {
		lappend list [list $userid public]
	    }
	}
    }

	# sort the list

    set list [lsort $list]
    return $list
}

# ------------------------------------------------------------------------------
# END: get a list of the keys in a particular PGP key ring 
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: fill in a frame with a display of the keys from a particular PGP key ring
# ------------------------------------------------------------------------------

proc createKeyWindow {frame width selectMode {priFilename {}} {pubFilename {}}} {

    global pgpDirectory

	# use defaults if necessary

    if {$priFilename == ""} {
	set priFilename $pgpDirectory/secring.pgp
    }

    if {$pubFilename == ""} {
	set pubFilename $pgpDirectory/pubring.pgp
    }

	# make the key list

    if {[catch {

	set list [makeKeyList $priFilename $pubFilename]

    } result]} {

	pgp_error $result
	return -1
    }

    if {![winfo exists $frame.top]} {

	    # create the listbox

	frame $frame.top
	frame $frame.bottom

	    # create the public key area

	frame $frame.pubscrollframe
        frame $frame.pubpad
	message $frame.publabel -text "Public only:" -justify left 
	scrollbar $frame.pubscroll \
	    -command "$frame.pubkeys yview"
	scrollbar $frame.pubhscroll -orient horizontal \
	    -command "$frame.pubkeys xview"
	listbox $frame.pubkeys -width $width -height 8 \
	    -yscrollcommand "$frame.pubscroll set" \
	    -xscrollcommand "$frame.pubhscroll set" \
	    -selectmode $selectMode

	pack $frame.pubhscroll -in $frame.pubscrollframe -side left -fill x -expand 1
        pack $frame.pubpad -in $frame.pubscrollframe -side right
	pack $frame.pubscrollframe -in $frame.bottom -side bottom -fill x
	pack $frame.publabel -in $frame.bottom -side top -anchor w
	pack $frame.pubscroll -in $frame.bottom -side right -fill y
	pack $frame.pubkeys -in $frame.bottom -side right -fill both -expand 1
	pack $frame.bottom -side bottom -fill both -expand 1

	set pad [winfo reqwidth $frame.pubscroll]
	$frame.pubpad configure -width $pad

        set pad [winfo reqwidth $frame.pubkeys]
	$frame.publabel configure -width $pad

	    # create the private/public key area

	frame $frame.priscrollframe
	frame $frame.pripad
	message $frame.prilabel -text "Private and public:" -justify left 
	scrollbar $frame.priscroll \
	    -command "$frame.prikeys yview"
        scrollbar $frame.prihscroll -orient horizontal \
	    -command "$frame.prikeys xview"
	listbox $frame.prikeys -width $width -height 8 \
	    -yscrollcommand "$frame.priscroll set" \
	    -xscrollcommand "$frame.prihscroll set" \
	    -selectmode $selectMode

	pack $frame.prihscroll -in $frame.priscrollframe -side left -fill x -expand 1
        pack $frame.pripad -in $frame.priscrollframe -side right
	pack $frame.priscrollframe -in $frame.top -side bottom -fill x
	pack $frame.prilabel -in $frame.top -side top -anchor w
	pack $frame.priscroll -in $frame.top -side right -fill y 
	pack $frame.prikeys -in $frame.top -side right -fill both -expand 1
	pack $frame.top -side bottom -fill both -expand 1

	set pad [winfo reqwidth $frame.priscroll]
	$frame.pripad configure -width $pad

        set pad [winfo reqwidth $frame.prikeys]
	$frame.prilabel configure -width $pad

    } else {

	    # erase what is there if window already exists

	$frame.prikeys delete 0 end
	$frame.pubkeys delete 0 end
    }

	# insert the private/public keys

    foreach i $list {
	if {[lindex $i 1] == "private"} {
	    $frame.prikeys insert end "[lindex $i 0]"
	} 
    }

	# insert the public keys

    foreach i $list {
	if {[lindex $i 1] != "private"} {
	    $frame.pubkeys insert end "[lindex $i 0]"
	}
    }

    return 0
}

# ------------------------------------------------------------------------------
# END: fill in a frame with a display of the keys from a particular PGP key ring
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: create and update the main window 
# ------------------------------------------------------------------------------

proc updateMainWindow {} {

    global pgpDirectory

	# update the key list

    if {[createKeyWindow .list 60 browse] != "0"} {
	return -1
    }

	# update the name of the PGP directory

    .list.directorylabel configure -text "PGP directory: $pgpDirectory"
    return 0
}

proc createMainWindow {} {

    global pgpDirectory

	# create the main window

    main create 
    wm title . "Managing the Agent Tcl keys"

	# create the menu bar

    frame .menubar -relief raised -bd 2
    pack .menubar -side top -fill x

	# create the menubar buttons

    menubutton .menubar.file -text File -underline 0 -menu .menubar.file.menu
    menubutton .menubar.keys -text Keys -underline 0 -menu .menubar.keys.menu
    menubutton .menubar.help -text Help -underline 0 -menu .menubar.help.menu
    pack .menubar.file .menubar.keys -side left
    pack .menubar.help -side right

	# create the menu items

    menu .menubar.file.menu

    .menubar.file.menu add command -label "Change PGP directory" \
	-command {changePgpDirectory}

    .menubar.file.menu add separator

    .menubar.file.menu add command -label "Quit" \
	-command {exit}

    menu .menubar.keys.menu

    .menubar.keys.menu add command -label "New server key" \
	-command {getServerKey}

    .menubar.keys.menu add command -label "New user key" \
	-command {getUserKey}

    .menubar.keys.menu add separator

    .menubar.keys.menu add command -label "Remove key" \
	-command {
	    removeKey .list 
	    updateMainWindow
	    if {[winfo exists .exporttop]} {
		updateKeyExportWindow main
	    }
	}

    .menubar.keys.menu add separator

    .menubar.keys.menu add command -label "Import/Export keys" \
	-command {exportKeys}

    .menubar.keys.menu add separator

    .menubar.keys.menu add command -label "This is me!" \
	-command {setMe}

	# make the frames for the menu buttons and for the key list

    frame .buttons
    pack .buttons -side left -fill y
    frame .list
    pack .list -side left -fill both -expand 1
 
	# create the buttons

    button .buttons.server -text "Generate a server key" \
	-command {getServerKey}

    button .buttons.user -text "Generate a user key" \
	-command {getUserKey}

    button .buttons.remove -text "Remove Key" \
	-command {
	    removeKey .list
	    updateMainWindow
	    if {[winfo exists .exporttop]} {
		updateKeyExportWindow main
	    }
	}

    button .buttons.export -text "Import/Export Keys" \
	-command {exportKeys}

    button .buttons.me -text "This is me!" \
	-command {setMe}

    button .buttons.change -text "Change PGP directory" \
	-command {changePgpDirectory}

    button .buttons.quit -text "Quit" \
	-command {exit}

    pack .buttons.server -side top -fill x -anchor w
    pack .buttons.user -side top -in .buttons -fill x -anchor w
    pack .buttons.remove -side top -fill x -anchor w 
    pack .buttons.export -side top -fill x -anchor w 
    pack .buttons.me -side top -fill x -anchor w
    pack .buttons.change -side top -fill x -anchor w
    pack .buttons.quit -side top -fill x -anchor w

	# create the label that shows the name of the PGP directory

    label .list.directorylabel -text "PGP directory: $pgpDirectory" -anchor w 
    pack .list.directorylabel -side top -fill x

	# create the key list

    if {[updateMainWindow] != "0"} {
	return -1
    }

	# set the minimum window size

    setMinimumSize .
    return 0 
}

# ------------------------------------------------------------------------------
# END: create and update the main window 
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: get the filename of an exported keyring
# ------------------------------------------------------------------------------

# the fileselect routines are adapted from the file selection dialog in
# "Practical Programming in Tcl and Tk" by Brent Welch.

proc fileselectResources {} {
    # path is used to enter the file name
    option add *Fileselect*path.relief		sunken	startup
    option add *Fileselect*path.background	white	startup
    option add *Fileselect*path.foreground	black	startup
    # Text for the label on pathname entry
    option add *Fileselect*l.text		File:	startup
    # Text for the OK and Cancel buttons
    option add *Fileselect*ok*text		OK	startup
    option add *Fileselect*ok*underline		0	startup
    option add *Fileselect*cancel.text		Cancel	startup
    option add *Fileselect*cancel.underline 	0	startup
    # Size of the listbox
    option add *Fileselect*list.width		20	startup
    option add *Fileselect*list.height		10	startup
}

# fileselect returns the selected pathname, or {}
proc fileselect {{why "File Selection"} {default {}} {mustExist 1} } {
	global fileselect

	set t [toplevel .fileselect -bd 4 -class Fileselect]
	fileselectResources
    
	message $t.msg -aspect 1000 -text $why
	pack $t.msg -side top -fill x
    
	# Create a read-only entry for the durrent directory
	set fileselect(dirEnt) [entry $t.dir -width 15 \
		-relief flat -state disabled]
	pack $t.dir -side top -fill x
    
	# Create an entry for the pathname
	# The value is kept in fileselect(path)
	frame $t.top
	label $t.top.l -padx 0
	set e [entry $t.top.path \
		-textvariable fileselect(path)]
	pack $t.top -side top -fill x
	pack $t.top.l -side left
	pack $t.top.path -side right -fill x -expand true

    
	# Create a listbox to hold the directory contents
	set lb [listbox $t.list \
		-yscrollcommand [list $t.scroll set]]
	scrollbar $t.scroll -command [list $lb yview]

	# Create the OK and Cancel buttons
	# The OK button has a rim to indicate it is the default
	frame $t.buttons -bd 10
	frame $t.buttons.ok -bd 2 -relief sunken
	set ok [button $t.buttons.ok.b \
		-command fileselectOK]
	set can [button $t.buttons.cancel \
		-command fileselectCancel]

	# Pack the list, scrollbar, and button box
	# in a horizontal stack below the upper widgets
	pack $t.list -side left -fill both -expand true
	pack $t.scroll -side left -fill y
	pack $t.buttons -side left -fill both
	pack $t.buttons.ok $t.buttons.cancel \
		-side top -padx 10 -pady 5
	pack $t.buttons.ok.b -padx 4 -pady 4

	fileselectBindings $t $e $lb $ok $can

	# Initialize variables and list the directory
	if {[string length $default] == 0} {
		set fileselect(path) {}
		set dir [pwd]
	} else {
		set fileselect(path) [file tail $default]
		set dir [file dirname $default]
	}
	set fileselect(dir) {}
	set fileselect(done) 0
	set fileselect(mustExist) $mustExist

	# Wait for the listbox to be visible so
	# we can provide feedback during the listing 
	tkwait visibility .fileselect.list
	fileselectList $dir

	tkwait variable fileselect(done)
	destroy $t
	return $fileselect(path)
}
proc fileselectBindings { t e lb ok can } {
	# t - toplevel
	# e - name entry
	# lb - listbox
	# ok - OK button
	# can - Cancel button

	# Elimate the all binding tag because we
	# do our own focus management
	foreach w [list $e $lb $ok $can] {
	    bindtags $w [list $t [winfo class $w] $w]
	}
	# Dialog-global cancel binding
	bind $t <Control-c> fileselectCancel

	# Entry bindings
	bind $e <Return> fileselectOK

	# A single click, or <space>, puts the name in the entry
	# A double-click, or <Return>, selects the name
	bind $lb <space> "fileselectTake $%W ; focus $e"
	bind $lb <Button-1> \
		"fileselectClick %W %y ; focus $e"
	bind $lb <Return> "fileselectTake %W ; fileselectOK"
	bind $lb <Double-Button-1> \
		"fileselectClick %W %y ; fileselectOK"

	# Focus management.  	# <Return> or <space> selects the name.
	bind $e <Tab> "focus $lb ; $lb select set 0"
	bind $lb <Tab> "focus $e"

	# Button focus.  Extract the underlined letter
	# from the button label to use as the focus key.
	foreach but [list $ok $can] {
		set char [string tolower [string index  \
			[$but cget -text] [$but cget -underline]]]
		bind $t <Alt-$char> "focus $but ; break"
	}
	bind $ok <Tab> "focus $can"
	bind $can <Tab> "focus $ok"

	# Set up for type in
	focus $e
}

proc fileselectList { dir {files {}} } {
	global fileselect

	if [catch {set dir [glob $dir]}] {
	    .fileselect.list insert 0 "Bad Directory"
	    return
	}

	# Update the directory display
	set e $fileselect(dirEnt)
	$e config -state normal
	$e delete 0 end
	$e insert 0 $dir
	$e config -state disabled
	# scroll to view the tail end
	$e xview moveto 1

	.fileselect.list delete 0 end
	set fileselect(dir) $dir

	if ![file isdirectory $dir] {
	    .fileselect.list insert 0 "Bad Directory"
	    return
	}
	.fileselect.list insert 0 Listing...
	update idletasks
	.fileselect.list delete 0
	if {[string length $files] == 0} {
		# List the directory and add an
		# entry for the parent directory
		set files [glob -nocomplain $fileselect(dir)/*]
		.fileselect.list insert end ../
	}
	# Sort the directories to the front
	set dirs {}
	set others {}
	foreach f [lsort $files] {
		if [file isdirectory $f] {
			lappend dirs [file tail $f]/
		} else {
			lappend others [file tail $f]
		}
	}
	foreach f [concat $dirs $others] {
		.fileselect.list insert end $f
	}
}
proc fileselectOK {} {
	global fileselect

	# Handle the parent directory specially
	if {[regsub {^\.\./?} $fileselect(path) {} newpath] != 0} {
		set fileselect(path) $newpath
		set fileselect(dir) [file dirname $fileselect(dir)]
		fileselectOK
		return
	}
  
	set path [file join $fileselect(dir) $fileselect(path)]
 
	if [file isdirectory $path] {
		set fileselect(path) {}
		fileselectList $path
		return
	}
	if [file exists $path] {
		set fileselect(path) $path
		set fileselect(done) 1
		return
	}
	# Neither a file or a directory.
	# See if glob will find something
	if [catch {glob $path} files] {
		if [file isdirectory \
			[file dirname $fileselect(path)]] {
			if !$fileselect(mustExist) {
				set fileselect(done) 1
			}
		}
		return
	} else {
		# Ok - current directory is ok,
		# either select the file or list them.
		if {[llength [split $files]] == 1} {
			set fileselect(path) $files
			fileselectOK
		} else {
			set fileselect(dir) [file dirname [lindex $files 0]]
			fileselectList $fileselect(dir) $files
		}
	}
}
proc fileselectCancel {} {
	global fileselect
	set fileselect(done) 1
	set fileselect(path) {}
}

proc fileselectClick { lb y } {
	# Take the item the user clicked on
	global fileselect
	set fileselect(path) [$lb get [$lb nearest $y]]
}
proc fileselectTake { lb } {
	# Take the currently selected list item
	global fileselect
	set fileselect(path) [$lb get [$lb curselection]]
}

proc checkExportFile {filename type} {

    if {![file exists $filename]} {

	if {[catch {

	    set fd [open $filename w 0600]
	    close $fd
	    exec rm $filename

	} result]} {

	    tk_dialog .t "Bad file" \
		"Unable to access file \"$filename\" (the $type\
		portion of the exported keys) due to error \"$result\"" \
		error 0 OK
	    
	    return -1
	}
    }

    return 0
}

proc getKeyExportFilename {} {

    global windows

        # are we allowed to display the window?

    if {![showWindow .fileselect $windows]} {
	return {}
    }

	# get the filename

    set filename [fileselect "Select import/export filename" [pwd]/keys.pub.pgp 0]

    if {$filename == ""} {
	return {}
    }

	# get the base filename

    set filename [string trim [string trimright $filename .]]
    set lowerFilename [string tolower $filename]

    if {[set index [string last .pub.pgp $lowerFilename]] != "-1"} {
	set base [string range $filename 0 [expr $index - 1]]
    } elseif {[set index [string last .pri.pgp $lowerFilename]] != "-1"} {
	set base [string range $filename 0 [expr $index - 1]]
    } else {
	set base $filename
    }

	# construct the filename of the private and public key rings

    set priFilename $base.pri.pgp
    set pubFilename $base.pub.pgp

	# make sure that the key rings are accessible

    if {[checkExportFile $pubFilename public] == "-1"} {
	return {}
    }

    if {[checkExportFile $priFilename private] == "-1"} {
	return {}
    }

	# success

    return [list $priFilename $pubFilename]
}

# ------------------------------------------------------------------------------
# END: get the filename of an exported keyring
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: create and update the key export window 
# ------------------------------------------------------------------------------

proc updateKeyExportWindow {which {priFilename {}} {pubFilename {}}} {

    if {($which == "main") || ($which == "both")} {

	if {[createKeyWindow .exporttop.mainkeys 30 extended] == "-1"} {
	    destroy .exporttop
	    return -1
	}
    }

    if {($which == "export") || ($which == "both")} {

	if {($priFilename == "") || ($pubFilename == "")} {
	    error "empty keyring filenames"
	}

	if {[createKeyWindow .exporttop.exportedkeys 30 extended $priFilename $pubFilename] == "-1"} {
	    destroy .exporttop
	    return -1
	}
    }

    return 0
}

proc exportKeys {} {

    global windows fileDone pgpDirectory

        # are we allowed to display the window?

    if {![showWindow .exporttop $windows]} {
	return
    }

	# get the export keyring 

    set filenames [getKeyExportFilename]

    if {$filenames == ""} {
	return
    }
 
    set priFilename [lindex $filenames 0]
    set pubFilename [lindex $filenames 1]

	# create the main import/export window

    toplevel .exporttop
    wm title .exporttop "Import/Export keys"

	# fill in the window

    frame .exporttop.mainkeys -relief raised -bd 1
    frame .exporttop.buttons -relief raised -bd 1
    frame .exporttop.exportedkeys -relief raised -bd 1
    frame .exporttop.auxbuttons -relief raised -bd 1

	# fill in the main buttons

    button .exporttop.exportboth \
	-text "Export both >>" \
	-command "exportKey .exporttop $priFilename $pubFilename private"

    button .exporttop.exportpriv \
	-text "Export public >>" \
	-command "exportKey .exporttop $priFilename $pubFilename public"

    button .exporttop.import \
	-text "<< Import" \
	-command "importKey .exporttop $priFilename $pubFilename"

    button .exporttop.done \
	-text "Done!" \
	-command "destroy .exporttop"

    pack .exporttop.exportboth -in .exporttop.buttons -side top -fill x -padx 3m -pady 3m 
    pack .exporttop.exportpriv -in .exporttop.buttons -side top -fill x -padx 3m -pady 3m 
    pack .exporttop.import -in .exporttop.buttons -side top -fill x -padx 3m -pady 3m 
    pack .exporttop.done -in .exporttop.buttons -side top -fill x -padx 3m -pady 3m 

	# fill in the aux buttons

    button .exporttop.remove \
	-text "Remove key" \
	-command "
	removeKey .exporttop.exportedkeys $priFilename $pubFilename \" from the export file\"
	updateKeyExportWindow export $priFilename $pubFilename
	"

    button .exporttop.removeall \
	-text "Remove all keys" \
	-command "
	removeAllKeys $priFilename $pubFilename \" from the export file\"
	updateKeyExportWindow export $priFilename $pubFilename
	"

    pack .exporttop.remove -in .exporttop.auxbuttons -side top -fill x -padx 3m -pady 3m 
    pack .exporttop.removeall -in .exporttop.auxbuttons -side top -fill x -padx 3m -pady 3m 

	# fill in the PGP directory name

    label .exporttop.directorylabel -text $pgpDirectory -anchor w 
    pack .exporttop.directorylabel -in .exporttop.mainkeys -side top -fill x

	# fill in the export filename

    label .exporttop.exportlabel -text $priFilename -anchor w 
    pack .exporttop.exportlabel -in .exporttop.exportedkeys -side top -fill x
 
	# pack in the frames

    pack .exporttop.mainkeys -side left -fill both -expand 1
    pack .exporttop.buttons -side left -fill y
    pack .exporttop.exportedkeys -side left -fill both -expand 1
    pack .exporttop.auxbuttons -side left -fill y

	# fill in the keys

    if {[updateKeyExportWindow both $priFilename $pubFilename] == "-1"} {
	return
    }

	# set the minimum window size

    setMinimumSize .exporttop
}

# ------------------------------------------------------------------------------
# END: create and update the key export window 
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: remove individual keys or an entire keyring
# ------------------------------------------------------------------------------

proc removeKey {w {priFilename {}} {pubFilename {}} {message {}}} {

    global pgpDirectory pgpProgram

	# use defaults if necessary

    if {$priFilename == ""} {
	set priFilename $pgpDirectory/secring.pgp
    }

    if {$pubFilename == ""} {
	set pubFilename $pgpDirectory/pubring.pgp
    }

	# get the indices of the currently selected keynames

    set indices [$w.prikeys curselection]
    set keytype private
    set window $w.prikeys

    if {$indices == ""} {

	set indices [$w.pubkeys curselection]
	set keytype public
	set window $w.pubkeys

	if {$indices == ""} {
	    bell
	    return
	}
    } 

	# remove each key

    set dispostion {}

    foreach index $indices {

	    # get the keyname

	set keyname [$window get $index]

	    # double check with user

	if {$dispostion == ""} {

	    set code [
		tk_dialog .t "Confirm" \
		"You are about to remove key \"$keyname\"$message.  Do you want to proceed?" \
		error 1 Yes NO! "Yes to all" "No to all"
	    ]

	    if {$code == 1} {
		continue
	    }

	    if {$code == 2} {
		set dispostion yes-to-all
	    }

	    if {$code == 3} {
		break
	    }
	}

	    # remove the key from the PGP files

	catch {

	    if {[file exists $pubFilename]} {
		exec $pgpProgram +nomanual=on +verbose=0 +batchmode +force -kr $keyname $pubFilename
	    }

	    pgp_error $result
	    break
	}

	catch {

	    if {($keytype == "private") && [file exists $priFilename]} {
		exec $pgpProgram +nomanual=on +verbose=0 +batchmode +force -kr $keyname $priFilename
	    }
	
	    pgp_error $result
	    break
	}
    }
}

proc removeAllKeys {priFilename pubFilename message} {

	# double check with user

    set code [
	tk_dialog .t "Confirm" \
	"You are about to remove all the keys$message.  Do you want to proceed?" \
	error 1 Yes NO! 
    ]

    if {$code != 0} {
	return
    }

	# remove the files

    catch {exec rm $priFilename}
    catch {exec rm $pubFilename}
}

# ------------------------------------------------------------------------------
# END: remove individual keys or an entire keyring
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# BEGIN: import/export keys 
# ------------------------------------------------------------------------------

proc exportKey {w priFilename pubFilename type} {

    global pgpDirectory pgpProgram

	# get the indices of the currently selected keyname

    set indices [$w.mainkeys.prikeys curselection]
    set keytype private
    set window $w.mainkeys.prikeys

    if {$indices == ""} {

	set indices [$w.mainkeys.pubkeys curselection]
	set keytype public 
	set window $w.mainkeys.pubkeys

	if {$indices == ""} {
	    bell
	    return
	}
    }

	# see if we are trying to export both private/public when we only
	# have public

    if {($type == "private") && ($keytype == "public")} {
	tk_dialog .t "No private key!" "There is no private key to export!." error 0 OK;
	return
    }

	# export each key

    foreach index $indices {

	    # get the keyname

	set keyname [$window get $index]

	    # export the key

	if {[catch {

	    exec $pgpProgram +nomanual=on +verbose=0 +batchmode +force -kx $keyname $pubFilename $pgpDirectory/pubring.pgp

	    if {$type == "private"} {
	        exec $pgpProgram +nomanual=on +verbose=0 +batchmode +force -kx $keyname $priFilename $pgpDirectory/secring.pgp
	    }

	} result]} {

	    pgp_error $result
	    break
	}
    }

	# update the key export windows

    updateKeyExportWindow export $priFilename $pubFilename
}

proc importKey {w priFilename pubFilename} {

    global pgpDirectory pgpProgram

	# get the indices of the currently selected keynames 

    set indices [$w.exportedkeys.prikeys curselection]
    set keytype private
    set window $w.exportedkeys.prikeys

    if {$indices == ""} {

	set indices [$w.exportedkeys.pubkeys curselection] 
	set keytype public
	set window $w.exportedkeys.pubkeys

	if {$indices == ""} {
	    bell
	    return
	}
    }

	# import each key

    foreach index $indices {

	    # get the keyname

	set keyname [$window get $index]

	    # import the key

	if {[catch {

	    exec $pgpProgram +nomanual=on +verbose=0 +batchmode +force -kx $keyname $pgpDirectory/pubring.pgp $pubFilename

	    if {$keytype == "private"} {
		exec $pgpProgram +nomanual=on +verbose=0 +batchmode +force -kx $keyname $pgpDirectory/secring.pgp $priFilename
	    }

	} result]} {

	    pgp_error $result
	    break
	}
    }

	# update the two key windows

    if {[updateKeyExportWindow main] != "0"} {
	destroy $w
	return
    }

    if {[updateMainWindow] != "0"} {
	destroy $w
	return
    }
}

# ---------------------------------------------------------------------------
# END: import/export keys
# ---------------------------------------------------------------------------

	# proc setMinimumSize and proc withdrawAndCenter
	#
	# Utilities to set a window's minimum size to its current requested
	# size and to center a window on the screen

proc setMinimumSize w {
    wm withdraw $w
    update idletasks
    set width [winfo reqwidth $w]
    set height [winfo reqheight $w]
    wm minsize $w $width $height
    wm deiconify $w
}

proc withdrawAndCenter w {
    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
}


proc showWindow {w list} {

    if {[winfo exists $w]} {
	raise $w
	focus $w
	return 0
    }

	# can not display if one of the other windows is active

    foreach i $list {
	if {$w != $i} {
	    if {[winfo exists $i]} {
		raise $i
		focus $i
		bell
		return 0
	    }
	}
    }

	# go ahead and display it

    return 1
}

# ---------------------------------------------------------------------------
#
# Routines to generate a key
#
# ---------------------------------------------------------------------------

proc generateKey {w type} {

    global pgpProgram

    if {$type == "server"} {

 	    # get the name of the server

	set server [string tolower [string trim [$w.server.entry get]]]

	    # make sure that the server name is not empty

	if {$server == ""} {
	    tk_dialog .t "Empty server name!" "You must specify a server name!" error 0 OK
	    return
	}

	    # get the IP address of the server

	if {[catch {

	    set ip [getip $server]

	} result]} {

	    tk_dialog .t "Unknown IP address!" "Unable to determine IP address of \"$server\"!" error 0 OK
	    return
	}

	    # construct the keyname

	set keyname "Agent Tcl server <$ip> <$server>"

    } else {

	    # get the keyname

	set keyname [string trim [$w.server.entry get]]

	    # make sure that the keyname is not empty

	if {$keyname == ""} {
	    tk_dialog .t "Empty user id!" "You must specify a userid!" error 0 OK
	    return
	}
    }  

	# error if the key already exists

    if {[catch {

	set list [makeKeyList]

    } result]} {

	pgp_error $result
	return
    }

    foreach i $list {
	if {[lindex $i 0] == $keyname} {
	    tk_dialog .t "Key exists!" "The key for $type \"$keyname\" has already been created!" error 0 OK
	    return
	}
    }

	# otherwise we must create the key -- first spawn a background pgp process 

    if {[catch {
 
	set pgpPid [exec $pgpProgram +force +nomanual=on -kg -u $keyname &]

    } result]} {

	pgp_error $result
	return
    }

	# then wait for the user to come back and say he's done

    tk_dialog .t "Key generation" "If you look at the terminal window in which you started up the keys utility, you will see a PGP prompt.  Please go to that window and answer the questions to complete the key generation process.  Then come back here and hit OK!" error 0 OK

	# see if the key was successfully created

    if {[catch {

	set list [makeKeyList]

    } result]} {

	pgp_error $result
	return
    }

    set exists 0

    foreach i $list {
	if {[lindex $i 0] == $keyname} {
	    set exists 1
	    break
	}
    }

	# error or success message depending on key existence

    if {$exists == 0} {

	catch {
	    exec kill $pgpPid
	}

	tk_dialog .t "Key failure!" "The key for $type \"$keyname\" was NOT created successfully!" error 0 OK

    } else {

	updateMainWindow
	tk_dialog .t "Success!" "The key for $type \"$keyname\" has been created!" info 0 OK
	clearKey $w 
    }
}

proc clearKey w {
  $w.server.entry delete 0 end 
  focus $w.server.entry
}

proc getKey {w type} {

    toplevel $w 

    if {$type == "server"} {
        set labelId "Server:"
	wm title $w "Generate a server key "
    } else {
	set labelId "User id:"
	wm title $w "Generate a user key "
    }

	# fill in the window
	# 1. entry box for the server name or userid

    frame $w.server -relief raised -bd 1
    label $w.server.label -text $labelId -width 20 -anchor nw 
    entry $w.server.entry -relief sunken 
    pack $w.server.label -side left 
    pack $w.server.entry -side right -fill x -expand true 
    bind $w.server.entry <Return> "generateKey $w $type"
    pack $w.server -fill x 

	# buttons

    frame $w.buttons -relief raised -bd 1
    button $w.ok -text "Create" -command "generateKey $w $type"
    button $w.clear -text "Clear" -command "clearKey $w"
    button $w.cancel -text "Done!" -command "destroy $w"
    pack $w.ok -in $w.buttons -side left -expand 1 -padx 3m -pady 2m
    pack $w.clear -in $w.buttons -side left -expand 1 -padx 3m -pady 2m
    pack $w.cancel -in $w.buttons -side left -expand 1 -padx 3m -pady 2m
    pack $w.buttons -fill x
   
	# set the server

    focus $w.server.entry
}

proc getServerKey {} {

    global windows

        # are we allowed to display the window?

    if {![showWindow .servertop $windows]} {
	return
    }

	# create the dialog box

    getKey .servertop server 
}

proc getUserKey {} {

    global windows

        # are we allowed to display the window?

    if {![showWindow .usertop $windows]} {
	return
    }

	# create the dialog box

    getKey .usertop user
}

proc setMe {} {

    global agentDirectory passDone

	# get the index of the key

    set index [.list.prikeys curselection]

    if {$index == ""} {

	if {[.list.pubkeys curselection] != ""} {
	    tk_dialog .t "Must have private key" \
	    "You must select a keyname for which you have both keys!" \
	    error 0 OK 
	}

	bell
	return
    }

	# get the key

    set keyname [.list.prikeys get $index]
    set keytype private
 
	# confirmation

    set code [
	tk_dialog .t "Confirm" \
	"You are about to set YOUR Agent Tcl identity to  \"$keyname\".  Do you want to proceed?" \
	error 1 Yes NO!
    ]

    if {$code != 0} {
	return
    }

	# see if you want the pass phrase on disk

    set code [

	option add *met.msg.wrapLength 8i 100

	tk_dialog .met "Pass phrase on disk" \
	    "Do you want to store your PGP pass phrase on disk (in a file called\
	     $agentDirectory/passphrase that will only be readable by you and the root user)?\
	     \n\nThis is convenient but it is also an\
	     INCREASED SECURITY RISK." \
	     error 1 Yes NO!
    ]

    if {$code != 0} {
	set phraseOnDisk 0
    } else {
	set phraseOnDisk 1
    }

	# if you do want the pass phrase on disk, we need to get that phrase

    if {$phraseOnDisk} {
		
        toplevel .phrase
        wm title .phrase "Pass phrase for disk file"

	    # entry box for the pass phrase

        frame .phrase.passes -relief raised -bd 1
    	frame .phrase.pass1
    	label .phrase.pass1.label -text "Pass phrase: " -width 20 -anchor nw 
    	entry .phrase.pass1.entry -relief sunken -show false
    	pack .phrase.pass1.label -side left -anchor nw
    	pack .phrase.pass1.entry -side right -fill x -expand true 
    	bind .phrase.pass1.entry <Return> "set passDone 1"
    	pack .phrase.pass1 -in .phrase.passes -fill x 

	    # entry box for the pass phrase copy

	frame .phrase.pass2 
	label .phrase.pass2.label -text "Pass phrase (again): " -width 20 -anchor nw 
	entry .phrase.pass2.entry -relief sunken -show false
	pack .phrase.pass2.label -side left -anchor nw
	pack .phrase.pass2.entry -side right -fill x -expand true 
	bind .phrase.pass2.entry <Return> "set passDone 1"
	pack .phrase.pass2 -in .phrase.passes -fill x 
	pack .phrase.passes -fill x

	    # buttons

	frame .phrase.buttons -relief raised -bd 1
	button .phrase.ok -text "Okay" -command "set passDone 1"
	button .phrase.cancel -text "Forget it!" -command "set passDone 0"
	pack .phrase.ok -in .phrase.buttons -side left -expand 1 -padx 3m -pady 2m
	pack .phrase.cancel -in .phrase.buttons -side left -expand 1 -padx 3m -pady 2m
	pack .phrase.buttons -fill both -expand 1
   
	    # set the starting entry box

	focus .phrase.pass1.entry

	    # center on screen

	withdrawAndCenter .phrase

	    # loop until canceled or correct input

	while {1} {

	    catch {unset passDone}
	    tkwait variable passDone

	        # bail if user said forget it

	    if {$passDone != 1} {
	        destroy .phrase
	        return
	    }

		# otherwise get the first pass phrase

	    set phrase [string trim [.phrase.pass1.entry get]]

	    if {$phrase == ""} {
		tk_dialog .t "Empty pass phrase!" "You must type in the pass phrase!" error 0 OK
		continue
	    }

		# compare the two pass phrases

	    if {$phrase != $phrasecopy} {
	        tk_dialog .t "Phrase mismatch!" "The two copies of the pass phrase are not the same!  Please try again." error 0 OK
	        continue
	    }

	    break
	}

	destroy .phrase
    }

	# get the directory ready

    if {[readyDirectory $agentDirectory] != 0} {
	return
    }

	# make the identity file

    if {[catch {

	set fd [open $agentDirectory/keyname w 0600]

    } result]} {

	tk_dialog .t "Can't create!" "Unable to create keyname file \"$agentDirectory/keyname\" ($result)!" error 0 OK
	return
    }

    puts $fd $keyname
    close $fd

	# make the pass phrase file

    if {$phraseOnDisk} {

	if {[catch {

	    set fd [open $agentDirectory/passphrase w 0600]

	} result]} {

	    catch {exec rm $agentDirectory/identity}
	    tk_dialog .t "Can't create!" "Unable to create pass phrase file \"$agentDirectory/passphrase\" ($result)!" error 0 OK
	    return
	}

        puts $fd $phrase
	close $fd

    } else {

	catch {exec rm $agentDirectory/passphrase}
    }

	# success

    tk_dialog .t "Success!" "Your agent identity is now \"$keyname\"!" error 0 OK
}


    # do tilde substitution in agentDirectory

set agentDirectory [tildeSubstitution $agentDirectory]

    # get the PGP directory ready

if {[readyPgpDirectory $pgpDirectory] != 0} {
    exit
}

    # create the main window

if {[createMainWindow] != 0} {
    exit
}
