/* Agent Tcl
   Bob Gray
   26 June 1995

   genConnection.cc

   This file implements the classes that handle message-oriented connections.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "platSigio.h"
#include "suppDBuffer.h"
#include "suppDString.h"
#include "mesgConnection.h"
#include "mesgPort.h"
#include "mesgTcpip.h"

extern "C" {
unsigned threadSelf(void);
}

Connection::Connection (void):
    m_status (Connection::e_INITIALIZED)
{
    // empty
}

Connection::~Connection () {
    // empty
}

SocketConnection::SocketConnection (void):
    Connection (),
    AsyncSupport (),
    BlockingSupport (),
    m_trap (NULL),
    m_handler (NULL),
    m_sockfd (-1),
    m_errorOnLast (e_FALSE),
    m_fdDisposition (e_Close)
{
    // empty
}

SocketConnection::SocketConnection (int sockfd, FdDisposition fdDisposition):
    Connection (),
    AsyncSupport (),
#ifdef FIX_LATER
    // we are assuming that sockfd is in blocking mode
#endif
    BlockingSupport (),
    m_trap (NULL),
    m_handler (NULL),
    m_sockfd (sockfd),
    m_errorOnLast (e_FALSE),
    m_fdDisposition (fdDisposition)
{
    FDMASK_ZERO (m_set);
    FDMASK_SET (m_sockfd, m_set);
    m_index = FDMASK_INDEX (m_sockfd);
    Connection::setStatus (Connection::e_CONNECTED);
}

SocketConnection::~SocketConnection () {

    turnBackgroundOff ();

#ifdef FIX_LATER
	/* 
	 * The close part should be in the constructors of the more specific
	 * classes so that we can call the correct close routine.
	 */
#endif

    if (m_fdDisposition == e_Close) {
	if (m_sockfd >= 0) {
	    tcpip_close (m_sockfd);
	}
    }
}

Connection::ConnectionErrors SocketConnection::read (int n, char *buffer, int &nread, struct timeval *stop)
{
    TcpipErrors tcpipRc;
    Connection::ConnectionErrors rc;

	// make sure that we are in a connected state

    assert (Connection::getStatus() == Connection::e_CONNECTED);

	// reset the socket error flag if necessary since we do not want 
	// "leftover" errors affecting the current operation -- e.g., if we 
	// wrote to the socket while it was disconnected, we do not want to 
	// get an EPIPE error during the read; instead we want to get any 
	// available data and then an EOF.

    if (m_errorOnLast) {
	tcpip_resetErrorFlag (m_sockfd);
    }

	// make sure the socket is blocking if we have a timeout, nonblocking
	// if we do not have a timeout

    if (stop == NULL) {

	if (!isBlocking()) {
	    makeBlocking();
	}

    } else {

	if (isBlocking()) {
	    makeNonblocking();
	}
    }

	// read the data with the blocking or timeout version of tcpip_readn

    if (stop == NULL) {
	tcpipRc = tcpip_blockingReadn (m_sockfd, buffer, n, nread);
    } else {
	tcpipRc = tcpip_readn (m_sockfd, buffer, n, stop, nread);
    }

	// determine the return code

    if (tcpipRc == e_TCPIP_TIMEOUT) {
	rc = Connection::e_Timeout;
	m_errorOnLast = e_FALSE;
    } else if ((tcpipRc != e_TCPIP_OK) || (nread < n)) {
	rc = Connection::e_LostConnection;
	m_errorOnLast = e_TRUE;
    } else {
	rc = Connection::e_Ok;
	m_errorOnLast = e_FALSE;
    }

    return (rc);
}

Connection::ConnectionErrors SocketConnection::write (int n, const char *buffer, int &nwritten, struct timeval *stop)
{
    TcpipErrors tcpipRc;
    Connection::ConnectionErrors rc;

	// make sure that we are in a connected state

    assert (Connection::getStatus() == Connection::e_CONNECTED);

	// reset the socket error flag if necessary since we do not want 
	// "leftover" errors affecting the current operation

    if (m_errorOnLast) {
	tcpip_resetErrorFlag (m_sockfd);
    }

	// make sure the socket is blocking if we have a timeout, nonblocking
	// if we do not have a timeout

    if (stop == NULL) {

	if (!isBlocking()) {
	    makeBlocking();
	}

    } else {

	if (isBlocking()) {
	    makeNonblocking(); 
	}
    }

	// write the data with the blocking or timeout version of tcpip_writen

    if (stop == NULL) {
	tcpipRc = tcpip_blockingWriten (m_sockfd, buffer, n, nwritten);
    } else {
	tcpipRc = tcpip_writen (m_sockfd, buffer, n, stop, nwritten);
    }

	// set the return code

    if (tcpipRc == e_TCPIP_TIMEOUT) {
	rc = Connection::e_Timeout;
	m_errorOnLast = e_FALSE;
    } else if ((tcpipRc != e_TCPIP_OK) || (nwritten < n)) {
	rc = Connection::e_LostConnection;
	m_errorOnLast = e_TRUE;
    } else {
	rc = Connection::e_Ok;
	m_errorOnLast = e_FALSE;
    }	

    return (rc);
}

void SocketConnection::sigioAlarm (int, int flags)
{
    Connection::ConnectionStatus status = getStatus();

	// have we just connected? 

    if ((status == Connection::e_WAITING_TO_CONNECT) && (flags & SigioTrap::e_WRITE_READY)) {
	setStatus (Connection::e_CONNECTED);
	m_trap -> connectionAlarm (*this, Connection::e_CONNECTED);
	m_handler -> changeFlags (SigioTrap::e_READ_READY);
	return;
    }

	// have we just gotten a connection request?

    if ((status == Connection::e_WAITING_TO_ACCEPT) && (flags & SigioTrap::e_READ_READY)) {
	setStatus (Connection::e_ACCEPT_READY);
	m_trap -> connectionAlarm (*this, Connection::e_ACCEPT_READY);
	return;
    }

	// are we reading from a connected socket?

    if ((status == Connection::e_CONNECTED) && (flags & SigioTrap::e_READ_READY)) {
	m_trap -> connectionAlarm (*this, Connection::e_READ_READY);
	return;
    }
	
	// ignore all other cases right now

    return;
}

void SocketConnection::makeBlocking (void)
{
    if (!isBlocking()) {
	::BlockingMode oldMode;
	BlockingSupport::makeBlocking();
	(void) tcpip_setBlockingMode (m_sockfd, ::e_Blocking, oldMode);
    }
}

void SocketConnection::makeNonblocking (void)
{
    if (isBlocking()) {
	::BlockingMode oldMode;
	BlockingSupport::makeNonblocking();
	(void) tcpip_setBlockingMode (m_sockfd, ::e_NonBlocking, oldMode);
    }
}

void SocketConnection::turnBackgroundOn (ConnectionTrap *trap, MonitorGroup group)
{
    int flags;

	// assertions on the parameters

    assert (trap != NULL);
    assert (m_trap == NULL);
    assert (m_handler == NULL);

	// call the AsyncSupport routine

    AsyncSupport::turnBackgroundOn (trap, group);

	// remember the trap since we are going to redirect through ours

    m_trap  = trap;

	// set up the handler

    flags = SigioTrap::e_READ_READY;

    if (getStatus() == Connection::e_WAITING_TO_CONNECT) {
	flags |= SigioTrap::e_WRITE_READY;
    }

    m_handler = new SigioHandler (m_sockfd, this, flags, group); 
}

void SocketConnection::turnBackgroundOff (void)
{
    if (isBackgroundOn()) {
	AsyncSupport::turnBackgroundOff();
	DELETE_OBJECT(m_handler);
    }
}

Connection::ConnectionErrors SocketConnection::checkReadyStatus (int /*inFlags*/, int &outFlags)
{
    int rc;
    struct timeval tv;
    register UINT_32 i;
    fd_mask readSet[MASK_SIZE];
    fd_mask writeSet[MASK_SIZE];

	/* default to nothing ready */

    outFlags = 0;

	/* check for both read and write */ 

    for (i = 0; i < MASK_SIZE; ++i) {
	readSet[i] = m_set[i];
	writeSet[i] = m_set[i];
    }

    do {
        tv.tv_sec  = 0;
        tv.tv_usec = 0;
	rc = select (m_sockfd + 1, (SELECT_MASK *) &readSet[0], (SELECT_MASK *) &writeSet[0], (SELECT_MASK *) NULL, &tv);
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

    if (rc < 0) {
	return Connection::e_SystemError;
    }

	/* success so set the flags */

    if (readSet[m_index] > 0) {
	outFlags |= SigioTrap::e_READ_READY;
    } 
	
    if (writeSet[m_index] > 0) {
	outFlags |= SigioTrap::e_WRITE_READY;
    }

    return Connection::e_Ok;
}

TcpipClientConnection::TcpipClientConnection (void):
    SocketConnection ()
{
    // empty
}

TcpipClientConnection::~TcpipClientConnection ()
{
    // empty
}

Connection::ConnectionErrors TcpipClientConnection::connect (const MachineId &machine, UINT_32 port, struct timeval stop)
{
    Connection::ConnectionErrors returnCode;

	// make sure that we are in the initialized state

    assert (Connection::getStatus() == Connection::e_INITIALIZED);

	// remember the machine and port

    m_machine = machine;
    m_port    = port;

	// mark that we are starting the connection

    Connection::setStatus (Connection::e_WAITING_TO_CONNECT);

	// get the IP address of the machine 

    TcpipErrors tcpipRc;

    if ((tcpipRc = tcpip_getIP (m_machine, stop)) == e_TCPIP_TIMEOUT) {

	returnCode = Connection::e_Timeout;

    } else if (tcpipRc != e_TCPIP_OK) {

	returnCode = Connection::e_NotFound;

    } else if (tcpip_socket(m_sockfd) != e_TCPIP_OK) {

	returnCode = Connection::e_SystemError;

    } else {

	FDMASK_ZERO (m_set);
	FDMASK_SET (m_sockfd, m_set);
	m_index = FDMASK_INDEX (m_sockfd);

        RemotePort serverInfo (m_machine.getName().value(), m_machine.getIp(), m_port);
        tcpipRc = tcpip_connect (m_sockfd, serverInfo, stop);

        switch (tcpipRc) {

	    case e_TCPIP_OK:	

	        returnCode = Connection::e_Ok;
	        break;

	    case e_TCPIP_NOT_FOUND:

	        returnCode = Connection::e_NotFound;
	        break;

	    case e_TCPIP_TIMEOUT:

	        returnCode = Connection::e_Timeout;
	        break;

	    default:

	        returnCode = Connection::e_NoServer;
		break;
        }
    }

	// set the connection status 

    if (returnCode == Connection::e_Ok) {
	Connection::setStatus (Connection::e_CONNECTED);
    } else {
	Connection::setStatus (Connection::e_FAILED);
    }

	// done 

    return (returnCode);
}

Connection::ConnectionErrors TcpipClientConnection::asyncConnect (const MachineId &machine, UINT_32 port)
{
    Connection::ConnectionErrors returnCode;

	// assertions

    assert (machine.getIp() != UNKNOWN_IP);

	// make sure that we are in the initialized state

    assert (Connection::getStatus() == Connection::e_INITIALIZED);

	// remember the machine and port

    m_machine = machine;
    m_port    = port;

	// mark that we are starting the connection

    Connection::setStatus (Connection::e_WAITING_TO_CONNECT);

	// make the connection

    if (tcpip_socket(m_sockfd) != e_TCPIP_OK) {

 	returnCode = Connection::e_SystemError;	

    } else {

	FDMASK_ZERO (m_set);
	FDMASK_SET (m_sockfd, m_set);
	m_index = FDMASK_INDEX (m_sockfd);

        RemotePort serverInfo (m_machine.getName().value(), m_machine.getIp(), m_port);
        TcpipErrors tcpipRc = tcpip_async_connect (m_sockfd, serverInfo);

        switch (tcpipRc) {

	    case e_TCPIP_OK:	

		returnCode = Connection::e_Ok;
	        break;

	    case e_TCPIP_PENDING:

		returnCode = Connection::e_Pending;
		break;

	    default:

		returnCode = Connection::e_NotFound;
        }
    }

	// set the connection status 

    if (returnCode == Connection::e_Ok) {
	Connection::setStatus (Connection::e_CONNECTED);
    } else if (returnCode != Connection::e_Pending) {
	Connection::setStatus (Connection::e_FAILED);
    }

	// done 

    return (returnCode);
}

TcpipServerConnection::TcpipServerConnection (void):
    SocketConnection ()
{
    // empty 
}

TcpipServerConnection::~TcpipServerConnection ()
{
    // empty
}

UnixClientConnection::UnixClientConnection (void):
    SocketConnection ()
{
    // empty 
}

UnixClientConnection::~UnixClientConnection () {
    // empty
}

Connection::ConnectionErrors UnixClientConnection::connect (const DynamicString &socketFilename, struct timeval stop) 
{ 
    Connection::ConnectionErrors returnCode;

	// make sure that we are in the initialized state

    assert (Connection::getStatus() == Connection::e_INITIALIZED);

	// mark that we are starting the connection

    Connection::setStatus (Connection::e_WAITING_TO_CONNECT);

	// remember the socket filename

    m_socketFilename = socketFilename; 

	// construct and connect the socket

    if (unix_socket(m_sockfd) != e_TCPIP_OK) {

	returnCode = Connection::e_SystemError;

    } else {

	FDMASK_ZERO (m_set);
	FDMASK_SET (m_sockfd, m_set);
	m_index = FDMASK_INDEX (m_sockfd);

        TcpipErrors tcpipRc = unix_connect (m_sockfd, m_socketFilename.value(), stop);

        switch (tcpipRc) {

	    case e_TCPIP_OK:	

		returnCode = Connection::e_Ok;
	        break;

	    case e_TCPIP_NOT_FOUND:

		returnCode = Connection::e_NotFound;
	        break;

	    case e_TCPIP_TIMEOUT:

		returnCode = Connection::e_Timeout;
	        break;

	    default:
	
		returnCode = Connection::e_NoServer;
	}
    }

	// set the connection status 

    if (returnCode == Connection::e_Ok) {
	Connection::setStatus (Connection::e_CONNECTED);
    } else {
	Connection::setStatus (Connection::e_FAILED);
    }

	// done

    return (returnCode);
}

UnixServerConnection::UnixServerConnection (void):
    SocketConnection ()
{
    // empty 
}

UnixServerConnection::~UnixServerConnection () {
    // empty
}

FileConnection::FileConnection (FileCallbacks &callbacks):
    m_callbacks (callbacks)
{
    // empty
}

FileConnection::~FileConnection () {
    // empty
}

Connection::ConnectionErrors FileConnection::checkReadyStatus (int inFlags, int &outFlags) 
{
    outFlags = inFlags;
    return Connection::e_Ok;
}

Connection::ConnectionErrors FileConnection::read (int n, char *buffer, int &nread, struct timeval * /*stop*/)
{
    Connection::ConnectionErrors rc = m_callbacks.read (n, buffer, nread);
    return (rc);
}

Connection::ConnectionErrors FileConnection::write (int n, const char *buffer, int &nwritten, struct timeval * /*stop*/)
{
    Connection::ConnectionErrors rc = m_callbacks.write (n, buffer, nwritten);
    return (rc);
}
