/* Agent Tcl
   Bob Gray
   16 July 1998

   mesgConnection.h

   This file defines classes that wrap file and socket descriptors.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _MESG_CONNECTION_H
#define _MESG_CONNECTION_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _PLAT_PORTING_H
#include "platPorting.h"	// fd_mask and abort_with_message
#endif
#ifndef _PLAT_EXCLUSION_H
#include "platExclusion.h"	// MonitorGroup
#endif
#ifndef _PLAT_SIGIO_H
#include "platSigio.h"		// SigioHandler
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"		// UINT_32
#endif
#ifndef _SUPP_DSTRING_H
#include "suppDString.h"	// DynamicString
#endif
#ifndef _MESG_MACH_ID_H
#include "mesgMachId.h"		// MachineId
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"		// BOOLEAN
#endif

    /* forward declaration to allow compilation */

class Connection;
class DynamicBuffer;

    /* general connection */

class Connection
{
    public:

	enum ConnectionStatus {
	    e_INITIALIZED		= 0x0,
	    e_WAITING_TO_CONNECT 	= 0x1,
	    e_WAITING_TO_ACCEPT 	= 0x2,
	    e_ACCEPT_READY		= 0x3,
	    e_CONNECTED			= 0x4,
	    e_READ_READY		= 0x5,
	    e_WRITE_READY		= 0x6,
	    e_BOTH_READY		= 0x7,
	    e_FAILED			= 0x8
	};

 	enum ConnectionErrors {
	    e_Ok			= 0x0,
	    e_LostConnection		= 0x1,
	    e_SystemError		= 0x2,
	    e_Timeout			= 0x3,
	    e_NotFound		 	= 0x4,
	    e_NoServer			= 0x5,
	    e_Pending			= 0x6,
	    e_CantRead			= 0x7,
	    e_CantWrite			= 0x8
	};

	    // constructor and destructor

	Connection (void);
	virtual ~Connection ();

	    /* see if the connection is available for reading and writing */

	virtual ConnectionErrors checkReadyStatus 
	    (int inFlags, int &outFlags) = 0;

	    /* read and write from the connection */

	virtual ConnectionErrors read 
	    (int n, char *buffer, int &nread, 
	     struct timeval *stop = NULL) = 0;

	virtual ConnectionErrors write
	    (int n, const char *buffer, int &nwritten, 
	     struct timeval *stop = NULL) = 0;

	    /* get the file descriptor associated with connection if there is one */

	virtual int getFd (void) = 0;

	    /* get the connection status */

	ConnectionStatus getStatus (void) {
	    return (m_status);
	}
	
    protected:

	    /* set the connection status */

	void setStatus (ConnectionStatus status) {
	    m_status = status;
	}

    private:

	ConnectionStatus m_status;
};

    // connection callback

class ConnectionTrap
{
    public:

	virtual void connectionAlarm 
	    (Connection &connection, Connection::ConnectionStatus status) = 0;
};

    // file access callbacks

class FileCallbacks
{ 
    public:

	virtual Connection::ConnectionErrors read 
	    (int n, char *buffer, int &nread) = 0;

	virtual Connection::ConnectionErrors write
	    (int n, const char *buffer, int &nwritten) = 0;
};

    // file connection

class FileConnection: public Connection
{
    protected:

	FileCallbacks &m_callbacks;      // callbacks for file access

    public:

	    // constructor and destructor

	FileConnection (FileCallbacks &callbacks);
	virtual ~FileConnection ();

	    /* see if the connection is available for reading and writing */

	ConnectionErrors checkReadyStatus
	    (int inFlags, int &outFlags);

	    /* read and write from the connection */

	ConnectionErrors read
	    (int n, char *buffer, int &nread, 
	     struct timeval *stop = NULL);

	ConnectionErrors write
	    (int n, const char *buffer, int &nwritten, 
	     struct timeval *stop = NULL);

	    /* get the file descriptor associated with the connection */

	int getFd (void) {
#ifdef FIX_LATER
	    // we do not need this fd yet
#else
	    abort_with_message ("FileConnection::getFd has not been implemented!");
	    return (-1);
#endif
	}
};

    // asynchronous I/O support

class AsyncSupport
{
    public:

	    // turn on and off the background handler

	virtual void turnBackgroundOn 
	    (ConnectionTrap *, MonitorGroup) {
	    m_asyncMode = e_Background;
	}
	 
	virtual void turnBackgroundOff (void) {
	    m_asyncMode = e_NoBackground;
	}

	    // see if the background handler is on

	BOOLEAN isBackgroundOn (void) {
	    return ((m_asyncMode == e_Background) ? e_TRUE : e_FALSE);
	}

	    // constructor and virtual destructor */

	AsyncSupport (void):
	    m_asyncMode (e_NoBackground)
	{
	    // empty
	}

	virtual ~AsyncSupport (void) {
	    // empty
	}

    private:

	enum AsyncMode {
	    e_NoBackground	= 0,
	    e_Background 	= 1
	};

	AsyncMode m_asyncMode;
};

    // blocking/nonblocking support

class BlockingSupport
{
    public:

	    // see if the connection is blocking

	BOOLEAN isBlocking (void) {
	    return ((m_blockingMode == e_Blocking) ? e_TRUE : e_FALSE);
	}

	    // make the connection blocking or nonblocking

	virtual void makeBlocking (void) {
	    m_blockingMode = e_Blocking;
        }

	virtual void makeNonblocking (void) {
	    m_blockingMode = e_Nonblocking;
	}

	    // constructor and destructor

	BlockingSupport (void):
	    m_blockingMode (e_Blocking)
	{
	    // empty
	}

        virtual ~BlockingSupport () {
	    // empty
	}

    private:

	enum BlockingMode {
	    e_Blocking		= 0,
	    e_Nonblocking	= 1
	};

	BlockingMode m_blockingMode;	// blocking or nonblocking
};

    // socket connection

class SocketConnection:
    public Connection,
    public AsyncSupport,
    public BlockingSupport,
    public SigioTrap
{
    public:

	    // close or don't close the provided sockfd

	enum FdDisposition {
	    e_Close	= 0,
	    e_NoClose   = 1
	};

    protected:

	ConnectionTrap *m_trap;	        // trap to call on asynchronous I/O
	SigioHandler *m_handler;        // SIGIO handler
	int m_sockfd;		        // socket descriptor
	fd_mask m_set [MASK_SIZE];      // set for select
	fd_mask m_index;                // index of m_sockfd in set
	BOOLEAN m_errorOnLast;		// e_TRUE if an error occurred 
				        // on the last read or write operation
	FdDisposition m_fdDisposition;  // close or don't close the sockfd

	    // callback from lower-level routines

	void sigioAlarm
	    (int fd, int flags);

    public:

	    // constructor and destructor 

	SocketConnection 
	    (void);

 	SocketConnection
	    (int sockfd, FdDisposition fdDisposition = e_Close);
	
	virtual ~SocketConnection ();

	    // make the connection blocking or nonblocking

	void makeBlocking 
	    (void);

	void makeNonblocking 
	    (void);

	    // turn on and off the background handler 

	void turnBackgroundOn 
	    (ConnectionTrap *newTrap, MonitorGroup group);

	void turnBackgroundOff 
	    (void);

	    // see if the connection is available for reading and writing 

	ConnectionErrors checkReadyStatus 
	    (int inFlags, int &outFlags);

	    // read and write from the connection

	ConnectionErrors read 
	    (int n, char *buffer, int &nread, 
	     struct timeval *stop = NULL);

	ConnectionErrors write
	    (int n, const char *buffer, int &nwritten,
	     struct timeval *stop = NULL);

	    // return the file descriptor associated with the connection

        int getFd (void) {
	    return (m_sockfd);
	}
};

    // client-side TCP/IP connection

class TcpipClientConnection: public SocketConnection
{
    protected:

	MachineId m_machine;        // machine on the other end
	UINT_32 m_port;		    // port on destination machine

    public:

	    // constructors and destructor

	TcpipClientConnection 
	    (void);
	
	virtual ~TcpipClientConnection
	    ();

	    /* connect to the destination machine */

	ConnectionErrors connect 
	    (const MachineId& machine, UINT_32 port, struct timeval stop);

	ConnectionErrors asyncConnect
	    (const MachineId& machine, UINT_32 port);
};

    // client-side Unix domain connection

class UnixClientConnection: public SocketConnection
{
    protected:

	DynamicString m_socketFilename;		// Unix socket filename

    public:

	    /* constructor and destructor */

	UnixClientConnection 
	   (void);

       ~UnixClientConnection 
	   ();

	    /* connect to the destination machine */

	ConnectionErrors connect 
	    (const DynamicString& socketFilename, struct timeval stop);
};

    // server-side TCP/IP connection

class TcpipServerConnection: public SocketConnection
{
    protected:

	UINT_32 m_port;		    // port on the local machine

    public:

	    // constructors and destructor

	TcpipServerConnection 
	    (void);
	
	virtual ~TcpipServerConnection
	    ();

	    // bind to a port and accept an incoming connection

	int setup
	    (UINT_32 port);

	SocketConnection *accept
	    (struct timeval stop);
};

    // client-side Unix domain connection

class UnixServerConnection: public SocketConnection
{
    protected:

	DynamicString m_socketFilename;		// Unix socket filename

    public:

	    // constructor and destructor

	UnixServerConnection 
	   (void);

       ~UnixServerConnection 
	   ();

	    /* connect to the destination machine */

	int setup
	    (UINT_32 port);

	SocketConnection *accept
	    (struct timeval stop);
};

#endif /* _MESG_CONNECTION_H */
