/* Agent Tcl 
   Bob Gray
   26 June 1995

   mesgTcpip.cc

   This file implements the routines that handle TCP/IP connections.  These
   routines are based on the examples in [W. Richard Stevens, Unix Network
   Programming, Prentice Hall, 1990] and on the code that Saurab Nog
   developed as part of his A.I. project.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "platSystem.h"
#include "platTimers.h"
#include "platTimeval.h"
#include "suppDString.h"
#include "suppFileTable.h"
#include "suppStrings.h"
#include "mesgMachId.h"         // MachineId class
#include "mesgPort.h"		// RemotePort
#include "mesgTcpip.h"
#include "truefalse.h"

#ifdef FIX_LATER
    /*
     * The blocking, timeout and nonblocking versions of the various
     * procedures have quite a bit of code in common.  This common code
     * should be extracted into utility procedures.
     */
#endif

    /* BACKLOG for a TCP/IP connection */

#ifdef SOMAXCONN
#define TCPIP_BACKLOG SOMAXCONN
#else
#define TCPIP_BACKLOG 5
#endif

    /* maximum name length for a Unix socket */

const int MAX_NAME_LENGTH = 128;

    /* actual IP address and name of the current host */

static UINT_32 actual_ip   = 0;
static char *  actual_name = NULL;

    /* IP address "127.0.0.1" */

static UINT_32 local_ip = 0;

/* tcpip_setActualIPAndHostname

   Purpose: Set the actual IP address and name of the current host.  Any 
            reference to the actual IP address or name will be turned into a 
            reference to "localhost".  This support is needed for mobile 
            computers that may have networking turned off.  If networking is 
            turned off, references to the actual IP address fail but 
	    references to "localhost" are fine.

     Input: ip   = actual IP address of the current host
		   (UINT_32)

	    name = name of the actual host
		   (char *)
 
    Output: The procedure sets the actual IP address and name of the current 
            host.
*/

void tcpip_setActualIPAndHostname (UINT_32 ip, char *name)
{
    struct in_addr address;
    address.s_addr = ip;
    actual_ip   = ip;
    actual_name = strcpyWithAlloc (name); 
    local_ip    = (UINT_32) inet_addr ("127.0.0.1");
}

/* tcpip_getLocalhostIP

   Purpose: Return the numeric representation of the localhost IP address
            "127.0.0.1"

     Input: None

    Output: The procedure returns the numeric representaton of the localhost
            IP address "127.0.0.1".
*/
 
UINT_32 tcpip_getLocalhostIP (void) {
    
    if (local_ip == 0) {
	local_ip = (UINT_32) inet_addr ("127.0.0.1");
    }

    return (local_ip); 
}

/* tcpip_setBlockingMode

   Purpose: Turn on or off blocking I/O for a socket

     Input: sockfd  = the socket descriptor
		      (int)

	    newMode = the new mode
		      (BlockingMode)

    Output: The procedure returns e_TCPIP_CONTROL on error and e_TCPIP_OK
	    otherwise.  It fills in the oldMode argument (BlockingMode)
	    with the previous mode of the socket.
*/

TcpipErrors tcpip_setBlockingMode 
    (int sockfd, BlockingMode newMode, BlockingMode &oldMode)
{
    int oldFlags;
    int newFlags;

	// get the current socket flags 

    if ((oldFlags = fcntl (sockfd, F_GETFL, 0)) < 0) {
	return e_TCPIP_CONTROL;
    }

	// remember the old mode

    if (oldFlags & NON_BLOCKING_FLAG) {
	oldMode = e_NonBlocking;
    } else {
	oldMode = e_Blocking;
    }

	// if the old mode is different than the desired newMode,
	// we need to change the socket flags

    if (newMode != oldMode) {

	if (newMode == e_Blocking) {
	    newFlags = oldFlags & ~NON_BLOCKING_FLAG;
	} else {
	    newFlags = oldFlags | NON_BLOCKING_FLAG;
	}


	if (fcntl (sockfd, F_SETFL, newFlags) < 0) {
	    return e_TCPIP_CONTROL;
	}
    }

	// successfully changed the mode

    return e_TCPIP_OK;
}

/* tcpip_getSockname

   Purpose: Get the IP address and the port number associated with the
            LOCAL end of a socket

     Input: sockfd = socket descriptor

    Output: The procedure returns e_TCPIP_UNABLE_TO_GET_PORT on error.
            Otherwise the procedure fills in ip and port with the IP 
            address and port number and returns e_TCPIP_OK.
*/

TcpipErrors tcpip_getSockname (int sockfd, UINT_32 &ip, UINT_32 &port)
{
    TcpipErrors returnCode;
    struct sockaddr_in address;  
    socklen_t size = sizeof(sockaddr_in);

    if (getsockname(sockfd, (struct sockaddr *) &address, &size) < 0) {
	returnCode = e_TCPIP_UNABLE_TO_GET_PORT;
    } else {
	ip         = (UINT_32) address.sin_addr.s_addr;
	port       = ntohs (address.sin_port);
	returnCode = e_TCPIP_OK;
    }

    return (returnCode);
}

/* tcpip_getPeername

   Purpose: Get the IP address and the port number associated with the
            REMOTE end of a CONNECTED socket

     Input: sockfd = socket descriptor

    Output: The procedure returns e_TCPIP_UNABLE_TO_GET_PORT on error.
            Otherwise the procedure fills in ip and port with the IP 
            address and port number and returns e_TCPIP_OK.
*/

TcpipErrors tcpip_getPeername (int sockfd, UINT_32 &ip, UINT_32 &port)
{
    TcpipErrors returnCode;
    struct sockaddr_in address;  
    socklen_t size = sizeof(sockaddr_in);

    if (getpeername(sockfd, (struct sockaddr *) &address, &size) < 0) {
	returnCode = e_TCPIP_UNABLE_TO_GET_PORT;
    } else {
	ip         = (UINT_32) address.sin_addr.s_addr;  
	port       = ntohs (address.sin_port);
	returnCode = e_TCPIP_OK;
    }

    return (returnCode);
}

/* tcpip_getIP

   Purpose: This procedure gets the IP address of the given machine *but* 
            fails with an error if it is unable to get the IP address before
            the specified stop time.  

#ifdef FIX_LATER
	    This procedure does not actually timeout.  It waits until
	    gethostbyname returns, either successfully or unsuccessfully.
#endif
	    
     Input: machine = machine name and IP address
		      (class MachineId &) 

	    stop    = wall time at which to give up
		      (struct timeval)
 
    Output: The procedure returns one of the following codes.

	    e_TCPIP_TIMEOUT	= timeout expired before IP address was found
	    e_TCPIP_NOT_FOUND	= unable to determine IP address
	    e_TCPIP_OK		= success

	    On success machine.ip is filled in with the IP address of the
	    machine.
*/

#ifdef MULTIPLE_THREADS

    /* lock for mututal exclusion around the gethostbyname call */
    /* and the flag that makes sure that we initialize the lock */
    /* only once                                                */

static MutexLock *s_gethostbynameLock = 
    (MutexLock *) NULL;

static MonitorOnce::OnceControl s_gethostbynameLockInitializerControl = 
    MonitorOnce::e_NEVER;

class GethostbynameLockInitializer: public OnceTrap
{
public:

    void initialize (void) {
	s_gethostbynameLock = new MutexLock();
    }
};

#endif

TcpipErrors tcpip_getIP (MachineId& machine, struct timeval stop)
{
    int length;
    UINT_32 ip;
    char *s_name;
    struct hostent *serverInfo;
    TcpipErrors rc = e_TCPIP_OK;

	/* assertions on the parameters */

    assert ((machine.getName().isEmpty() == e_FALSE) || (machine.getIp() != UNKNOWN_IP));
    assert ((stop.tv_sec >= 0) && (stop.tv_usec >= 0));

#ifdef MULTIPLE_THREADS

	/* create the lock if needed */

    if (s_gethostbynameLockInitializerControl != MonitorOnce::e_DONE) {
	GethostbynameLockInitializer gethostbynameLockInitializer;
	MonitorOnce::doOnce (s_gethostbynameLockInitializerControl, 
	    gethostbynameLockInitializer);
    }

#endif

	/* do nothing if we know the IP address */

    if (machine.getIp() != UNKNOWN_IP) {
	return e_TCPIP_OK;
    }

	/* see if we have something to get the IP address of */

    if (((s_name = stripWhitespaceWithAlloc (machine.getName().value())) == NULL) || (*s_name == '\0')) { 

	return e_TCPIP_NOT_FOUND;

    } else {

	    /* see if the name corresponds to the current host */

	if (actual_name != NULL) {

	    length = strlen (s_name);

	    if (!strncmp (s_name, actual_name, length)) {
		if ((s_name[length-1] == '.') || (actual_name[length] == '.') || (actual_name[length] == '\0')) {
		    machine.setIp (actual_ip);
		    DELETE_ARRAY_OBJECT(s_name);
		    return e_TCPIP_OK;
		}
	    }
	}

	    /* see if the name is an ASCII represention of an IP */ 

	if ((ip = (UINT_32) inet_addr (s_name)) != INADDR_NONE) {
	    machine.setIp (ip);
	    DELETE_ARRAY_OBJECT(s_name);
	    return e_TCPIP_OK;
	}

	    /* get the IP address */

#ifdef FIX_LATER
	    /* A lock around the gethostbyname call is a bad way to  */
	    /* prevent problems since we effectively serialize all   */
	    /* requests for IP addresses -- we need to make a        */
	    /* gethostbyname_r for Java (and use the gethostbyname_r */
	    /* available in pthreads and other standard thread       */
	    /* packages)                                             */
#endif

#ifdef MULTIPLE_THREADS
	s_gethostbynameLock -> acquire();
#endif
	
	serverInfo = gethostbyname (s_name);

	if (serverInfo != NULL) {

	    memcpy ((void *) &ip, (void *) serverInfo -> h_addr, sizeof(UINT_32));
	    machine.setIp (ip);
	    rc = e_TCPIP_OK;

	} else {

	    rc = e_TCPIP_NOT_FOUND;
	}

#ifdef MULTIPLE_THREADS
	s_gethostbynameLock -> release();
#endif
    }

	/* done */

    DELETE_OBJECT(s_name);
    return (rc);
}
      
/* tcpip_getActualHostname

   Purpose: Get the name of the current host

     Input: None

    Output: The procedure returns NULL if it is unable to get the host name.
            Otherwise the procedure returns a pointer to a dyanmically
            allocated string that contains the hostname.
*/

char *tcpip_getActualHostname (void)
{
    char *nameCopy;
    char name[MAX_NAME_LENGTH];

    if (gethostname (name, MAX_NAME_LENGTH) < 0) {
	return NULL;
    }

    nameCopy = new char [strlen(name) + 1];
    strcpy (nameCopy, name);

    for (char *sp = nameCopy; *sp != '\0'; ++sp) {
	if ((*sp >= 'A') && (*sp <= 'Z')) {
	    *sp = *sp - 'A' + 'a';
	}
    }

    return (nameCopy);
}

/* tcpip_socket

   Purpose: This function creates a socket.

     Input: None

   Purpose: The procedure returns e_TCPIP_UNABLE_TO_CREATE if it is unable to
            create the socket.  Otherwise the procedure loads the socket
            descriptor into sockfd (int) and returns e_TCPIP_OK.
*/

TcpipErrors tcpip_socket (int &sockfd)
{
    int flag = 1;

	/* create the socket */

    if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
	return e_TCPIP_UNABLE_TO_CREATE;
    }

	/* turn on the TCP_NODELAY flag */

    if (setsockopt (sockfd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int)) < 0) {
	close (sockfd);
	return e_TCPIP_UNABLE_TO_CREATE;
    }

	/* add the socket to the file table */

    if (FileTable::isFileRecordingOn()) {
	FileTable::add (sockfd, FileTable::e_TCPIP_SOCKET);
    }

    return e_TCPIP_OK;
}

/* unix_socket

   Purpose: This function creates a Unix domain socket.

     Input: None

    Output: The procedure returns e_TCPIP_UNABLE_TO_CREATE if it is unable to
            create the socket.  Otherwise the procedure loads the socket
            descriptor into sockfd (int) and returns e_TCPIP_OK.
*/

TcpipErrors unix_socket (int &sockfd)
{
    TcpipErrors rc;

    if ((sockfd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0) {

	rc = e_TCPIP_UNABLE_TO_CREATE;

    } else {

	if (FileTable::isFileRecordingOn()) {
	    FileTable::add (sockfd, FileTable::e_UNIX_SOCKET);
  	}

	rc = e_TCPIP_OK;
    }

    return (rc);
}
 
/* tcpip_async_connect 

   Purpose: Connect a socket to a port on a remote machine wihout waiting
	    for the connection to complete here

     Input: sockfd    = socket descriptor
		        (int)

	    port      = machine name and IP address and the port number
			(struct RemotePort &).  IP address must be known!

    Output: The procedure returns one of the following codes.

            e_TCPIP_UNABLE_TO_CONNECT = connection request was rejected
	    e_TCPIP_PENDING	      = connection is now in progress
	    e_TCPIP_OK                = success
*/

TcpipErrors tcpip_async_connect (int sockfd, RemotePort& port)
{
    int rc;
    TcpipErrors returnCode;
    struct sockaddr_in serverAddress;    

	/* assertions */

    assert (port.ip != UNKNOWN_IP);

	/* use "localhost" for the local host */

    if (port.ip == actual_ip) {
	printf ("using localhost!!!\n");
	port.ip = local_ip;
    }

	/* fill in the server address */

    memset ((void *) &serverAddress, 0, sizeof (sockaddr_in));
    memcpy ((void *) &serverAddress.sin_addr, (void *) &port.ip, sizeof(UINT_32));
    serverAddress.sin_family = AF_INET;
    serverAddress.sin_port   = htons (port.port);

	/* mark the socket as nonblocking */

    BlockingMode oldMode;
    (void) tcpip_setBlockingMode (sockfd, e_NonBlocking, oldMode);
    (void) tcpip_resetErrorFlag (sockfd);

	/* start the asynchronous connection */

    do {
	rc = connect (sockfd, (struct sockaddr *) &serverAddress, sizeof(sockaddr_in));
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	/* see if the connection has completed already */

    if (rc == 0) {
	returnCode = e_TCPIP_OK;
    } else if ((errno == EWOULDBLOCK) || (errno == EINPROGRESS)) {
	returnCode = e_TCPIP_PENDING;
    } else {
	returnCode = e_TCPIP_UNABLE_TO_CONNECT;
    }

	/* mark the socket as blocking */

    (void) tcpip_setBlockingMode (sockfd, oldMode, oldMode);
    return (returnCode);
}

/* tcpip_connect 

   Purpose: Connect a socket to a port on a remote machine

     Input: sockfd    = socket descriptor
		        (int)

	    port      = machine name and IP address and the port number
			(struct RemotePort &)

	    stop      = wall time at which to abandon the connection attempt
		        (struct timeval)
						
    Output: The procedure returns one of the following codes.

            e_TCPIP_NOT_FOUND         = unable to determine network location of destination machine
	    e_TCPIP_TIMEOUT           = timeout expired before connection
            e_TCPIP_UNABLE_TO_CONNECT = connection request was rejected
	    e_TCPIP_OK                = success
	    e_TCPIP_CONTROL           = unable to change socket characteristics
*/

TcpipErrors tcpip_connect (int sockfd, RemotePort& port, struct timeval stop)
{
    int rc;
    int serrno;
    TcpipErrors tcpipRc;
    TcpipErrors returnCode;
    socklen_t serrnoLen = sizeof(int);
    struct timeval timeout; 
    fd_mask writeSet[MASK_SIZE];
    struct sockaddr_in serverAddress;    

	/* convert the machine name to an IP number */

    MachineId machine (port.machine, port.ip);

    if ((tcpipRc = tcpip_getIP (machine, stop)) != e_TCPIP_OK) { 
	returnCode = tcpipRc;
	return (returnCode);
    }

	/* use "localhost" for the local host */

    if (machine.getIp() == actual_ip) {
	machine.setIp (local_ip);
    }

	/* fill in the server address */

    UINT_32 ip = machine.getIp();
    memset ((void *) &serverAddress, 0, sizeof (sockaddr_in));
    memcpy ((void *) &serverAddress.sin_addr, (void *) &ip, sizeof(UINT_32));
    serverAddress.sin_family = AF_INET;
    serverAddress.sin_port   = htons (port.port);

	/* mark the socket as nonblocking */

    BlockingMode oldMode;
    (void) tcpip_setBlockingMode (sockfd, e_NonBlocking, oldMode);
    (void) tcpip_resetErrorFlag (sockfd);

	/* start the connection */

    do {
	rc = connect (sockfd, (struct sockaddr *) &serverAddress, sizeof(sockaddr_in));
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	/* wait for the connection to complete if necessary */

    if (rc == 0) {

	returnCode = e_TCPIP_OK;

    } else {

	if ((errno == EWOULDBLOCK) || (errno == EINPROGRESS)) {

	    do {
	        FDMASK_ZERO (writeSet);
	        FDMASK_SET (sockfd, writeSet);
	        timeout = TimevalUtil::subTimevals (stop, SystemUtil::getCurrentWall());
	        rc = select (sockfd + 1, (SELECT_MASK *) NULL, (SELECT_MASK *) &writeSet[0], (SELECT_MASK *) NULL, &timeout);
	    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    if (rc < 0) {
	   	returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	    } else if (rc == 0) {
		returnCode = e_TCPIP_TIMEOUT;
	    } else if (getsockopt (sockfd, SOL_SOCKET, SO_ERROR, (char *) &serrno, &serrnoLen) < 0) {
	 	returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	    } else if (serrno != 0) {
		returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	    } else {
		returnCode = e_TCPIP_OK;
	    }

	} else {

	    returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	}
    }

	/* turn on the TCP_NODELAY flag if everything is okay */

    if (returnCode == e_TCPIP_OK) {

	int flag = 1;

	if (setsockopt (sockfd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int)) < 0) {
	    returnCode = e_TCPIP_CONTROL;
	}
    }

	/* undo the mode change and return */

    (void) tcpip_setBlockingMode (sockfd, oldMode, oldMode);
    return (returnCode);
}

/* unix_connect

   Purpose: Connect to a Unix domain socket

     Input: sockfd = socket descriptor
		     (int)

            name   = filename of the Unix domain socket
		     (char *)

    Output: The procedure returns e_TCPIP_TIMEOUT if the timeout expires
            before the connection could be made, e_TCPIP_UNABLE_TO_CONNECT if 
            the connection could not be made for any other reason, and
            e_TCPIP_OK if the connection was made successfully.
*/

TcpipErrors unix_connect (int sockfd, char *name, struct timeval stop)
{
    int rc;
    int serrno;
    unsigned len;
    struct timeval timeout; 
    TcpipErrors returnCode;
    socklen_t serrnoLen = sizeof(int);
    fd_mask writeSet[MASK_SIZE];
    struct sockaddr_un unix_addr;

	/* determine the appropriate length */

#ifdef SUNLEN_EXISTS
    len = sizeof(unix_addr.sun_family) + sizeof(unix_addr.sun_len) + strlen(name) + 1;
#else
    len = sizeof(unix_addr.sun_family) + strlen(name);
#endif

	/* check the length */

    if (len >= sizeof(sockaddr_un)) {
	return e_TCPIP_UNABLE_TO_CONNECT;
    }

	/* fill in the sockaddr structure */

    memset ((void *) &unix_addr, 0, sizeof(sockaddr_un));
#ifdef SUNLEN_EXISTS
    unix_addr.sun_len = len;
#endif
    unix_addr.sun_family = AF_UNIX;
    strcpy (unix_addr.sun_path, name);

	/* mark the socket as nonblocking */

    BlockingMode oldMode;
    (void) tcpip_setBlockingMode (sockfd, e_NonBlocking, oldMode);
    (void) tcpip_resetErrorFlag (sockfd);

	/* start the connection */

    do {
	rc = connect (sockfd, (struct sockaddr *) &unix_addr, len);
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	/* wait for the connection to complete if necessary */

    if (rc == 0) {

	returnCode = e_TCPIP_OK;

    } else {

        if ((errno == EWOULDBLOCK) || (errno == EINPROGRESS)) {

	    do {
	        FDMASK_ZERO (writeSet);
	        FDMASK_SET (sockfd, writeSet);
	        timeout = TimevalUtil::subTimevals (stop, SystemUtil::getCurrentWall());
	        rc = select (sockfd + 1, (SELECT_MASK *) NULL, (SELECT_MASK *) &writeSet[0], (SELECT_MASK *) NULL, &timeout);
	    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    if (rc < 0) {
		returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	    } else if (rc == 0) {
		returnCode = e_TCPIP_TIMEOUT;
	    } else if (getsockopt (sockfd, SOL_SOCKET, SO_ERROR, (char *) &serrno, &serrnoLen) < 0) {
		returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	    } else if (serrno != 0) {
		returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	    } else {
		returnCode = e_TCPIP_OK;
	    }

        } else {
	
	    returnCode = e_TCPIP_UNABLE_TO_CONNECT;
	}
    }

	/* mark the socket as blocking */

    (void) tcpip_setBlockingMode (sockfd, oldMode, oldMode);
    return (returnCode);
}
 
TcpipErrors unix_async_connect (int sockfd, char *name)
{
    int rc;
    unsigned len;
    TcpipErrors returnCode;
    struct sockaddr_un unix_addr;

	/* determine the appropriate length */

#ifdef SUNLEN_EXISTS
    len = sizeof(unix_addr.sun_family) + sizeof(unix_addr.sun_len) + strlen(name) + 1;
#else
    len = sizeof(unix_addr.sun_family) + strlen(name);
#endif

	/* check the length */

    if (len >= sizeof(sockaddr_un)) {
	return e_TCPIP_UNABLE_TO_CONNECT;
    }

	/* fill in the sockaddr structure */

    memset ((void *) &unix_addr, 0, sizeof(sockaddr_un));
#ifdef SUNLEN_EXISTS
    unix_addr.sun_len = len;
#endif
    unix_addr.sun_family = AF_UNIX;
    strcpy (unix_addr.sun_path, name);

	/* mark the socket as nonblocking */

    BlockingMode oldMode;
    (void) tcpip_setBlockingMode (sockfd, e_NonBlocking, oldMode);
    (void) tcpip_resetErrorFlag (sockfd);

	/* start the asynchronous connection */

    do {
	rc = connect (sockfd, (struct sockaddr *) &unix_addr, len);
    } while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	/* see if the connection has completed already */ 

    if (rc == 0) {
	returnCode = e_TCPIP_OK;
    } else if ((errno == EWOULDBLOCK) || (errno == EINPROGRESS)) {
	returnCode = e_TCPIP_PENDING;
    } else {
	returnCode = e_TCPIP_UNABLE_TO_CONNECT;
    }

	/* mark the socket as blocking */

    (void) tcpip_setBlockingMode (sockfd, oldMode, oldMode);
    return (returnCode);
}

/* tcpip_listen

   Purpose: Listen to a socket

     Input: sockfd = socket descriptor
		     (int)

    Output: The procedure returns e_TCPIP_UNABLE_TO_LISTEN on error.  
	    Otherwise the procedure returns e_TCPIP_OK.
*/

TcpipErrors tcpip_listen (int sockfd)
{
    TcpipErrors rc;

    if (listen(sockfd, TCPIP_BACKLOG) < 0) {
	rc = e_TCPIP_UNABLE_TO_LISTEN;
    } else {
	rc = e_TCPIP_OK;
    }

    return (rc);
}

/* tcpip_bind

   Purpose: Bind a socket to a local port

     Input: sockfd = socket descriptor
		     (int)

            port   = port number
                     (int)

    Output: The procedure returns e_TCPIP_UNABLE_TO_BIND if it can not bind to
            the port.  The procedure returns e_TCPIP_UNABLE_TO_GET_PORT if it
	    is unable to verify the port number of the bound port.  Otherwise
	    the procedure returns e_TCPIP_OK and fills in selectedPort (int)
            with the TCPIP/IP port of the socket.  The selected port will be 
            equal to the requested port unless the requested port was 
	    TCPIP_ANY_PORT in which case the system has chosen an arbitrary 
            port.
*/

const int c_MAXIMUM_BIND_FAILURES = 5;

TcpipErrors tcpip_bind (int sockfd, int port, int &selectedPort)
{
    int bindFailures = 0;
    struct sockaddr_in address;
    socklen_t size = sizeof(sockaddr_in);

	/* keep trying if we are asking the system for a port */

    while (1) {

	    /* bind the local address so that a client can send to us */

	memset ((void *) &address, 0, sizeof(address));
	address.sin_family      = AF_INET;
	address.sin_port        = htons (port);
	address.sin_addr.s_addr = htonl (INADDR_ANY);

	if (bind (sockfd, (struct sockaddr *) &address, size) < 0) {

	    if ((port == TCPIP_ANY_PORT) && (errno == EADDRINUSE)) {

		if (++bindFailures < c_MAXIMUM_BIND_FAILURES) {
		    continue;
		}
	    }

	    return (e_TCPIP_UNABLE_TO_BIND);
	}

	break;
    }

    if (port != TCPIP_ANY_PORT) {

	selectedPort = port;

    } else {

	if (getsockname(sockfd, (struct sockaddr *) &address, &size) < 0) {
	    return (e_TCPIP_UNABLE_TO_GET_PORT);
	}

	selectedPort = ntohs (address.sin_port);
    }
 
    return (e_TCPIP_OK);
}

/* unix_bind

   Purpose: Bind a Unix domain socket to a pathname

     Input: sockfd = socket descriptor
		     (int)

            name   = pathname within the filesystem of the Unix domain socket
                     (char *)

    Output: The procedure returns e_TCPIP_UNABLE_TO_BIND if it is unable to
            bind the socket to the specified port.  Otherwise the procedure
            returns e_TCPIP_OK.
*/

TcpipErrors unix_bind (int sockfd, char *name)
{
    unsigned len;
    struct sockaddr_un unix_addr;

	/* determine the appropriate length */

#ifdef SUNLEN_EXISTS
    len = sizeof(unix_addr.sun_family) + sizeof(unix_addr.sun_len) + strlen(name) + 1;
#else
    len = sizeof(unix_addr.sun_family) + strlen(name);
#endif

	/* check the length */

    if (len >= sizeof(unix_addr)) {
	return e_TCPIP_UNABLE_TO_BIND;
    }

	/* remove the filename if it exists */

    unlink (name);

	/* create the sockaddr structure */

    memset ((void *) &unix_addr, 0, sizeof(unix_addr));
#ifdef SUNLEN_EXISTS
    unix_addr.sun_len = len;
#endif
    unix_addr.sun_family = AF_UNIX;
    strcpy (unix_addr.sun_path, name);

	/* bind to the filename */

    if (bind (sockfd, (struct sockaddr *) &unix_addr, len) < 0) {
	return e_TCPIP_UNABLE_TO_BIND;
    }

    return e_TCPIP_OK;
}

/* tcpip_setup

   Purpose: This function creates, binds and listens to a socket.

     Input: port = port to which the socket should be bound
		   (int)

    Output: The procedure returns one of the error codes from TcpipErrors on 
            error.  Otherwise the procedure fills in selectedPort (int) with 
            the port to which the socket has been bound, fills in sockfd (int) 
            with the socket descruptor, and returns e_TCPIP_OK.
	    
            selectedPort will be equal to the requested port unless the
            requested port was TCPIP_ANY in which case the system has chosen
            an arbitrary free port.
*/

TcpipErrors tcpip_setup (int port, int &sockfd, int &selectedPort)
{
    TcpipErrors rc;

	/* get a socket */ 

    if ((rc = tcpip_socket (sockfd)) != e_TCPIP_OK) {
	return (rc);
    }

	/* bind the socket to the port*/

    if ((rc = tcpip_bind (sockfd, port, selectedPort)) != e_TCPIP_OK) {
	close (sockfd);
	return (rc);
    }

	/* listen at the sockt */

    if ((rc = tcpip_listen (sockfd)) != e_TCPIP_OK) {
	close (sockfd);
	return (rc);
    }

    return (e_TCPIP_OK);
}

/* unix_setup

   Purpose: This function creates, binds and listens to a Unix domain socket.

     Input: name = pathname within the filesystem of the Unix domain socket 
	           (char *)

    Output: The procedure returns one of the error codes from TcpipErrors on
            error.  Otherwise the procedure loads the socket descriptor into
            sockfd (int) and returns e_TCPIP_OK.
*/

TcpipErrors unix_setup (char *name, int &sockfd)
{
    TcpipErrors rc;

	/* open the socket */

    if ((rc = unix_socket (sockfd)) != e_TCPIP_OK) {
	return (rc);
    }

	/* bind the socket */

    if ((rc = unix_bind (sockfd, name)) != e_TCPIP_OK) {
	close (sockfd);
	return (rc);
    }

	/* listen at the sockt */

    if ((rc = tcpip_listen (sockfd)) != e_TCPIP_OK) {
	close (sockfd);
	return (rc);
    }

    return (e_TCPIP_OK);
}

/* tcpip_accept 

   Purpose: Accept a connection on a socket

     Input: sockfd = socket that has been bound to a port and that has been 
		     put into listening mode -- i.e., a socket that has been
		     returned from tcpip_setup 
		     (int)

	    stop   = time at which to give up
		     (struct timeval)
 
    Output: The procedure loads the socket descriptor of the new accepted
	    connection into newSockfd (int) and returns e_TCPIP_OK on success.
            Otherwise the procedure returns one of the following error codes:

	    e_TCPIP_TIMEOUT          = timeout expired
	    e_TCPIP_UNABLE_TO_ACCEPT = system error while trying to accept  
	    e_TCPIP_CONTROL          = system error while trying to change 
                                       socket characteristics
*/

TcpipErrors tcpip_accept (int sockfd, int &newSockfd, struct timeval stop)
{
    int rc;				// return code from "select"
    int flag = 1;                    	// flag for setsockopt        
    TcpipErrors returnCode;		// return code from this procedure
    struct timeval timeout;		// timeout for "select"
    fd_mask readSet[MASK_SIZE];		// read mask for "select"
    struct sockaddr_in address;      	// client address           
    socklen_t size = sizeof(sockaddr_in);  	// size of the client address

	// put the socket into nonblocking mode */

    BlockingMode oldMode;
    (void) tcpip_setBlockingMode (sockfd, e_NonBlocking, oldMode);
    (void) tcpip_resetErrorFlag (sockfd);

	// loop until acceptance or timeout

    while (1) {

	    // do a nonblocking accept

	do {
	    newSockfd = accept (sockfd, (struct sockaddr *) &address, &size);
 	} while ((newSockfd < 0) && (errno == EINTR));

	    // done if the accept was successful 

	if (newSockfd >= 0) {

		// make sure the TCP_NODELAY option is set for the new socket     
		// This should not be necessary but on some machines the 
		// accept() function does not correctly propagate the 
		// TCP_NODELAY flag  from sockfd to newSockfd.                                     

#ifdef FIX_LATER
		// we use tcpip_accept for non-TCP/IP sockets where it is not
		// necessary to set the NODELAY flag
#endif

	    (void) setsockopt (newSockfd, IPPROTO_TCP, TCP_NODELAY, (char *) &flag, sizeof(int));
	    returnCode = e_TCPIP_OK;
	    break;
	}

	    // any code other than EAGAIN/EWOULDBLOCK indicates a fatal socket 
	    // error

	if ((errno != EAGAIN) && (errno != EWOULDBLOCK)) {
	    returnCode = e_TCPIP_UNABLE_TO_ACCEPT;
	    break;
	}

	    // wait for a connection request

	do {
	    FDMASK_ZERO (readSet);
	    FDMASK_SET (sockfd, readSet);
	    timeout = TimevalUtil::subTimevals (stop, SystemUtil::getCurrentWall());
	    rc = select (sockfd + 1, (SELECT_MASK *) &readSet[0], (SELECT_MASK *) NULL, (SELECT_MASK *) NULL, &timeout);
	} while ((rc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    // check for fatal socket error or timeout

	if (rc < 0) {
	    returnCode = e_TCPIP_UNABLE_TO_ACCEPT;
	    break;
	}

	if (rc == 0) {
	    returnCode = e_TCPIP_TIMEOUT;
	    break;
	}
    }

	// add the new socket to the file table on success, mark the original 
	// socket as blocking and return

    if (returnCode == e_TCPIP_OK) {
	if (FileTable::isFileRecordingOn()) {
	    FileTable::add (newSockfd, FileTable::e_TCPIP_SOCKET);
	}
    }

    (void) tcpip_setBlockingMode (sockfd, oldMode, oldMode);
    return (returnCode);
}

TcpipErrors tcpip_blockingAccept (int sockfd, int &newSockfd)
{
    TcpipErrors returnCode;
    struct sockaddr_in address;      	// client address           
    socklen_t size = sizeof(sockaddr_in);  	// size of the client address

	// accept the connection

    do {
	newSockfd = accept (sockfd, (struct sockaddr *) &address, &size);
    } while ((newSockfd < 0) && (errno == EINTR));

	// set the return code and add the new socket to the file table
	// on success

    if (newSockfd < 0) {

	if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
	    abort_with_message ("internal error: use a different version of tcpip_accept for nonblocking I/O");
	}

	returnCode = e_TCPIP_UNABLE_TO_ACCEPT;

    } else {

	if (FileTable::isFileRecordingOn()) {
	    FileTable::add (newSockfd, FileTable::e_TCPIP_SOCKET);
	}

	returnCode = e_TCPIP_OK;
    }

    return (returnCode);
}

TcpipErrors unix_blockingAccept (int sockfd, int &newSockfd)
{
    TcpipErrors returnCode;
    struct sockaddr_in address;      	// client address           
    socklen_t size = sizeof(sockaddr_in);  	// size of the client address

	// accept the connection

    do {
	newSockfd = accept (sockfd, (struct sockaddr *) &address, &size);
    } while ((newSockfd < 0) && (errno == EINTR));

	// set the return code and add the new socket to the file table
	// on success

    if (newSockfd < 0) {

	if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
	    abort_with_message ("internal error: use a different version of tcpip_accept for nonblocking I/O");
	}

	returnCode = e_TCPIP_UNABLE_TO_ACCEPT;

    } else {

	if (FileTable::isFileRecordingOn()) {
	    FileTable::add (newSockfd, FileTable::e_UNIX_SOCKET);
	}

	returnCode = e_TCPIP_OK;
    }

    return (returnCode);
}

TcpipErrors tcpip_nonblockingAccept (int sockfd, int &newSockfd)
{
    TcpipErrors returnCode;
    struct sockaddr_in address;      	// client address           
    socklen_t size = sizeof(sockaddr_in);  	// size of the client address

	// accept the connection

    do {
	newSockfd = accept (sockfd, (struct sockaddr *) &address, &size);
    } while ((newSockfd < 0) && (errno == EINTR));

	// set the return code and add the new socket to the file table
	// on success

    if (newSockfd < 0) {

	if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
	    returnCode = e_TCPIP_PENDING;
	} else {
	    returnCode = e_TCPIP_UNABLE_TO_ACCEPT;
	}

    } else {

	if (FileTable::isFileRecordingOn()) {
	    FileTable::add (newSockfd, FileTable::e_TCPIP_SOCKET);
	}

	returnCode = e_TCPIP_OK;
    }

    return (returnCode);
}

/* tcpip_blockingReadn

   Purpose: This function is used in place of "read" when reading from a
            socket.

     Input: fd     = socket descriptor
		     (int)

            ptr    = buffer into which the characters are read
		     (char *)

            nbytes = number of characters to read
		     (size_t)

    Output: In all cases nread is set to the number of bytes that were actually
            read from the socket.  The procedure returns e_TCPIP_OK if the
            requested number of bytes was read (in which case nread is equal to
            nbytes) or if EOF occurred before the requested number of bytes
            was read (in which case nread is less than nbytes).  The procedure
            returns e_TCPIP_CANT_READ on any error.
*/

TcpipErrors tcpip_blockingReadn (int fd, char *ptr, size_t nbytes, int &nread)
{
    int count;      		/* number of bytes read per "read" call */
    size_t nleft;   		/* number of bytes left to read         */
    TcpipErrors returnCode;

    nread = 0;
    nleft = nbytes;
	
    while (1) {

	if (nleft == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	}

	do {
	    count = read (fd, ptr, nleft);
	} while ((count < 0) && (errno == EINTR));

	if (count < 0) {			/* ERROR */
	
	    if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
		abort_with_message ("internal error: use a different version of tcpip_blockingReadn for nonblocking I/O");
	    }

	    returnCode = e_TCPIP_CANT_READ;
	    break;
	} 

	if (count == 0) {		/* EOF */
	    returnCode = e_TCPIP_OK;
	    break;
	}

	nread += count;
	nleft -= count;
	ptr   += count;
    }

    return (returnCode);
}

TcpipErrors tcpip_readn (int fd, char *ptr, size_t nbytes, struct timeval *stop, int &nread)
{
    int count;      		/* number of bytes read per "read" call */
    size_t nleft;   		/* number of bytes left to read         */
    int selectRc;
    TcpipErrors returnCode;

    nread = 0;
    nleft = nbytes;

    while (1) {

	if (nleft == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	}

	do {
	    count = read (fd, ptr, nleft);
	} while ((count < 0) && (errno == EINTR));

	if (count < 0) {		/* ERROR */

	    if ((errno != EAGAIN) && (errno != EWOULDBLOCK)) {
		returnCode = e_TCPIP_CANT_READ;
		break;
	    }

	    struct timeval timeout; 
	    struct timeval *timeoutPtr;
	    fd_mask readSet[MASK_SIZE];

	    do {

	        FDMASK_ZERO (readSet);
	        FDMASK_SET (fd, readSet);

		if (stop == NULL) {
		    timeoutPtr = NULL;
		} else {
		    timeout = TimevalUtil::subTimevals (*stop, SystemUtil::getCurrentWall());
		    timeoutPtr = &timeout;
		}

	        selectRc = select (fd + 1, (SELECT_MASK *) &readSet[0], (SELECT_MASK *) NULL, (SELECT_MASK *) NULL, timeoutPtr);

	    } while ((selectRc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    if (selectRc < 0) {
		returnCode = e_TCPIP_CANT_READ;
		break;
	    }

	    if (selectRc == 0) {
		returnCode = e_TCPIP_TIMEOUT;
		break;
	    }

	    continue;
	} 

	if (count == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	} 

	nread += count;
	nleft -= count;
	ptr   += count;
    }

    return (returnCode);
}

TcpipErrors tcpip_nonblockingReadn (int fd, char *ptr, size_t nbytes, int &nread)
{
    int count;      		/* number of bytes read per "read" call */
    size_t nleft;   		/* number of bytes left to read         */
    TcpipErrors returnCode;

    nread = 0;
    nleft = nbytes;

    while (1) {

	if (nleft == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	}

	do {
	    count = read (fd, ptr, nleft);
	} while ((count < 0) && (errno == EINTR));

	if (count < 0) {			/* ERROR */
	
	    if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
		returnCode = e_TCPIP_PENDING;
	    } else {
		returnCode = e_TCPIP_CANT_READ;
	    }

	    break;
	}

	if (count == 0) {			/* EOF */
	    returnCode = e_TCPIP_OK;
	    break;
	}

	nread += count;
	nleft -= count;
	ptr   += count;
    }

    return (returnCode);
}

/* tcpip_writev

   Purpose: This function is used in place of "writev" when writing to a
            socket.

     Input: fd          = file descriptor
		          (int)

	    bufferCount = number of buffers
			  (int)

	    buffers     = the buffers
			  (char **)

	    lengths     = the buffer lengths
			  (int *)

    Output: In all cases nwritten is set to the number of bytes that were
            actually written to the socket.  The procedure returns e_TCPIP_OK
            if the contents of all the buffers were written to the socket in
            their entirety.  The procedure returns e_TCPIP_CANT_WRITE on any
            error.
*/

const int STATIC_BUFFER_SIZE = 1536;
	    
TcpipErrors tcpip_writev (int fd, int bufferCount, char **buffers, int *lengths, int &nwritten)
{
#ifdef NO_WRITEV_FUNCTION 

    int i;
    TcpipErrors tcpipRc;
    int totalLength = 0;

    for (i = 0; i < bufferCount; ++i) {
	totalLength += lengths[i];
    }

    char *buffer = new char [totalLength];
    char *bp     = buffer;
    
    for (i = 0; i < bufferCount; ++i) {
	memcpy (bp, buffers[i], lengths[i]);
	bp += lengths[i];
    }

    tcpipRc = tcpip_blockingWriten (fd, buffer, totalLength, nwritten);
    DELETE_ARRAY_OBJECT(buffer);
    return (tcpipRc);

#else

    int i;
    char *bp;
    int nleft;
    int count;
    int buffersLeft;
    int totalLength = 0;
    struct iovec *ioPtr;
    struct iovec *ioBuffers = new struct iovec [bufferCount];
    char staticBuffer[STATIC_BUFFER_SIZE];

    for (i = 0; i < bufferCount; ++i) {
	totalLength += lengths[i];
    }

    if (totalLength < STATIC_BUFFER_SIZE) {

	bp = staticBuffer;

	for (i = 0; i < bufferCount; ++i) {
	    memcpy (bp, buffers[i], lengths[i]);
	    bp += lengths[i];
	}

	return (tcpip_blockingWriten (fd, staticBuffer, totalLength, nwritten));
    }

    for (i = 0; i < bufferCount; ++i) {
	ioBuffers[i].iov_base = buffers[i];
	ioBuffers[i].iov_len  = lengths[i];
    }

    nwritten    = 0;
    nleft       = totalLength;
    buffersLeft = bufferCount;
    ioPtr       = ioBuffers;

    while (nleft > 0) {

	do {
	    count = writev (fd, ioPtr, buffersLeft);
	} while ((count < 0) && (errno == EINTR));

	if (count <= 0) {

	    if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
		abort_with_message ("internal error: use a different version of tcpip_writev for nonblocking I/O");
	    }

	    return (e_TCPIP_CANT_WRITE);
	}

	nwritten += count;
	nleft    -= count;

	if (nleft > 0) {

	    while (count > (int) ioPtr->iov_len) {
		count       -= ioPtr -> iov_len;
		buffersLeft -= 1;
		ioPtr       += 1;
	    }

	    ioPtr -> iov_len -= count;
	}
    }

    DELETE_ARRAY_OBJECT(ioBuffers);
    return (e_TCPIP_OK);

#endif
}

/* tcpip_blockingWriten

   Purpose: This function is used in place of "write" when writing to a 
            socket.

     Input: fd     = file descriptor
		     (int)

            ptr    = buffer from which characters are written
		     (char *)

            nbytes = number of bytes to write
		     (int)

    Output: In all cases nwritten is set to the number of bytes that were
	    actually written to the socket.  The procedure returns e_TCPIP_OK
	    if all bytes could be written.  Otherwise the procedure returns
            e_TCPIP_CANT_WRITE.
*/

TcpipErrors tcpip_blockingWriten (int fd, const char *ptr, size_t nbytes, int &nwritten)
{
     register size_t nleft;   /* number of bytes left to write            */
     int count;               /* number of bytes written per "write" call */
     TcpipErrors returnCode;
 
     nleft    = nbytes;
     nwritten = 0;

     while (1) {

	if (nleft == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	}

	do {
            count = write (fd, ptr, nleft);
	} while ((count < 0) && (errno == EINTR));

	if (count <= 0) {

	    if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
		abort_with_message ("internal error: use a different version of tcpip_blockingWriten for nonblocking I/O");
	    }

	    returnCode = e_TCPIP_CANT_WRITE;
	    break;
	}

	nwritten += count;
	nleft    -= count;
	ptr      += count;
     }

     return (returnCode);
}

TcpipErrors tcpip_writen (int fd, const char *ptr, size_t nbytes, struct timeval *stop, int &nwritten)
{
    int count;      		/* number of bytes read per "write" call */
    size_t nleft;   		/* number of bytes left to write         */
    int selectRc;
    TcpipErrors returnCode;

    nwritten = 0;
    nleft    = nbytes;

    while (1) {

	if (nleft == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	}

	do {
	    count = write (fd, ptr, nleft);
	} while ((count < 0) && (errno == EINTR));

	if (count <= 0) {		/* ERROR */

	    if ((errno != EAGAIN) && (errno != EWOULDBLOCK)) {
		returnCode = e_TCPIP_CANT_WRITE;
		break;
	    }

	    struct timeval timeout;
	    struct timeval *timeoutPtr;
	    fd_mask writeSet[MASK_SIZE];

	    do {

	        FDMASK_ZERO (writeSet);
	        FDMASK_SET (fd, writeSet);

		if (stop == NULL) {
		    timeoutPtr = NULL;
		} else {
		    timeout = TimevalUtil::subTimevals (*stop, SystemUtil::getCurrentWall());
		    timeoutPtr = &timeout;
		}

	        selectRc = select (fd + 1, (SELECT_MASK *) NULL, (SELECT_MASK *) &writeSet[0], (SELECT_MASK *) NULL, timeoutPtr);

	    } while ((selectRc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    if (selectRc < 0) {
		returnCode = e_TCPIP_CANT_WRITE;
		break;
	    }

	    if (selectRc == 0) {
		returnCode = e_TCPIP_TIMEOUT;
		break;
	    }

	    continue;
	} 

	if (count == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	} 

	nwritten += count;
	nleft    -= count;
	ptr      += count;
    }

    return (returnCode);
}

TcpipErrors tcpip_nonblockingWriten (int fd, const char *ptr, size_t nbytes, int &nwritten)
{
    int count;      		/* number of bytes read per "write" call */
    size_t nleft;   		/* number of bytes left to read         */
    TcpipErrors returnCode;

    nwritten = 0;
    nleft    = nbytes;

    while (1) {

	if (nleft == 0) {
	    returnCode = e_TCPIP_OK;
	    break;
	}

	do {
	    count = write (fd, ptr, nleft);
	} while ((count < 0) && (errno == EINTR));

	if (count <= 0) {			/* ERROR */
	
	    if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
		returnCode = e_TCPIP_PENDING;
	    } else {
		returnCode = e_TCPIP_CANT_WRITE;
	    }

	    break;
	}

	nwritten += count;
	nleft    -= count;
	ptr      += count;
    }

    return (returnCode);
}

/* tcpip_errorCodeToString

   Purpose: Convert one of the error codes to a string

     Input: errorCode = error code
		 	(TcpipErrors)
  
    Output: The procedure returns a pointer to a STATIC string that describes
            the error code. 
*/

const char *tcpip_errorCodeToString (int code)
{
    const char *string;

    switch (code) {

	case e_TCPIP_OK:
            string = "ok"; break;
	case e_TCPIP_UNABLE_TO_CREATE:
            string = "unable to open a socket"; break;
	case e_TCPIP_UNABLE_TO_BIND:
            string = "unable to bind sockfd to the port"; break;
	case e_TCPIP_UNABLE_TO_LISTEN:
            string = "unable to listen on sockfd "; break;
	case e_TCPIP_UNABLE_TO_GET_PORT:
            string = "unable to get the port number of sockfd"; break;
	case e_TCPIP_UNABLE_TO_ACCEPT:
            string = "unable to accept on sockfd"; break;
	case e_TCPIP_NOT_FOUND:
            string = "unable to determine network address of machine"; break;
	case e_TCPIP_UNABLE_TO_CONNECT:
            string = "unable to connect sockfd to remote port"; break;
	case e_TCPIP_TIMEOUT:
            string = "timeout"; break;
 	case e_TCPIP_CONTROL:
            string = "unable to determine socket characteristics"; break;
	case e_TCPIP_PENDING:
            string = "waiting for connection to complete"; break;
	case e_TCPIP_NOT_AN_IP:
            string = "not an IP address"; break;
	default:
            string = "unknown error";
    }

    return (string);
}

/* tcpip_readLong and tcpip_writeLong
 
   Purpose: These procedures read and write a long integer on a socket.
*/

TcpipErrors tcpip_readLong (int fd, UINT_32 &number)
{
    int nread;		
    UINT_32 netLong;	/* long integer in network format */

    if ((tcpip_blockingReadn (fd, (char *) &netLong, sizeof(UINT_32), nread) != e_TCPIP_OK)
	|| (nread != sizeof(UINT_32))) {
	return e_TCPIP_CANT_READ;
    }

    number = ntohl (netLong);
    return e_TCPIP_OK;
}

TcpipErrors tcpip_writeLong (int fd, UINT_32 number)
{
    int nwritten; 
    UINT_32 netLong;

    netLong  = htonl (number);

    if ((tcpip_blockingWriten (fd, (char *) &netLong, sizeof(UINT_32), nwritten) != e_TCPIP_OK) 
	|| (nwritten != sizeof(UINT_32))) {
	return e_TCPIP_CANT_WRITE;
    }

    return e_TCPIP_OK;
}

TcpipErrors tcpip_stringToIP (char *string, UINT_32 &ip)
{
    TcpipErrors returnCode;

    if ((ip = (UINT_32) inet_addr(string)) == INADDR_NONE) {
	returnCode = e_TCPIP_NOT_AN_IP;
    } else {
	returnCode = e_TCPIP_OK;
    }

    return (returnCode);
}

char *tcpip_IPToString (UINT_32 ip)
{
    struct in_addr inaddr;
    inaddr.s_addr = ip;
    return (strcpyWithAlloc (inet_ntoa (inaddr)));
}

TcpipErrors tcpip_resetErrorFlag (int sockfd)
{
    int optvalue;
    TcpipErrors rc;
    socklen_t optlength = sizeof(int);

    if (getsockopt (sockfd, SOL_SOCKET, SO_ERROR, (char *) &optvalue, &optlength) < 0) {
	rc = e_TCPIP_CONTROL;
    } else {
	rc = e_TCPIP_OK;
    }

    return (rc);
}

/* tcpip_close and unix_close

   Purpose: Close a socket
*/

TcpipErrors tcpip_close (int sockfd)
{
    if (FileTable::isFileRecordingOn()) {
	FileTable::remove (sockfd);
    }

    close (sockfd);
    return (e_TCPIP_OK);
}

TcpipErrors unix_close (int sockfd)
{
    return (tcpip_close(sockfd));
}
