/* Agent Tcl
   Bob Gray
   26 June 1995

   mesgTcpip.h

   This file defines the routines that handle TCP/IP onnections.  These
   routines are based on the examples in [W. Richard Stevens, Unix Network
   Programming, Prentice Hall, 1990] and on the code that Saurab Nog
   developed as part of his A.I. project.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _MESG_TCPIP_H
#define _MESG_TCPIP_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _PLAT_PORTING_H
#include "platPorting.h"
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"
#endif

    /*
     * Many of the routines defined here assume that the socket is in blocking
     * or nonblocking mode.  Such assumptions are essential for efficiency

     * at this low level.
     */

    /* 
     * forward declarations 
     */

class MachineId;
class RemotePort;

    /*
     * INADDR_NONE
     */

#ifndef INADDR_NONE
#define INADDR_NONE 0xffffffff
#endif

    /* 
     * the "any port" constant
     */

const int TCPIP_ANY_PORT = 0;

    /*
     * error codes
     */

enum TcpipErrors {
    e_TCPIP_OK,
    e_TCPIP_UNABLE_TO_CREATE,
    e_TCPIP_UNABLE_TO_BIND,
    e_TCPIP_UNABLE_TO_LISTEN,
    e_TCPIP_UNABLE_TO_GET_PORT,
    e_TCPIP_UNABLE_TO_ACCEPT,
    e_TCPIP_NOT_FOUND,
    e_TCPIP_UNABLE_TO_CONNECT,
    e_TCPIP_TIMEOUT,
    e_TCPIP_CONTROL,
    e_TCPIP_PENDING,
    e_TCPIP_NOT_AN_IP,
    e_TCPIP_CANT_READ,
    e_TCPIP_CANT_WRITE
};

    /* reset the error flag associated with the socket */

TcpipErrors tcpip_resetErrorFlag (int sockfd);

   /* create, bind AND listen in INTERNET or UNIX domain */

TcpipErrors tcpip_setup		
	(int port, int &sockfd, int &selectedPort);

TcpipErrors unix_setup
	(char *name, int &sockfd);

   /* 
    * socket, bind and connect in the INTERNET or UNIX domain
    *
    * NOTES:
    *
    * 1. tcpip_async_connect assumes that the IP address is already known
    *    (i.e., that port.ip is NOT equal to UNKNOWN_IP). 
    */

TcpipErrors tcpip_socket	
	(int &sockfd);

TcpipErrors tcpip_bind		
	(int sockfd, int port, int &selectedPort);

TcpipErrors tcpip_connect	
	(int sockfd, RemotePort &port, struct timeval stop);

TcpipErrors tcpip_async_connect
	(int sockfd, RemotePort &port);

TcpipErrors unix_socket	
	(int &sockfd);

TcpipErrors unix_bind	
	(int sockfd, char *name);

TcpipErrors unix_connect  
	(int sockfd, char *name, struct timeval stop);

TcpipErrors unix_async_connect  
	(int sockfd, char *name);

   /* 
    * accept and listen
    *
    * NOTES:
    *
#ifdef FIX_LATER
    * 1. tcpip_blockingAccept, tcpip_accept and tcpip_nonblockingAccept are
    *    used for both Unix domain and TCP/IP sockets.  We should rename these 
    *    functions.
#endif
    * 2. tcpip_blockingAccept ASSUMES that the socket is in blocking mode.
    * 3. tcpip_nonblockingAccept ASSUMES that the socket is in nonblocking mode.
    * 4. tcpip_nonblockingAccept returns e_TCPIP_OK if a connection was
    *    accepted, e_TCPIP_PENDING if no connection was available, and
    *    e_TCPIP_UNABLE TO_ACCEPT if an error occurs.
    * 5. tcpip_accept returns e_TCPIP_OK if a connection was accepted,
    *    e_TCPIP_TIMEOUT if the timeout expired before a connection became
    *    available, and e_TCPIP_UNABLE_TO_ACCEPT if an error occurs.
    * 6. tcpip_blockingAccept returns e_TCPIP_OK if a connection was accepted,
    *    and e_TCPIP_UNABLE_TO_ACCEPT if an error occurs.
    */

TcpipErrors tcpip_listen
	(int sockfd);

TcpipErrors tcpip_blockingAccept
	(int sockfd, int &newSockfd);

TcpipErrors unix_blockingAccept
	(int sockfd, int &newSockfd);

TcpipErrors tcpip_accept
	(int sockfd, int &newSockfd, struct timeval stop);

TcpipErrors tcpip_nonblockingAccept
	(int sockfd, int &newSockfd);

    /* close sockets */

TcpipErrors tcpip_close
	(int sockfd);

TcpipErrors unix_close
	(int sockfd);

    /* turn on/off blocking I/O */

enum BlockingMode {
    e_Blocking = 0,
    e_NonBlocking = 1
};

TcpipErrors tcpip_setBlockingMode
	(int sockfd, BlockingMode newMode, BlockingMode &oldMode);

    /* get the IP address of the given machine */

TcpipErrors tcpip_getIP 
	(MachineId &machine, struct timeval stop);

    /* 
     * set the IP address and name of the current host; get the name of 
     * of the current host                                             
     *
     * NOTES:
     *
     * 1. tcpip_getActualHostname returns a dynamically allocated string
     */
	
void tcpip_setActualIPAndHostname
    (UINT_32 ip, char *name);

char *tcpip_getActualHostname 
	(void);
 
    /* get the localhost IP address 127.0.0.1 */

UINT_32 tcpip_getLocalhostIP 
    (void);
 
   /* 
    * convert an IP address to a string or a string to an IP address
    *
    * NOTES:
    *
    * 1. tcpip_IPToString returns a dynamically allocated string.
    */

char *tcpip_IPToString 
	(UINT_32 ip);

TcpipErrors tcpip_stringToIP 
	(char *string, UINT_32 &ip);

    /* get IP address and port number of local end of the connection; */
    /* get IP address and port number of remote end of the connection */

TcpipErrors tcpip_getSockname 
    (int sockfd, UINT_32 &ip, UINT_32 &port);

TcpipErrors tcpip_getPeername 
    (int sockfd, UINT_32 &ip, UINT_32 &port);

   /* convert an error code to a string */

const char *tcpip_errorCodeToString 
    (TcpipErrors errorCode);

    /* 
     * read and write n bytes -- return e_TCPIP_OK if full number of bytes  
     * was read/written or if EOF occured during the read (in which case nread
     * will be less than nbytes); return e_TCPIP_CANT_READ/e_TCPIP_CANT_WRITE
     * on any error.                                                          
     *
     * NOTES:
     *
     * 1. tcpip_blockingReadn  ASSUMES that the socket is in blocking mode.
     * 2. tcpip_blockingWriten ASSUMES that the socket is in blocking mode.
     */

TcpipErrors tcpip_blockingReadn 
    (int fd, char *ptr, size_t nbytes, int &nread);

TcpipErrors tcpip_blockingWriten 
    (int fd, const char *ptr, size_t nybtes, int &nwritten);

    /* 
     * read and write n bytes -- return e_TCPIP_OK if full number of bytes  
     * was read/written or if EOF occured during the read (in which case nread
     * will be less than nbytes); return e_TCPIP_TIMEOUT if timeout expired
     * before full number of bytes were read/written; return e_TCPIP_CANT_READ/
     * e_TCPIP_CANT_WRITE on any other error.                                                    
     *
     * NOTES:
     *
     * 1. tcpip_readn  ASSUMES that the socket is in nonblocking mode.
     * 2. tcpip_writen ASSUMES that the socket is in nonblocking mode.
     * 3. Both functions will block as long as necessary if the timeout
     *    paramater, stop, is NULL.
     */

TcpipErrors tcpip_readn 
	(int fd, char *ptr, size_t nbytes, struct timeval *stop,
	 int &nread);

TcpipErrors tcpip_writen
	(int fd, const char *ptr, size_t nybtes, struct timeval *stop,
	 int &nwritten);

    /*
     * read and write n bytes -- If the requested number of bytes could
     * be read/written without blocking, return e_TCPIP_OK.  If EOF occurs
     * while reading, return e_TCPIP_OK with nread set to the number of
     * bytes that were actuallhy read.  If only some of of the bytes could be 
     * read/written without blocking, return e_TCPIP_PENDING with nread/nwritten
     * set to the number of bytes that were actually read/written.  If any
     * error occurs, return e_TCPIP_CANT_READ/e_TCPIP_CANT_WRITE.
     *
     * NOTES:
     *
     * 1. tcpip_nonblockingReadn  ASSUMES that the socket is in nonblocking mode.
     * 2. tcpip_nonblockingWriten ASSUMES that the socket is in nonblocking mode.
     */

TcpipErrors tcpip_nonblockingReadn 
	(int fd, char *ptr, size_t nbytes, int &nread);

TcpipErrors tcpip_nonblockingWriten
	(int fd, const char *ptr, size_t nybtes, int &nwritten);

    /* 
     * write a set of buffers -- return e_TCPIP_OK if all buffers were written;
     * return e_TCPIP_CANT_WRITE on any error                                   
     *
     * NOTES:
     *
     * 1. tcpip_writev ASSUMES that the socket is in blocking mode
     */

TcpipErrors tcpip_writev
	(int fd, int bufferCount, char **buffers, int *lengths, int &nwritten);

   /* 
    * read and write longs
    *
    * NOTES:
    *
    * 1. tcpip_readlong  ASSUMES that the socket is in blocking mode.
    * 2. tcpip_writelong ASSUMES that the socket is in blocking mode.
    */

TcpipErrors tcpip_readLong 
	(int fd, UINT_32 &number);

TcpipErrors tcpip_writeLong 
	(int fd, UINT_32 number);

#endif
