/* Agent Tcl
   Bob Gray
   16 November 1996

   platPorting.cc

   This implementation file handles porting issues that occur because
   of differences between systems.  It reads in UNIX-related
   header files and sets up UNIX-related macros for Agent Tcl's 
   UNIX core.  It should be the only file that contains #ifdefs 
   to handle different flavors of UNIX.  This file sets up the
   union of all UNIX-related things needed by any of the Tcl
   core files.  This file depends on configuration #defines such
   as NO_DIRENT_H that are set up by the "configure" scripts".

   Copyright (c) 1995-1996, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"

/*
 * Portable versions of FD_ZERO, FD_SET, FD_CLR and FD_ISSET
 * and two convenience routines FD_INDEX and FD_BIT
 */

void FDMASK_ZERO (fd_mask *mask)
{
    for (unsigned i = 0; i < MASK_SIZE; i++) {
	mask[i] = 0;
    }
}

void FDMASK_SET (int fd, fd_mask *mask)
{
    if (fd > (int) FD_SETSIZE) {
	abort_with_message ("fd out of range in FDMASK_SET");
    }

    int index = fd / (NBBY * sizeof(fd_mask));
    fd_mask bit = 1 << (fd % (NBBY * sizeof(fd_mask))); 
    mask[index] |= bit;
}

void FDMASK_CLR (int fd, fd_mask *mask)
{
    if (fd > (int) FD_SETSIZE) {
	abort_with_message ("fd out of range in FDMASK_CLR");
    }
    
    int index = fd / (NBBY * sizeof(fd_mask));
    fd_mask bit = 1 << (fd % (NBBY * sizeof(fd_mask))); 
    mask[index] &= ~bit;
}

int FDMASK_ISSET (int fd, fd_mask *mask)
{
    if (fd > (int) FD_SETSIZE) {
	abort_with_message ("fd out of range in FDMASK_ISSET");
    }
    
    int index = fd / (NBBY * sizeof(fd_mask));
    fd_mask bit = 1 << (fd % (NBBY * sizeof(fd_mask))); 
    return (mask[index] & bit ? 1 : 0);
}

int FDMASK_INDEX (int fd)
{
    if (fd > (int) FD_SETSIZE) {
	abort_with_message ("fd out of range in FDMASK_INDEX");
    }
    
    return (fd / (NBBY * sizeof(fd_mask)));
}

fd_mask FDMASK_BIT (int fd)
{
    if (fd > (int) FD_SETSIZE) {
	abort_with_message ("fd out of range in FDMASK_BIT");
    }

    return (1 << (fd % (NBBY * sizeof(fd_mask))));
}

/*
 * unsetenv if needed
 */

#ifdef NO_UNSETENV_FUNCTION

extern char **environ;

int unsetenv (char *name)
{
    for (char **environPtr = environ; *environPtr != (char *) NULL; ++environPtr) {

	for (char *sp = *environPtr; *sp != '\0'; ++sp) {

	    if (*sp == '=') {

		int length = sp - *environPtr;

		if (!strncmp(name, *environPtr, length)) {

		    while (*environPtr != (char *) NULL) {
			*environPtr = *(environPtr + 1);
			++environPtr;
		    }

		    return (0);
		}

		break;
	    }
	}
    }

    return (-1);
}

#endif
