/* Agent Tcl
   Bob Gray
   28 Spetember 1995

   platTimers.cc

   This file implements the timer functions that are constant across platforms.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include <math.h>
#include "platDelete.h"
#include "platTimers.h"
#include "platTimeval.h"

     /* starting size of the timer heap */

const int DEFAULT_TIMER_MAX = 8;

/* TimerTrap::~TimerTrap

   Purpose: This is the virtual distructor for class TimerTrap.
*/

TimerTrap::~TimerTrap ()
{
    // empty
}

/* TimerQueue::TimerQueue

   Purpose: These procedures are the constructor and destructor for class 
	    TimerQueue.
*/

TimerQueue::TimerQueue (void):
    timerMax (DEFAULT_TIMER_MAX),
    timerCount (0),
    timerHeap (new Timer[DEFAULT_TIMER_MAX])
{
    // empty
}

TimerQueue::~TimerQueue ()
{
    DELETE_ARRAY_OBJECT(timerHeap);
}

/* TimerQueue::expand

   Purpose: Expand the queue if necessary 
*/

void TimerQueue::expand (void)
{
    Timer *newHeap;

    newHeap = new Timer [timerMax << 1];
    
    for (int i = 0; i < timerMax; i++) {
	newHeap[i] = timerHeap[i];
    }

    delete [] timerHeap;
    timerMax = timerMax << 1;
    timerHeap = newHeap;
}

/* TimerQueue::add

   Purpose: Add a timer to the heap

     Input: stop = the timer
		   (struct timeval)

	    trap = the timer callbacks
		   (class TimerTrap *)
*/

void TimerQueue::add (struct timeval stop, TimerTrap *trap)
{
    int newIndex;

    if (timerCount == timerMax) {
	expand ();
    }

    newIndex = timerCount;
    timerHeap[newIndex] = Timer (stop, trap);
    timerCount += 1;
    bubble_up (newIndex);
}

/* TimerQueue::removeAllAttachedToTrap

   Purpose: Remove all timers with a specific trap from the heap

     Input: trap = the timer trap
		   (class TimerTrap *)
*/

void TimerQueue::removeAllAttachedToTrap (TimerTrap *trap)
{
    int hole;
    int minRemovalIndex;

	// remove all the timers with the given trap

    hole            = 0;
    minRemovalIndex = timerCount;

    while (hole < timerCount) {

	if (trap == timerHeap[hole].trap) {

	    timerCount -= 1;

	    if (hole != timerCount) {
		timerHeap[hole] = timerHeap[timerCount];
	    }

	    if (hole < minRemovalIndex) {
		minRemovalIndex = hole;
	    }

	} else {

	    hole += 1;
	}
    }

	// re-heapify

    for (int i = minRemovalIndex; i < timerCount; ++i) {
	bubble_up (i);
    }
}

/* TimerQueue::remove

   Purpose: Remove a timer from the heap

     Input: stop = the timer
		   (struct timeval)

	    trap = the timer callbacks
		   (class TimerTrap *)
*/

void TimerQueue::remove (struct timeval stop, TimerTrap *trap)
{
    int hole;

#ifdef FIX_LATER
	/* we need to find the timer more efficiently */
#endif

    for (hole = 0; hole < timerCount; ++hole) {
	if ((stop.tv_sec == timerHeap[hole].stop.tv_sec) && 
	    (stop.tv_usec == timerHeap[hole].stop.tv_usec) &&
	    (trap == timerHeap[hole].trap)) {
	    break;
	}
    }

    if (hole == timerCount) {	// did not find the timer
	return;
    }

	/* bubble the new empty hole down */

    hole = bubble_down (hole);

	/* copy last item and bubble up in order to get a full tree again */

    timerCount -= 1;

    if (hole != timerCount) {
	timerHeap[hole] = timerHeap[timerCount];
	bubble_up (hole);
    }
}

/* TimerQueue::bubble_down

   Purpose: Bubble a hole down to its proper position in the heap

     Input: hole = index of the hole in the heap
	           (int)

    Output: The procedure returns the *new* index of the hole.
*/

int TimerQueue::bubble_down (int hole)
{
    int left;
    int right;

	/* bubble the hole down */

    left  = (hole << 1) + 1;
    right = (hole << 1) + 2;
 
    while (left < timerCount) {

	if ((right >= timerCount) || (TimevalUtil::compareTimevals (timerHeap[left].stop, timerHeap[right].stop) < 0)) {
	    swap (left, hole);
	    hole = left;
	} else {
	    swap (right, hole);
	    hole = right;
	} 

	left  = (hole << 1) + 1;
	right = (hole << 1) + 2;
    }

    return (hole);
}

/* TimerQueue::bubble_up

   Purpose: Bubble an item up to its proper position in the heap

     Input: i = index of the item in the heap

    Output: The procedure bubbles the item up to its proper position.
*/

void TimerQueue::bubble_up (int i)
{
    int child;           /* index of the child in the heap  */
    int parent;          /* index of the parent in the heap */

	/* bubble up the item */

    child  = i;
    parent = (child - 1) >> 1;

    while ((parent >= 0) && (TimevalUtil::compareTimevals (timerHeap[child].stop, timerHeap[parent].stop) < 0)) {
	swap (child, parent);
	child  = parent;
	parent = (child - 1) >> 1;
    }
}

/* TimerQueue::swap

   Purpose: Swap two elements of the heap

     Input: i = index of first element
            j = index of second element

    Output: The procedure swaps the two elements.
*/

void TimerQueue::swap (int i, int j)
{
    Timer temp   = timerHeap[i];
    timerHeap[i] = timerHeap[j];
    timerHeap[j] = temp;
}
 
/* TimerQueue::removeMinimum

   Purpose: Remove the smallest timer from the heap

     Input: None

    Output: The procedure removes and returns the smallest timer. 

      NOTE: The procedure assumes that SIGALRM is blocked.
*/

void TimerQueue::removeMinimum (struct Timer& minTimer)
{
    int hole;          /* the hole that we are bubbling down */

	/* assert that there is one or more timers in the heap */

    assert (timerCount > 0);

	/* copy out the min element */

    minTimer = timerHeap[0];

	/* bubble the new empty hole down */

    hole = bubble_down (0);

	/* copy last item and bubble up in order to get a full tree again */

    timerCount -= 1;

    if (hole != timerCount) {
	timerHeap[hole] = timerHeap[timerCount];
	bubble_up (hole);
    }
}

/* Timer::Timer

   Purpose: These procedures are the constructors for class TIMER.
*/

Timer::Timer (struct timeval p_stop, TimerTrap *p_trap)
{
    Timer::stop = p_stop;
    Timer::trap = p_trap; 
}

Timer::Timer (const Timer &timer)
{
    stop = timer.stop;
    trap = timer.trap;
}

Timer::Timer (void)
{
    stop.tv_sec  = 0;
    stop.tv_usec = 0;
    trap         = (TimerTrap *) NULL;
}
