/* Agent Tcl
   Bob Gray
   26 January 1997

   platTimeval.cc

   This file implements utility routines for working with timevals. 

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include <sys/times.h>
#include <math.h>
#include "platMath.h"
#include "platTimeval.h"

/* TimevalUtil::addTimevals

   Purpose: Add two timevals

     Input: t1 = the first timeval
		 (struct timeval)

	    t2 = the second timeval
		 (struct timeval)

    Output: The procedure returns the sum of the two timevals.
*/

struct timeval TimevalUtil::addTimevals (struct timeval t1, struct timeval t2)
{
    struct timeval sum;

#ifdef FIX_LATER
	/* the int casts below make the code fragment portable to all */
	/* platforms but also means that the code will break in the   */
	/* year 2014                                                  */
#endif

    sum.tv_sec  = MathUtil::addWithCeiling ((int) t1.tv_sec, (int) t2.tv_sec);
    sum.tv_usec = MathUtil::addWithCeiling ((int) t1.tv_usec, (int) t2.tv_usec);

    if (sum.tv_usec >= 1000000) {
	sum.tv_sec = MathUtil::addWithCeiling (sum.tv_sec, 1);
        sum.tv_usec -= 1000000;
    }

    return (sum);
}

/* TimevalUtil::subTimevals

   Purpose: Subtract two timevals

     Input: t1 = the first timeval
		 (struct timeval)

	    t2 = the second timeval
		 (struct timeval)

    Output: The procedure returns the difference t1 - t2 (or a timeval of 0
	    if t1 > t2).
*/

struct timeval TimevalUtil::subTimevals (struct timeval t1, struct timeval t2)
{
    struct timeval diff;

	/* timeval of 0 if t1 is smaller than t2 */

    if (t1.tv_sec < t2.tv_sec) {

	diff.tv_sec  = 0;
	diff.tv_usec = 0; 

    } else {

	diff.tv_sec = t1.tv_sec - t2.tv_sec;

	if (t1.tv_usec < t2.tv_usec) {

	    if (diff.tv_sec == 0) {
		diff.tv_usec = 0;
	    } else {
	  	diff.tv_sec -= 1;
		diff.tv_usec = 1000000 + t1.tv_usec - t2.tv_usec;
	    }

	} else {

	    diff.tv_usec = t1.tv_usec - t2.tv_usec;
	}
    }

    return (diff);
}

/* TimevalUtil::compareTimevals

   Purpose: Compare two timevals

     Input: t1 = the first timeval
		 (struct timeval)

	    t2 = the second timeval
	 	 (struct timeval)
  
    Output: The procedure returns -1 if t1 < t2, 0 if t1 == t2 and 1 if t1 > t2.
*/

int TimevalUtil::compareTimevals (struct timeval t1, struct timeval t2)
{
    if (t1.tv_sec < t2.tv_sec) {
	return -1;
    } else if (t1.tv_sec > t2.tv_sec) {
	return 1;
    } else if (t1.tv_usec < t2.tv_usec) {
	return -1;
    } else if (t1.tv_usec > t2.tv_usec) {
	return 1;
    } else {
	return 0;
    }
}

/* TimevalUtil::secondsToTimeval

   Purpose: Convert seconds to a timeval

     Input: seconds = the number of seconds
		      (double)

    Output: The procedure returns a timeval structure that specifies the given
	    number of seconds.
*/

struct timeval TimevalUtil::secondsToTimeval (double seconds)
{
    double integer;
    double fraction;
    struct timeval tv;

    fraction = modf (seconds, &integer);
    tv.tv_sec = (long) (integer);
    tv.tv_usec = (long) (fraction * 1000000.0);

    if (tv.tv_usec >= 1000000) {
	tv.tv_usec -= 1000000;
	tv.tv_sec += 1;
    }

    return (tv);
}

/* TimevalUtil::timevalToSeconds

   Purpose: Convert a timeval to seconds

     Input: tv = the timeval
                 (struct timeval)

    Output: The procedure returns the number of seconds that corresponds to
	    the given timeval.
*/

double TimevalUtil::timevalToSeconds (struct timeval tv)
{
    double seconds;

    seconds = (double) tv.tv_sec;
    seconds += (((double) tv.tv_usec) / 1000000.0);
    return (seconds);
}
