/* Bob Gray
   Agent Tcl
   13 July 1996

   platExclusion.h

   This file defines the classes that provide simple mutual exclusion.
*/

#ifndef _PLAT_EXCLUSION_H
#define _PLAT_EXCLUSION_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _PLAT_DELETE_H
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"
#endif

   /* forward declarations */

struct timeval;
class MonitorLockImp;
class MutexLockImp;

   /* This is a recursive monitor that provides acquire, release, */
   /* try_acquire, wakeup and wait operations.                    */

class MonitorLock
{
	MonitorLockImp *m_this;  // implementation details

    public:

	    /* get or return a condition variable */

	BOOLEAN getConditionVariable (UINT_32 &handle);
	BOOLEAN returnConditionVariable (UINT_32 handle);

	    /* acquire and release */

	BOOLEAN acquire (void);
	BOOLEAN try_acquire (void);
	BOOLEAN release (void);

	    /* wait and wakeup (with builtin condition variable or */
	    /* specified condition variable)                       */

	BOOLEAN wait       (void);
	BOOLEAN timed_wait (struct timeval delay);
	BOOLEAN wakeup     (void);
	BOOLEAN wait       (UINT_32 conditionHandle, BOOLEAN debug = e_FALSE);
	BOOLEAN timed_wait (UINT_32 conditionHandle, struct timeval delay);
	BOOLEAN wakeup     (UINT_32 conditionHandle, BOOLEAN debug = e_FALSE);
	
	    /* assignment operator */

	MonitorLock& operator= (const MonitorLock& lock);

	    /* constructors and destructor */

	MonitorLock (void);
	MonitorLock (const MonitorLock& lock);
       ~MonitorLock ();
};

    /* MonitorGroup is a list of monitor locks */

class MonitorGroup
{
	    // maximum number of locks

	enum {
	    MAX_LOCKS = 4
	};

	    // variables

	int m_count;                       // number of locks in the list
 	MonitorLock *m_locks[MAX_LOCKS];   // list of locks

    public:

	    // constructors

	MonitorGroup (void):
	    m_count (0)
	{
	}

#ifdef FIX_NOW
	    /* eliminate the dynamic memory allocation */
#endif

	MonitorGroup (const MonitorLock& lock) {
	    m_locks[0] = new MonitorLock (lock);
	    m_count = 1;
	}

	MonitorGroup (const MonitorGroup& list) {

	    m_count = list.m_count;

	    for (int i = 0; i < m_count; i++) {
		m_locks[i] = new MonitorLock (*list.m_locks[i]);
	    }
	}

	    // assignment operator 

	MonitorGroup& operator= (const MonitorGroup &list) {

	    int i;

	    if (this == &list) {
		return (*this);
	    }

	    for (i = 0; i < m_count; ++i) {
		DELETE_OBJECT(m_locks[i]);
	    }

	    for (i = 0; i < list.m_count; ++i) {
		m_locks[i] = new MonitorLock (*list.m_locks[i]);
	    }

	    m_count = list.m_count;
	    return (*this);
	}

	    // push or pop a lock

	void push (const MonitorLock &lock) {
	    m_locks[m_count++] = new MonitorLock (lock); 
	}

	void pop (void) {
	    m_count -= 1;
	    DELETE_OBJECT(m_locks[m_count]);
	}

	    // acquire all the locks (in the order they were added)

	void acquireAll (void) {
	    for (int i = 0; i < m_count; i++) {
		m_locks[i] -> acquire ();
	    }
	}

	    // release all the locks (in the order they were acquired)

	void releaseAll (void) {
	    for (int i = m_count - 1; i >= 0; i--) {
		m_locks[i] -> release();
	    }
	}

	    // destructor

       ~MonitorGroup () {
	    for (int i = 0; i < m_count; ++i) {
		DELETE_OBJECT(m_locks[i]);
	    }
	}
};

    /* This is a guard that ensures that an acquired monitor lock is */
    /* released.  This should eventually be a template.              */

class Guard
{
	MonitorLock &lock;

    public:

	Guard (MonitorLock &p_lock):
	    lock (p_lock)
	{
	    lock.acquire ();
	}

       ~Guard () {
	    lock.release ();
	}
};

   /* This is a *nonrecursive* lock that provides acquire, release */
   /* and try_acquire operations.                                  */

class MutexLock
{
	MutexLockImp *m_this;  // implementation details

    public:

	    /* acquire and release */

	BOOLEAN acquire (void);
	BOOLEAN try_acquire (void);
	BOOLEAN release (void);

	    /* assignment operator */

	MutexLock& operator= (const MutexLock& lock);

	    /* constructors and destructor */

	MutexLock (void);
	MutexLock (const MutexLock& lock);
       ~MutexLock ();
};

    /* This is a guard that ensures that an acquired mutex lock is */
    /* released.  This should eventually be a template.            */

#ifdef FIX_LATER
    /* MutexGuard should be available in all environments. */
#endif

#ifdef MULTIPLE_THREADS

class MutexGuard
{
	MutexLock &lock;

    public:

	MutexGuard (MutexLock &p_lock):
	    lock (p_lock)
	{
	    lock.acquire ();
	}

       ~MutexGuard () {
	    lock.release ();
	}
};

#endif

    /* This class is used when some initialization code must be performed */
    /* only once.                                                         */

class OnceTrap
{
    public:

	virtual void initialize (void) = 0;
};

class MonitorOnce
{
    public:

	    // control flags

	enum OnceControl {
	    e_NEVER	  = 0,
	    e_IN_PROGRESS = 1,
	    e_DONE	  = 2
 	};

	    // do the code only once

	static void doOnce (OnceControl &onceControl, OnceTrap &trap);
};

#endif
