/* Bob Gray
   Agent Tcl
   13 July 1996

   platShortExclusion.h

   Copyright (C) 1998-1999, Robert S. Gray, Dartmouth College

   This file defines the class ShortMutexLock which provides mututal
   exclusion around EXTREMELY SHORT SECTIONS OF CODE (e.g., incrementing
   a reference count).  In general, it will be implemented completely
   differently than the class MutexLock in platExclusion.h.
*/

#ifndef _PLAT_SHORT_EXCLUSION_H
#define _PLAT_SHORT_EXCLUSION_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _TRUEFALSE_H
#include "truefalse.h"
#endif

#if defined(LINUX_I386_PLATFORM) && defined(USE_ASSEMBLY_LOCKS)

    // spin lock data type

typedef int spin_lock_t;

    // spin lock initializer

const int c_SPIN_LOCK_INITIALIZER = 0;

    // short mutex lock (based on a spin lock)
   
class ShortMutexLock 
{
    spin_lock_t m_lock;		// spin lock

	// copy-constructor and assignment operator are not supported

    ShortMutexLock (const ShortMutexLock&) {};
    ShortMutexLock& operator= (const ShortMutexLock&) { return *this; }

	// yield to another thread if we are trying to ACQUIRE the lock
	// but the lock is currently held

    static void yield (void);

public:

	// try to acquire, acquire and release

    BOOLEAN try_acquire (void) {
	register int locked;
	__asm__ __volatile ("xchgl %0, %1"
	    : "=&r" (locked), "=m" (m_lock) : "0" (1));
	return ((!locked) ? e_TRUE : e_FALSE);
    }

    BOOLEAN acquire (void) {

	while (!try_acquire()) {
	    ShortMutexLock::yield();
	}

	return (e_TRUE);
    }

    BOOLEAN release (void) {
	register int unlocked;
	__asm__ __volatile ("xchgl %0, %1"
	    : "=&r" (unlocked), "=m:" (m_lock) : "0" (0));
	return (e_TRUE);
    }

	// constructor and destructor

    ShortMutexLock (void):
	m_lock (c_SPIN_LOCK_INITIALIZER)
    {}

   ~ShortMutexLock () {}
};

#else

    // we need MutexLock for the default ShortMutexLock

#ifndef _PLAT_EXCLUSION_H
#include "platExclusion.h"
#endif

    // default short mutex lock (just a wrapper around MutexLock)

class ShortMutexLock
{
    MutexLock m_lock;

	// copy-constructor and assignment operator are not supported

    ShortMutexLock (const ShortMutexLock&) {};
    ShortMutexLock& operator= (const ShortMutexLock&) { return *this; }

public:

	/* acquire, try to acquire and release */

    BOOLEAN acquire (void) {
	return (m_lock.acquire());
    }

    BOOLEAN try_acquire (void) {
	return (m_lock.try_acquire());
    }

    BOOLEAN release (void) {
	return (m_lock.release());
    }

	/* constructors and destructor */

    ShortMutexLock (void):
	m_lock()
    {}

   ~ShortMutexLock () {}
};

#endif
#endif
