/* Agent Tcl
   Bob Gray
   28 Spetember 1995

   platTimers.h

   This file defines a class that allows a process to define any number of
   timers.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _PLAT_TIMERS_H
#define _PLAT_TIMERS_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _TRUEFALSE_H
#include "truefalse.h"
#endif

     /* forward declarations */
    
class Timers;
class TimerTrap;
class TimersImplementation;
struct timeval;
 
    /* any class that needs a timer interrupt inherits TimerTrap */

class TimerTrap
{
	friend class Timers;

    public:

	    /* a wall timer has expired */

	virtual void wallAlarm (void) = 0;

	    /* a cpu timer has expired */

	virtual void cpuAlarm (void) = 0;

	    /* virtual destructor */

	virtual ~TimerTrap();
}; 

class Timers
{
	    /* implementation-specific data structures */

	static TimersImplementation *m_this;

    private:

	static void disposeWallTimers (void);
	static void disposeCpuTimers (void);

    public:

	    /* dispose of timers */

	static void disposeTimers (void);

	    /* set a timer */ 

	void add (struct timeval stop, TimerTrap *trap);
	void addCpu (struct timeval stop, TimerTrap *trap);
	void remove (struct timeval stop, TimerTrap *trap);
	void removeCpu (struct timeval stop, TimerTrap *trap);
	void removeAllAttachedToTrap (TimerTrap *trap);

	    /* constructor and destructor */

	Timers (void);
       ~Timers ();
};

    /* a single timer */
 
struct Timer
{
    struct timeval stop;    /* timer expiration */
    TimerTrap *trap;       /* timer trap       */

	/* constructors */

    Timer (void);
    Timer (const Timer &timer);
    Timer (struct timeval stop, TimerTrap *trap);

	/* assignment operator */

    Timer& operator= (const Timer &timer) {
	stop = timer.stop;
        trap = timer.trap;
	return (*this);
    }
};

    /* all active timers */

class TimerQueue 
{
	int timerMax;           /* maxium number of timers in the heap  */
	int timerCount;         /* number of timers in the heap         */
	Timer *timerHeap;       /* a min-heap of timers                 */

	    /* swap two elements */

	void swap (int i, int j);

	    /* bubble an element up */

	int bubble_down (int hole);
	void bubble_up (int i);

	    /* expand the heap if necessary */

	void expand (void);

    public:

	    /* constructor and destructor */

	TimerQueue (void);
       ~TimerQueue (void);

	    /* remove and return the minimum timer */

	void removeMinimum (struct Timer &minTimer);

	    /* add and remove a timer */

	void add (struct timeval stop, TimerTrap *trap);
	void remove (struct timeval stop, TimerTrap *trap);

	    /* remove all timers with a specific trap */

	void removeAllAttachedToTrap (TimerTrap *trap);

	    /* get the minimum timer */

	BOOLEAN getMinimum (struct Timer &timer) {

	    if (timerCount == 0) {
		return e_FALSE;
	    }

	    timer = timerHeap[0];
	    return e_TRUE;
	} 
};

#endif
