/* Bob Gray
   Agent Tcl
   13 July 1996

   platExclusionImp.cc

   This file implements the classes that provide simple mutual exclusion.

   NOTE:

   In the original version of this file, I had variations on the following theme
   at several points:

       class MonitorLock {
	   BOOLEAN m_waitingToAcquire;
	   BOOLEAN m_canAcquire;
	   pthread_mutex_t m_lock;
	   pthread_cond_t m_condition;
	   void release (void);
	   ...
       }

       void MonitorLock::release (void) {

	   BOOLEAN sendSignal = FALSE;

	   pthread_mutex_lock (&m_lock);

	   if (m_waitingToAcquire) {
	       m_canAcquire = TRUE; 
	       sendSignal = TRUE;
	   }

	   pthread_mutex_unlock (&m_lock);

	   if (sendSignal) {
	       pthread_cond_signal (&m_condition);
	   }
       }

   This logic would be fine if m_condition was NOT a per-object variable and 
   was guaranteed to outlive the MonitorLock instance.  However, m_condition 
   is a per-object variable, which will be deleted by the destructor.  To see
   why this is a problem, suppose that one thread (A) needs to be signaled by
   some other thread (B) when a particular task is finished.  The two threads
   use a MonitorLock instance (X) for this synchronization.  In addition, 
   suppose A does some other work while waiting for B.  In other words, 
   thread A is not always blocked in the MonitorLock::wait method.  Now consider
   the following sequence

	A: doing some work and 
           eventually yields the 
           CPU
                                    B: does some work and decides the task is
                                       finished
	                            B: invokes X.acquire();
	                            B: sets a "done" flag to TRUE
				    B: invokes X.wakeup();
				    B: invokes X.release() BUT YIELDS THE CPU
				       RIGHT AFTER THE pthread_mutex_unlock
                                       call in MonitorLock::wakeup()
        A: decides to check if
           the task is finished
        A: invokes X.acquire() 
           WHICH SUCCEEDS SINCE
	   THE m_canAcquire FLAG
           IS SET (as it should be)
        A: sees the "done" flag
           is set 
        A: invokes X.release()
    ==> A: deletes X as part of
           task cleanup and
           eventually yields the
           CPU
				    B: SEG FAULT on the pthread_cond_signal
                                       call since X and hence m_condition has
                                       been deleted

   There are two basic ways to fix this:

   (1) Move the pthread_cond_signal inside the mutual section and suffer a
       slight performance penalty (since we sometimes wakeup a thread that then
       immediately blocks waiting for m_lock).

   (2) Require the client code to further synchronize their task cleanup (i.e.,
       with a second supporting MonitorLock THAT IS NOT DELETED AS PART OF
       TASK CLEANUP).

   (3) Add an extra pthread_mutex_t to control deletion, which would obviously
       require obtain two locks rather than just one in most method calls

   We adopt solution (1) for now, since the performance hit is much smaller
   than the performance hit of solution (3), and seems minor compared to the 
   programming headaches of solution (2).  The client programmer should be able
   to assume that MonitorLock::acquire will not succeed until MonitorLock::
   release() has truly finished (i.e., no longer needs the MonitorLock instance
   to remain in existence).
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "platExclusionImp.h"
#include "platSystem.h"
#include "platTimeval.h"
#include "platThreadPort.h"

#ifdef FIX_LATER
    /*
     * check the return code from all the pthread_* functions
     */
#endif

    // static variables

#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)

extern "C" int pthread_mutexattr_setkind_np 
	(pthread_mutexattr_t *attr, int kind);

#ifndef PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP
# define PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP \
  {0, 0, 0, PTHREAD_MUTEX_RECURSIVE_NP, {0, 0}}
#endif

MUTEX_T MonitorLockImp::referenceMutex = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
MUTEX_T MutexLockImp::referenceMutex   = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

#else

MUTEX_T MonitorLockImp::referenceMutex = PTHREAD_MUTEX_INITIALIZER;
MUTEX_T MutexLockImp::referenceMutex   = PTHREAD_MUTEX_INITIALIZER;

#endif

    // condition variables

ConditionVariable::ConditionVariable (void)
{
    pthread_cond_init (&waitingOnCondition, NULL);
    waitingCount  = 0;
    signaledCount = 0;
}

ConditionVariable::~ConditionVariable (void)
{
    pthread_cond_destroy (&waitingOnCondition);
}

    // get a condition variable

BOOLEAN MonitorLock::getConditionVariable (UINT_32 &handle)
{
	// acquire the lock

    pthread_mutex_lock (&(m_this -> mutex));

	// expand the condition-variable array if needed

    if (m_this -> conditionFirstFree == -1) {

	UINT_32 i;
	int *newFree;
	UINT_32 newSlots;
	ConditionVariable **newVars;

	newSlots = (m_this -> numConditionSlots) << 1;
	newVars  = new ConditionVariable * [newSlots];
	newFree  = new int [newSlots];

	for (i = 0; i < m_this -> numConditionSlots; ++i) {
	    newVars[i] = m_this -> userConditionVariables[i];
	    newFree[i] = -1;
	}

	for (i = m_this -> numConditionSlots; i < newSlots; ++i) {
	    newVars[i] = NULL;
	    newFree[i] = -1;
	}

	for (i = m_this -> numConditionSlots; i < newSlots - 1; ++i) {
	    newFree[i] = i + 1;
	}

	m_this -> conditionFirstFree = m_this -> numConditionSlots;
	m_this -> numConditionSlots  = newSlots;
	DELETE_ARRAY_OBJECT(m_this->userConditionVariables);
	DELETE_ARRAY_OBJECT(m_this -> conditionFreeList);
	m_this -> userConditionVariables = newVars;
	m_this -> conditionFreeList = newFree;
    }

	// get a condition variable

    handle = m_this -> conditionFirstFree;
    m_this -> userConditionVariables[handle] = new ConditionVariable();
    m_this -> conditionFirstFree = m_this -> conditionFreeList[handle]; 
   
	// done
 
    pthread_mutex_unlock (&(m_this -> mutex));
    return (e_TRUE);
}

    // return a condition variable

BOOLEAN  MonitorLock::returnConditionVariable (UINT_32 handle)
{
    BOOLEAN rc = e_FALSE;

	// acquire the lock

    pthread_mutex_lock (&(m_this -> mutex));

	// delete the condition variable and put its slot onto the free list
	// (if the given handle refers to a valid condition variable)

    if (handle >= m_this -> numConditionSlots) {

	rc = e_FALSE;

    } else if (m_this -> userConditionVariables[handle] == NULL) {

	rc = e_FALSE;

    } else {

	DELETE_OBJECT(m_this->userConditionVariables[handle]);
	m_this -> userConditionVariables[handle] = NULL;
	m_this -> conditionFreeList[handle] = m_this -> conditionFirstFree;
	m_this -> conditionFirstFree = handle;
	rc = e_TRUE;
    }

	// done
 
    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}


    // acquire and release

BOOLEAN MutexLock::acquire (void)
{
    pthread_mutex_lock (&(m_this -> mutex));
    return (e_TRUE);
}
    
BOOLEAN MutexLock::try_acquire (void)
{
    abort_with_message ("MutexLock::try_acquire has not been implemented");
    return (e_FALSE);
}

BOOLEAN MutexLock::release (void) 
{
    pthread_mutex_unlock (&(m_this -> mutex));
    return (e_TRUE);
}

    // acquire, release and wait

BOOLEAN MonitorLock::acquire (void) 
{
    pthread_mutex_lock (&(m_this -> mutex));

    if (m_this -> lockCount != 0) {

	if (m_this -> holdingThread != Thread::self()) {
	    
	    while (m_this -> lockCount != 0) {
		pthread_cond_wait (&(m_this -> waitingToAcquireMonitor), &(m_this -> mutex));
	    }
	}
    }

    m_this -> lockCount    += 1;
    m_this -> holdingThread = Thread::self();
    pthread_mutex_unlock (&(m_this -> mutex));
    return (e_TRUE);
}
    
BOOLEAN MonitorLock::try_acquire (void)
{
    BOOLEAN acquired;

    pthread_mutex_lock (&(m_this -> mutex));

    if (m_this -> lockCount == 0) {
	acquired = e_TRUE;
    } else if (m_this -> holdingThread == Thread::self()) {
	acquired = e_TRUE;
    } else {
	acquired = e_FALSE;
    }

    if (acquired) {
	m_this -> lockCount    += 1;
	m_this -> holdingThread = Thread::self();
    }

    pthread_mutex_unlock (&(m_this -> mutex));
    return (acquired);
}

BOOLEAN MonitorLock::release (void) 
{
    pthread_mutex_lock (&(m_this -> mutex));
    
    m_this -> lockCount -= 1;
    
    if (m_this -> lockCount == 0) {
	m_this -> holdingThread = ThreadId();
	pthread_cond_signal (&(m_this -> waitingToAcquireMonitor));
    }

    pthread_mutex_unlock (&(m_this -> mutex));
    return (e_TRUE);
}

static BOOLEAN monitorWait (MonitorLockImp *m_this, ConditionVariable *m_condition, BOOLEAN debug = e_FALSE)
{
    ThreadId threadId;
    unsigned int lockCount;

	// release the monitor

    threadId  = m_this -> holdingThread;
    lockCount = m_this -> lockCount;
    m_this -> holdingThread = ThreadId();
    m_this -> lockCount     = 0;
    pthread_cond_signal (&(m_this -> waitingToAcquireMonitor));

	// wait until the monitor's condition variable is signalled

    if (debug) {
	printf ("DEBUG: going to sleep!\n");
    }

    m_condition -> waitingCount += 1;
 
    while (m_condition -> signaledCount == 0) {

	pthread_cond_wait (&(m_condition -> waitingOnCondition), &(m_this -> mutex));

	if (debug) {
	    printf ("DEBUG: awake and checking (%d/%d)!\n", m_condition->waitingCount, m_condition->signaledCount);
	}
    }

    m_condition -> waitingCount  -= 1;
    m_condition -> signaledCount -= 1;

	// reacquire the monitor

    while (m_this -> lockCount != 0) {
	pthread_cond_wait (&(m_this -> waitingToAcquireMonitor), &(m_this -> mutex));
    }

    if (debug) {
	printf ("DEBUG: now awake!\n");
    }

    m_this -> holdingThread = threadId;
    m_this -> lockCount     = lockCount;
    pthread_mutex_unlock (&(m_this -> mutex));
    return (e_TRUE);
}

BOOLEAN MonitorLock::wait (void) 
{
    BOOLEAN rc;
    pthread_mutex_lock (&(m_this -> mutex));
    rc = monitorWait (m_this, &(m_this -> builtinConditionVariable));
    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}

BOOLEAN MonitorLock::wait (UINT_32 conditionHandle, BOOLEAN debug)
{
    BOOLEAN rc;

    pthread_mutex_lock (&(m_this -> mutex));

    if (conditionHandle >= m_this -> numConditionSlots) {
	rc = e_FALSE;
    } else if (m_this -> userConditionVariables[conditionHandle] == NULL) {
	rc = e_FALSE;
    } else {
	rc = monitorWait (m_this, m_this -> userConditionVariables[conditionHandle], debug);
    } 

    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}

static BOOLEAN monitorTimedWait (MonitorLockImp *m_this, ConditionVariable *m_condition, struct timeval delay)
{
    ThreadId threadId;
    unsigned int lockCount;

	// release the monitor
 
    threadId  = m_this -> holdingThread;
    lockCount = m_this -> lockCount;
    m_this -> holdingThread = ThreadId();
    m_this -> lockCount     = 0;
    pthread_cond_signal (&(m_this -> waitingToAcquireMonitor));

	// wait until the monitor's condition variable is signalled
	// or until the timeout expires

    BOOLEAN timeoutExpired = e_FALSE;
    struct timeval stopTimeval = TimevalUtil::addTimevals (delay, SystemUtil::getCurrentWall());
    struct timespec stopTimespec;
    stopTimespec.tv_sec  = stopTimeval.tv_sec;
    stopTimespec.tv_nsec = stopTimeval.tv_usec * 1000;

    m_condition -> waitingCount += 1;

    while ((m_condition -> signaledCount == 0) && (!timeoutExpired)) {

	int rc = pthread_cond_timedwait (&(m_condition -> waitingOnCondition), &(m_this -> mutex), &stopTimespec);

	if (rc == ETIMEDOUT) {
	    timeoutExpired = e_TRUE;
	}
    }
    
    m_condition -> waitingCount -= 1;

    if (m_condition -> signaledCount > 0) {
	m_condition -> signaledCount -= 1;
	timeoutExpired = e_FALSE;
    }

	// reacquire the monitor

    while (m_this -> lockCount != 0) {
	pthread_cond_wait (&(m_this -> waitingToAcquireMonitor), &(m_this -> mutex));
    }

    m_this -> holdingThread = threadId;
    m_this -> lockCount     = lockCount;
    return (timeoutExpired ? e_FALSE : e_TRUE);
}

BOOLEAN MonitorLock::timed_wait (struct timeval delay)
{
    BOOLEAN rc;
    pthread_mutex_lock (&(m_this -> mutex));
    rc = monitorTimedWait (m_this, &(m_this -> builtinConditionVariable), delay);
    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}

BOOLEAN MonitorLock::timed_wait (UINT_32 conditionHandle, struct timeval delay)
{
    BOOLEAN rc;

    pthread_mutex_lock (&(m_this -> mutex));

    if (conditionHandle >= m_this -> numConditionSlots) {
	rc = e_FALSE;
    } else if (m_this -> userConditionVariables[conditionHandle] == NULL) {
	rc = e_FALSE;
    } else {
	rc = monitorTimedWait (m_this, m_this -> userConditionVariables[conditionHandle], delay);
    }

    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}
    
static BOOLEAN monitorWakeup (ConditionVariable *m_condition, BOOLEAN &sendSignal, BOOLEAN debug = e_FALSE)
{
    sendSignal = e_FALSE;

    if (debug) {
	printf ("DEBUG: send signal?  Is %d < %d?\n", m_condition->signaledCount, m_condition->waitingCount);
    }

    if (m_condition -> signaledCount < m_condition -> waitingCount) {
	m_condition -> signaledCount += 1;
	sendSignal = e_TRUE;
    }

    return (e_TRUE);
}

BOOLEAN MonitorLock::wakeup (void)
{
    BOOLEAN rc;
    BOOLEAN sendSignal = e_FALSE;

    pthread_mutex_lock (&(m_this -> mutex));

    rc = monitorWakeup (&(m_this -> builtinConditionVariable), sendSignal);

    if (sendSignal) {
	pthread_cond_signal (&(m_this -> builtinConditionVariable.waitingOnCondition));
    }

    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}

BOOLEAN MonitorLock::wakeup (UINT_32 conditionHandle, BOOLEAN debug)
{
    BOOLEAN rc;
    BOOLEAN sendSignal = e_FALSE;

    pthread_mutex_lock (&(m_this -> mutex));

    if (conditionHandle >= m_this -> numConditionSlots) {
	rc = e_FALSE;
    } else if (m_this -> userConditionVariables[conditionHandle] == NULL) {
	rc = e_FALSE;
    } else {
	rc = monitorWakeup (m_this -> userConditionVariables[conditionHandle], sendSignal, debug);
    }

    if (sendSignal) {

	if (debug) {
	    printf ("DEBUG: sending wakeup signal!\n");
	}

	pthread_cond_signal 
	    (&(m_this->userConditionVariables[conditionHandle]->
		waitingOnCondition));
    }

    pthread_mutex_unlock (&(m_this -> mutex));
    return (rc);
}

    /* constructors, destructor and assignment operator */

MonitorLockImp::MonitorLockImp (void)
{
	// variables that control entry to the monitor and ensure that the
	// monitor is not actually deleted until all references to it are 
	// gone

#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)
    pthread_mutexattr_init (&mutexAttr);
    pthread_mutexattr_setkind_np (&mutexAttr, PTHREAD_MUTEX_RECURSIVE_NP);
    pthread_mutex_init (&mutex, &mutexAttr);
#else
    pthread_mutex_init (&mutex, NULL);
#endif
    pthread_cond_init (&waitingToAcquireMonitor, NULL);
    referenceCount = 1;
    lockCount      = 0;

	// user-specified condition variables

#ifdef FIX
	/*
	 * many MonitorLock instances do not use extra condition variables,
	 * so we should not do this until we know that we need one
	 */
#endif

    numConditionSlots      = e_DEFAULT_CONDITION_VARIABLES;
    userConditionVariables = new ConditionVariable * [numConditionSlots];
    conditionFreeList      = new int                 [numConditionSlots];

    for (UINT_32 i = 0; i < numConditionSlots; ++i) {
	userConditionVariables[i] = NULL;
	conditionFreeList     [i] = -1;
    }

    for (UINT_32 i = 0; i < numConditionSlots - 1; ++i) {
	conditionFreeList[i] = i + 1;
    }

    conditionFirstFree = 0;
}

MonitorLockImp::~MonitorLockImp ()
{
#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)
    pthread_mutexattr_destroy (&mutexAttr);
#endif
    pthread_mutex_destroy (&mutex);
    pthread_cond_destroy  (&waitingToAcquireMonitor);

    for (UINT_32 i = 0; i < numConditionSlots; ++i) {
	DELETE_OBJECT(userConditionVariables[i]);
    }

    DELETE_ARRAY_OBJECT(userConditionVariables);
    DELETE_ARRAY_OBJECT(conditionFreeList);
}

MutexLockImp::MutexLockImp (void)
{
	// lock and reference count

#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)
    pthread_mutexattr_init (&mutexAttr);
    pthread_mutexattr_setkind_np (&mutexAttr, PTHREAD_MUTEX_RECURSIVE_NP);
    pthread_mutex_init (&mutex, &mutexAttr);
#else
    pthread_mutex_init (&mutex, NULL);
#endif
    referenceCount = 1;
}

MutexLockImp::~MutexLockImp ()
{
#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)
    pthread_mutexattr_destroy (&mutexAttr);
#endif
    pthread_mutex_destroy (&mutex);
}

MonitorLock::MonitorLock (void)
{
    m_this = new MonitorLockImp ();
}

MonitorLock::MonitorLock (const MonitorLock& lock)
{
    pthread_mutex_lock (&MonitorLockImp::referenceMutex);
    m_this                    = lock.m_this;
    m_this -> referenceCount += 1;
    pthread_mutex_unlock (&MonitorLockImp::referenceMutex);
}

MonitorLock& MonitorLock::operator= (const MonitorLock& lock)
{
    if (this == &lock) {
	return (*this);
    }

    pthread_mutex_lock (&MonitorLockImp::referenceMutex);

    if (m_this != lock.m_this) {

	if (--m_this->referenceCount == 0) {
	    DELETE_OBJECT(m_this);
	}

	m_this                    = lock.m_this;
	m_this -> referenceCount += 1;
    }

    pthread_mutex_unlock (&MonitorLockImp::referenceMutex);
    return (*this);
}

MonitorLock::~MonitorLock ()
{
    pthread_mutex_lock (&MonitorLockImp::referenceMutex);

    if (--m_this->referenceCount == 0) {
	DELETE_OBJECT(m_this);
    }

    pthread_mutex_unlock (&MonitorLockImp::referenceMutex);
}

MutexLock::MutexLock (void)
{
    m_this = new MutexLockImp ();
}

MutexLock::MutexLock (const MutexLock& lock)
{
    pthread_mutex_lock (&MutexLockImp::referenceMutex);
    m_this                    = lock.m_this;
    m_this -> referenceCount += 1;
    pthread_mutex_unlock (&MutexLockImp::referenceMutex);
}

MutexLock& MutexLock::operator= (const MutexLock& lock)
{
    if (this == &lock) {
	return (*this);
    }

    pthread_mutex_lock (&MutexLockImp::referenceMutex);

    if (m_this != lock.m_this) {

	if (--m_this->referenceCount == 0) {
	    DELETE_OBJECT(m_this);
	}

	m_this                    = lock.m_this;
	m_this -> referenceCount += 1;
    }

    pthread_mutex_unlock (&MutexLockImp::referenceMutex);
    return (*this);
}

MutexLock::~MutexLock ()
{
    pthread_mutex_lock (&MutexLockImp::referenceMutex);

    if (--m_this->referenceCount == 0) {
	DELETE_OBJECT(m_this);
    }

    pthread_mutex_unlock (&MutexLockImp::referenceMutex);
}

/*
 * MonitorOnce::doOnce is based on the pthread_once implementation from
 * the Linux Threads package.
 */

static MUTEX_T     s_onceMasterLock = PTHREAD_MUTEX_INITIALIZER;
static CONDITION_T s_onceFinished   = PTHREAD_COND_INITIALIZER;

void MonitorOnce::doOnce (OnceControl &onceControl, OnceTrap &trap)
{
	// check without locking first for speed

    if (onceControl == e_DONE) {
	return;
    }

	// lock and test again

    pthread_mutex_lock (&s_onceMasterLock);

	// if the initialization trap is being called from another routine,
	// wait until it finishes

    while (onceControl == e_IN_PROGRESS) {
	pthread_cond_wait (&s_onceFinished, &s_onceMasterLock);
    }
	
	// do the initialization if it has not been done yet.  Note 
	// that here onceControl must be either e_NEVER or e_DONE.

    if (onceControl == e_NEVER) {
	onceControl = e_IN_PROGRESS;
	pthread_mutex_unlock (&s_onceMasterLock);
	trap.initialize();
	pthread_mutex_lock (&s_onceMasterLock);
	onceControl = e_DONE;
	pthread_cond_broadcast (&s_onceFinished);
    }

    pthread_mutex_unlock (&s_onceMasterLock);
}
