/* Agent Tcl
   Bob Gray
   22 January 1998

   plstInterrupt.cc

   This file implements a reliable signal handler "install_signal".  This
   handler is based on the examples in "UNIX Network Programming" by W. Richard
   Stevens [Prentice-Hall, 1990].

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platThread.h"
#include "platThreadPort.h"
#include "truefalse.h"

    // constants -- c_endFlag marks the end of the signal list

const int c_endFlag =
	0xFFFF;

    // type definitions -- funcType is the type of a signal handler and
    // SignalInformation is information about each signal

typedef void (*funcType) (int, SignalHandlers::ClientData);

struct SignalInformation
{
	// signal type

    enum SignalType {
        e_Synchronous,
        e_Asynchronous,
	e_Unsupported     // by the Signal class, not necessary by the
    };			  // underlying system

	// signal information

    int signo;             		     // signal number
    funcType handler;      		     // handler 
    SignalHandlers::ClientData clientData;   // client data for handler
    SignalType type;                         // type of signal
#ifdef _LINUXTHREADS
    BOOLEAN fired;                           // e_TRUE if signal has fired
#endif
    BOOLEAN catchable;                       // e_TRUE if client program can 
                                             // associate a handler with the 
				             // signal

	// constructors and destructor

    SignalInformation ();
    SignalInformation (int signo);
   ~SignalInformation ();

	// set the signo

    void setSigno (int p_signo) { signo = p_signo; };

private:

	// no copy constructor or assignment operator

    SignalInformation (const SignalInformation&);
    SignalInformation& operator= (const SignalInformation&);
};

    // static data

static sigset_t s_signalsToCatch;
static MUTEX_T s_mutex;
static int s_signalArraySize;
static struct SignalInformation *s_signalArray;
#ifdef _LINUXTHREADS
static pid_t s_signalHandlerPid;
#endif

static struct {
    int signo;
    SignalInformation::SignalType type;
} s_signalList [] = {
    {SIGHUP   , SignalInformation::e_Asynchronous},
    {SIGPIPE  , SignalInformation::e_Asynchronous},
    {SIGTERM  , SignalInformation::e_Asynchronous},
    {SIGALRM  , SignalInformation::e_Asynchronous},
    {SIGINT   , SignalInformation::e_Asynchronous},
    {SIGQUIT  , SignalInformation::e_Asynchronous},
    {SIGPROF  , SignalInformation::e_Asynchronous},
    {SIGIO    , SignalInformation::e_Asynchronous},
    {SIGCHLD  , SignalInformation::e_Asynchronous},
    {SIGSEGV  , SignalInformation::e_Synchronous },
#ifdef SIGBUS
    {SIGBUS   , SignalInformation::e_Synchronous },
#endif
#ifdef SIGABORT
    {SIGABORT , SignalInformation::e_Synchronous },
#endif
#ifdef SIGABRT
    {SIGABRT  , SignalInformation::e_Synchronous },
#endif
    {c_endFlag, SignalInformation::e_Unsupported }
};

    // SignalInformation methods

SignalInformation::SignalInformation (void):
    signo (0),
    handler ((funcType) SIG_DFL),
    clientData ((SignalHandlers::ClientData) NULL),
    type (SignalInformation::e_Unsupported),
#ifdef _LINUXTHREADS
    fired (e_FALSE),
#endif
    catchable (e_FALSE)
{
    // empty
}

SignalInformation::SignalInformation (int p_signo):
    signo (p_signo),
    handler ((funcType) SIG_DFL),
    clientData ((SignalHandlers::ClientData) NULL),
    type (SignalInformation::e_Unsupported),
#ifdef _LINUXTHREADS
    fired (e_FALSE),
#endif
    catchable (e_FALSE)
{
    // empty
}

SignalInformation::~SignalInformation ()
{
    // empty
}

    // SignalHandlerBody is the thread that handles asynchronous signals

class SignalInitializer;

class SignalHandlerBody: public ThreadBody
{
	// initializer that is creating this thread

    SignalInitializer *m_signalInitializer;

	// handle a signal

    void handleSignal (int signo);

public:

	// loop and wait for a signal

    void run (void);

	// constructor and destructor

    SignalHandlerBody (SignalInitializer *signalInitializer):
	m_signalInitializer (signalInitializer)
    {
	// empty
    }
};

    // SignalInitializer is the initializer that sets up the signal-handler
    // thread and the associated signal masks

class SignalInitializer: public OnceTrap
{
    MonitorLock m_monitor;
    BOOLEAN m_signalThreadReady;

public:

    SignalInitializer (void):
	m_signalThreadReady (e_FALSE) 
    {
	// empty
    }

    void initialize (void);

    void markSignalThreadReady (void) {
	m_monitor.acquire();
	m_signalThreadReady = e_TRUE;
	m_monitor.wakeup();
	m_monitor.release();
    }

    virtual ~SignalInitializer() {
	// empty
    }
};

    // methods

void SignalHandlerBody::handleSignal (int signo) {

        // acquire the mutex and handle the signal

    pthread_mutex_lock (&s_mutex);

    if ((signo >= s_signalArraySize) ||
	(s_signalArray[signo].catchable == e_FALSE)) {
	char buffer[128];
	sprintf (buffer, 
	    "Signal handling subsystem: caught an unexpected asynchronous " \
	    "signal %s\n", SignalHandlers::SignoToString(signo));
	abort_with_message (buffer);
    }

    if (s_signalArray[signo].handler == (funcType) SIG_IGN) {

	pthread_mutex_unlock (&s_mutex);

    } else if (s_signalArray[signo].handler == (funcType) SIG_DFL) {

	if (signo != SIGCHLD) {
	    char buffer[128];
	    sprintf (buffer, 
		"Signal handling subsystem: caught an asynchronous signal " \
		"%s with no defined handler: default action is to terminate", 
		SignalHandlers::SignoToString(signo));
	    abort_with_message (buffer);
	}

	pthread_mutex_unlock (&s_mutex);

    } else {

	funcType func = s_signalArray[signo].handler;
	SignalHandlers::ClientData clientData = s_signalArray[signo].clientData;
	pthread_mutex_unlock (&s_mutex);
	(*func) (signo, clientData);
    }
}

void SignalHandlerBody::run (void) {

#ifdef _LINUXTHREADS

	    // Remember the pid of the signal-handling thread (i.e., this thread).
	    // This is necessary for those platforms on which each kernel thread
	    // has its own process id.

	s_signalHandlerPid = getpid();

#endif

	    // Tell the initializer the we are ready to proceed

  	m_signalInitializer->markSignalThreadReady();

	    // wait for signals

	while (1) {

		// wait for a signal

#ifdef _LINUXTHREADS

	    sigset_t emptySet;
	    sigemptyset (&emptySet);
	    sigsuspend (&emptySet);

	    for (int i = 0; s_signalList[i].signo != c_endFlag; ++i) {

		int signo = s_signalList[i].signo;

	 	if (s_signalArray[signo].fired) {
		    handleSignal (signo);
		    s_signalArray[signo].fired = e_FALSE;
	   	}
	    }

#else

	    int signo;
	    sigwait (&s_signalsToCatch, &signo);
	    handleSignal (signo);

#endif
	}
}

void synchronousHandler (int signo)
{
	// Note that this is a handler for SYNCHRONOUS signals such as
	// SIGSEGV, SIGABRT, etc.

    if ((signo < s_signalArraySize) && s_signalArray[signo].catchable) {

	funcType func = s_signalArray[signo].handler;
	SignalHandlers::ClientData clientData = s_signalArray[signo].clientData;
	(*func) (signo, clientData);

    } else {

	char buffer[128];
	sprintf (buffer, "Signal handling subsystem: " \
	    "caught an unexpected synchronous signal %s\n",
	    SignalHandlers::SignoToString(signo));
	abort_with_message (buffer);
    }
}

void dummyHandler (int signo) 
{

#ifdef _LINUXTHREADS

    assert (s_signalHandlerPid != 0);

    if (signo == SIGABRT) {
	exit (1);
    }

    if (getpid() != s_signalHandlerPid) {

	kill (s_signalHandlerPid, signo);

    } else {

	if ((signo < s_signalArraySize) && s_signalArray[signo].catchable) {
	    s_signalArray[signo].fired = e_TRUE;
	}
    }

#endif
}

static void installUnixSignalHandler (SignalInformation& signalInformation)
{
    struct sigaction act;
    int signo = signalInformation.signo;

    if (signalInformation.type == SignalInformation::e_Unsupported) {

	char buffer[128];
	sprintf (buffer, 
	    "Signal handling subsystem: can not install a Unix signal " \
	    "handler for signal %s\n", SignalHandlers::SignoToString(signo));
	abort_with_message (buffer);

    } else if (signalInformation.type == SignalInformation::e_Asynchronous) {

#ifdef SIGACTION_CAST
	act.sa_handler = (void (*)(...)) dummyHandler;
#else
#ifdef SIGACTION_PAREN_CAST
	act.sa_handler = (void (*)()) dummyHandler;
#else
	act.sa_handler = (void (*)(int)) dummyHandler;
#endif
#endif

    } else if (signalInformation.handler == (funcType) SIG_DFL) {

	act.sa_handler = SIG_DFL;

    } else if (signalInformation.handler == (funcType) SIG_IGN) {

	act.sa_handler = SIG_IGN;

    } else {

#ifdef SIGACTION_CAST
	act.sa_handler = (void (*)(...)) synchronousHandler;
#else
#ifdef SIGACTION_PAREN_CAST
	act.sa_handler = (void (*)()) synchronousHandler;
#else
	act.sa_handler = (void (*)(int)) synchronousHandler;
#endif
#endif
    }

    sigemptyset (&act.sa_mask);
    act.sa_flags = 0;

    if (signo == SIGCHLD) {
#ifdef SA_NOCLDSTOP
	act.sa_flags |= SA_NOCLDSTOP;
#endif
    }
	
    if (sigaction (signo, &act, 0) < 0) {
	abort_with_message ("SignalHandler::initializeSignals: " \
	    "unable to set up Unix signal handler");
    }
}

void SignalInitializer::initialize (void) {

    int i;

	// block all ASYNCHRONOUS signals that are handled through these 
	// classes -- we do not block synchronous signals since they must
	// be handled in the appropriate thread

    sigemptyset (&s_signalsToCatch);

    for (i = 0; s_signalList[i].signo != c_endFlag; ++i) {
	if (s_signalList[i].type == SignalInformation::e_Asynchronous) {
	    sigaddset (&s_signalsToCatch, s_signalList[i].signo);
        }
    }

#ifdef _LINUXTHREADS
    sigprocmask (SIG_BLOCK, &s_signalsToCatch, NULL);
#else
    pthread_sigmask (SIG_BLOCK, &s_signalsToCatch, NULL);
#endif

	// find the maximum signal number (and hence the necessary size of the 
	// signal array)

    s_signalArraySize = 0;

    for (i = 0; s_signalList[i].signo != c_endFlag; ++i) {
	if (s_signalList[i].signo > s_signalArraySize) {
	    s_signalArraySize = s_signalList[i].signo;
	}
    }

    s_signalArraySize += 1;

	// allocate and initialize the signal-information array

    s_signalArray = new SignalInformation [s_signalArraySize];

    for (i = 0; i < s_signalArraySize; ++i) {
	s_signalArray[i].setSigno(i);
    }

	// finish initializing the signals that can actually be caught

    for (i = 0; s_signalList[i].signo != c_endFlag; ++i) {
	int signo = s_signalList[i].signo;
	s_signalArray[signo].catchable = e_TRUE;
	s_signalArray[signo].type = s_signalList[i].type;
    }

	// establish a dummy handler for each ASYNCHRONOUS signal, and make
	// sure that the default action will be taken for each SYNCHRONOUS
	// signal -- Note that with LinuxThreads, the "dummy" handler actually
	// performs the critical task of passing aschronous signals on to the
	// signal-handling thread.  

    for (i = 0; s_signalList[i].signo != c_endFlag; ++i) {
	int signo = s_signalList[i].signo;
	installUnixSignalHandler (s_signalArray[signo]);
    }

	// initialize the mutex

    pthread_mutex_init (&s_mutex, NULL);

   	// create the signal handler thread

    SignalHandlerBody *handlerBody = new SignalHandlerBody (this);
    Thread thread ("signal-handler", Thread::e_DAEMON_THREAD, handlerBody);
    thread.resume();

	// wait for the signal handler thread to tell us that it is ready

    m_monitor.acquire();

    while (m_signalThreadReady != e_TRUE) {
	m_monitor.wait();
    }

    m_monitor.release();

	// if we are using the LINUXTHREADS package, unblock the signals,
	// since each thread in the LINUXTHREADS package has its own 
	// process id and receives signals independently of the rest of the
	// program.  Thus we unblock the signals so that whichever thread
	// gets the signal will call dummyHandler and pass the signal along
	// to the signal-handling thread.

#ifdef _LINUXTHREADS
    sigprocmask (SIG_UNBLOCK, &s_signalsToCatch, (sigset_t *) NULL);
#endif
}

void SignalHandlers::initializeSignals (void)
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

	/* assertions on the parameters */

    if (onceControl != MonitorOnce::e_DONE) {
	SignalInitializer signalInitializer;
	MonitorOnce::doOnce (onceControl, signalInitializer);
    }
}

int SignalHandlers::install_signal 
    (int signo, void (*func) (int, ClientData), ClientData clientData)
{
    pthread_mutex_lock (&s_mutex);

    if (signo >= s_signalArraySize) {
	return (-1);
    }

    if (s_signalArray[signo].catchable == e_FALSE) {
	return (-1);
    }

    s_signalArray[signo].signo      = signo;
    s_signalArray[signo].handler    = func;
    s_signalArray[signo].clientData = clientData;

    if (s_signalArray[signo].type == SignalInformation::e_Synchronous) {
	installUnixSignalHandler (s_signalArray[signo]);
    }

    pthread_mutex_unlock (&s_mutex);
    return (0);
}

int SignalHandlers::install_signal_intr 
    (int signo, void (*func) (int, ClientData), ClientData clientData)
{
    return (install_signal (signo, func, clientData));
}
