/* Agent Tcl
   Bob Gray
   27 March 1997

   platSigtermImp.cc

   This file implements the class which handles the SIGTERM interrupt.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platSigterm.h"
#include "platSigtermImp.h"

    /* static data */

int SigtermHandlerImp::count
	= 0;

int SigtermHandlerImp::exitcode
	= 1;

SigtermHandlerImp *SigtermHandlerImp::head
	= NULL;

MonitorLock *SigtermHandlerImp::lock
	= NULL;

/* gen_sigterm_handler

   Purpose: This procedure is called when SIGTERM fires.

     Input: signo      = SIGTERM
		         (int)

	    clientData = client data
                         (unused)

    Output: The procedure calls the SigtermTrap associated with each
	    instance of SigtermHandler.
*/

void gen_sigterm_handler (int, SignalHandlers::ClientData)
{
    SigtermHandlerImp *next;
    SigtermHandlerImp *handle;

	/* mutual exclusion -- automatically released */ 

    Guard guard (*SigtermHandlerImp::lock);

	/* call the check functions */

    for (handle = SigtermHandlerImp::head; handle != NULL; handle = next) {
	next = handle -> next;
	SigtermHandlerImp::lock -> release ();
	handle -> trap -> sigtermAlarm ();
	SigtermHandlerImp::lock -> acquire ();
    }

	/* exit */
 
    exit (SigtermHandlerImp::exitcode);
}
   
/* SigtermHandler::SigtermHandler

   Purpose: This procedure is the constructor for class SigtermHandler.

     Input: function = function to be called when SIGIO fires
            data     = client data to be passed to the function
*/

class SigtermOnceTrap: public OnceTrap
{
    MonitorLock *&m_lock;
    
public:

    SigtermOnceTrap (MonitorLock *&lock):
	m_lock (lock)
    {
	// empty
    }

    void initialize (void) {
	m_lock = new MonitorLock();
	SignalHandlers::install_signal_intr (SIGTERM, gen_sigterm_handler, 0);
    }
};

SigtermHandler::SigtermHandler (SigtermTrap *trap)
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

	/* assertions on the parameters */

    assert (trap != NULL);

	/* allocate the lock and install the signal handler if necessary */ 

    if (onceControl != MonitorOnce::e_DONE) {
	SigtermOnceTrap onceTrap (SigtermHandlerImp::lock);
	MonitorOnce::doOnce (onceControl, onceTrap);
    }

	/* allocate an m_this and remember the trap */

    m_this = new SigtermHandlerImp ();
    m_this -> trap  = trap;

	/* mutual exclusion -- automatically release */

    Guard guard (*SigtermHandlerImp::lock);
 
	/* is this the first handler? */

    if (m_this -> count == 0) {

	m_this -> head     = m_this;
	m_this -> next     = NULL;
	m_this -> previous = NULL;
	m_this -> count    = 1;

    } else {

	m_this -> head -> previous = m_this;
	m_this -> next             = m_this -> head;
	m_this -> previous         = NULL;
	m_this -> head             = m_this;
	m_this -> count		   += 1;
    }
}

/* SigtermHandler::~SigtermHandler

   Purpose: This procedure is the destructor for class SigtermHandler.
*/

SigtermHandler::~SigtermHandler (void)
{
	/* mutual exclusion -- automatically release */

    Guard guard (*SigtermHandlerImp::lock);
 
	/* splice the instance out of the list */

    m_this -> count -= 1;

    if (m_this -> count == 0) {

	m_this -> head = NULL;

    } else {

	if (m_this -> next != NULL) {
	    m_this -> next -> previous = m_this -> previous;
	}

	if (m_this -> previous != NULL) {
	    m_this -> previous -> next = m_this -> next;
	} else {
	    m_this -> head = m_this -> next;
	}
    }
}

/* SigtermHandler::setExitCode

   Purpose: Set the exit code that will be used when a SIGTERM is received

     Input: exitcode = the exitcode
		       (int) 
*/

void SigtermHandler::setExitCode (int exitcode) 
{
    SigtermHandlerImp::exitcode = exitcode;
}
