/* Agent Tcl
   Bob Gray
   29 September 1995

   platThreadImp.cc

   This file implements the class which handles the creation and execution 
   of threads.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platPerThread.h"
#include "platThread.h"
#include "platThreadImp.h"
#include "platThreadPort.h"
#include "truefalse.h"

    // forward declarations

static void *posix_thread_startup (void *arg);

    // static variables

static PerThreadPointer *s_idToThreadMap  = NULL;

#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)

extern "C" int pthread_mutexattr_setkind_np 
	(pthread_mutexattr_t *attr, int kind);

#ifndef PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP
# define PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP \
  {0, 0, 0, PTHREAD_MUTEX_RECURSIVE_NP, {0, 0}}
#endif

MUTEX_T ThreadImp::referenceMutex = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

#else

MUTEX_T ThreadImp::referenceMutex = PTHREAD_MUTEX_INITIALIZER;

#endif

    // methods

ThreadImp::ThreadImp (ThreadBody *p_threadBody)
{
    referenceCount = 1;
    threadBody     = p_threadBody;
    attachedThread = NULL;
    suspended      = e_TRUE;
    pthread_attr_init (&attributes);
    pthread_attr_setdetachstate (&attributes, PTHREAD_CREATE_DETACHED);
#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)
    pthread_mutexattr_init (&mutexAttr);
    pthread_mutexattr_setkind_np (&mutexAttr, PTHREAD_MUTEX_RECURSIVE_NP);
    pthread_mutex_init (&mutex, &mutexAttr);
#else
    pthread_mutex_init (&mutex, NULL);
#endif
    pthread_cond_init (&waitingToStart, NULL);
}

ThreadImp::~ThreadImp ()
{
    pthread_attr_destroy (&attributes);
#if defined(_LINUXTHREADS) && defined(_LINUXTHREADS_NO_FAST)
    pthread_mutexattr_destroy (&mutexAttr);
#endif
    pthread_mutex_destroy (&mutex);
    pthread_cond_destroy (&waitingToStart);
    DELETE_OBJECT(threadBody);
}

class MapInitializer: public OnceTrap
{
public:
    void initialize (void) {
	s_idToThreadMap = new PerThreadPointer();
    }
};

Thread::Thread (const char *name, Thread::ThreadType threadType, ThreadBody *threadBody) 
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;
	
	// create the id-to-thread map if necessary

    if (onceControl != MonitorOnce::e_DONE) {
	MapInitializer mapInitializer;
	MonitorOnce::doOnce (onceControl, mapInitializer);
    }

	// create the thread

    m_this = new ThreadImp (threadBody);
    pthread_create (&(m_this -> posixId), &(m_this -> attributes), posix_thread_startup, (void *) m_this);
    m_id = tidToThreadId (m_this -> posixId);

	// remember that the thread goes with the id

    s_idToThreadMap -> setValue (m_id, (void *) m_this);
}

Thread::Thread (const Thread& thread)
{
    pthread_mutex_lock (&ThreadImp::referenceMutex);
    m_this	              = thread.m_this;
    m_id	              = thread.m_id;
    m_this -> referenceCount += 1;
    pthread_mutex_unlock (&ThreadImp::referenceMutex);
}

void Thread::resume (void)
{
    pthread_mutex_lock (&(m_this -> mutex));

    if (m_this -> suspended) {
	m_this -> attachedThread = new Thread (*this);
	m_this -> suspended      = e_FALSE;
	pthread_cond_signal (&(m_this -> waitingToStart));
    }

    pthread_mutex_unlock (&(m_this -> mutex));
}

Thread::~Thread ()
{
    pthread_mutex_lock (&ThreadImp::referenceMutex);

    if (--m_this->referenceCount == 0) {
	DELETE_OBJECT(m_this);
    }

    pthread_mutex_unlock (&ThreadImp::referenceMutex);
}

/*
 * STATIC METHODS
 */

void Thread::exit (void)
{
	// delete the dynamically-allocated Thread instance that exists only
	// as long as the thread does

    ThreadId threadId = Thread::self();
    ThreadImp *threadImp = (ThreadImp *) s_idToThreadMap -> getValue(threadId);
    assert (threadImp != NULL);
    Thread *attachedThread = threadImp -> attachedThread;
    threadImp -> attachedThread = NULL;
    DELETE_OBJECT(attachedThread);

	// then exit

    pthread_exit (0);
}

ThreadId Thread::self (void)
{
    return (tidToThreadId (pthread_self()));
}

/*
 * PTHREADS THREAD STARTUP FUNCTION
 */

static void *posix_thread_startup (void *arg)
{
    ThreadImp *m_this = (ThreadImp *) arg;
    assert (m_this != NULL);

	// wait until we can start

    pthread_mutex_lock (&(m_this -> mutex));

    while (m_this -> suspended) {
	pthread_cond_wait (&(m_this -> waitingToStart), &(m_this -> mutex));
    }

    pthread_mutex_unlock (&(m_this -> mutex));

	// run the thread

    assert (m_this -> threadBody != NULL);
    (m_this -> threadBody) -> run();

	// done

    Thread::exit();
    abort_with_message ("should never get here!\n");
}
