/* Agent Tcl
   Bob Gray
   28 Spetember 1995

   platTimersImp.cc

   This file provides the UNIX implementation of Timers class.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platPerThread.h"
#include "platSystem.h"
#include "platThread.h"
#include "platTimers.h"
#include "platTimeval.h"
#include "platTimersImp.h"
#include "truefalse.h"

TimersImplementation *Timers::m_this
	= NULL;

void sigprof_handler (int /*signo*/, SignalHandlers::ClientData arg)
{
	/* get the timer class */

    TimersImplementation *timersImp = (TimersImplementation *) arg;
    assert (timersImp != NULL);

	/* wakeup the timer thread if it is asleep */

    Guard guard (timersImp -> timerLock);
    timersImp -> timerLock.wakeup ();
}

class TimerBody: public ThreadBody
{
    TimersImplementation *m_timersImp;

public:

    TimerBody (TimersImplementation *timersImp):
	m_timersImp (timersImp)
    {
	// empty
    }

   ~TimerBody () {
	DELETE_OBJECT(m_timersImp);
    }

    void run (void)
    {
        int rc;
        struct Timer timer;      
        struct timeval wall;
        struct timeval cpu;     
        struct timeval interval; 
        struct itimerval itimer; 

 	    /* obtain the lock */

        Guard guard (m_timersImp -> timerLock);

	    /* install the SIGPROF interrupt handler */

	SignalHandlers::install_signal (SIGPROF, sigprof_handler, (SignalHandlers::ClientData) m_timersImp);

	    /* loop forever */

        while (1) {

	        /* dispose of the wall timers */

	    wall = SystemUtil::getCurrentWall();

            while ((m_timersImp -> wallTimers.getMinimum (timer))  && 
	           (TimevalUtil::compareTimevals (wall, timer.stop) >= 0)) 
	    {
	 	    /* remove the expired timer */

	        m_timersImp -> wallTimers.removeMinimum (timer);

		    /* release the lock, call the timer trap and reacquire the lock */

	        m_timersImp -> timerLock.release ();
 
	        if (timer.trap != NULL) {
	            timer.trap -> wallAlarm ();
	        } 

	        m_timersImp -> timerLock.acquire ();
	        wall = SystemUtil::getCurrentWall ();
	    }

	        /* dispose of the cpu timers */

	    cpu = SystemUtil::getCurrentCpu ();

	    while (((rc = m_timersImp -> cpuTimers.getMinimum(timer)) > 0) && 
	           (TimevalUtil::compareTimevals (cpu, timer.stop) >= 0)) 
	    {
		    /* remove the expired timer */

	        m_timersImp -> cpuTimers.removeMinimum (timer);

		    /* release the lock, call the timer trap and reacquire the lock */

	        m_timersImp -> timerLock.release ();
 
	        if (timer.trap != NULL) {
		    timer.trap -> cpuAlarm ();
	        }

	        m_timersImp -> timerLock.acquire ();
	        cpu = SystemUtil::getCurrentCpu ();
	    }

	        /* set a SIGPROF for the next cpu timer (if we have one) */

	    if (rc > 0) {
	        interval = TimevalUtil::subTimevals (timer.stop, cpu);
	        itimer.it_interval.tv_sec = 0;
	        itimer.it_interval.tv_usec = 0;
	        itimer.it_value.tv_sec = interval.tv_sec;
	        itimer.it_value.tv_usec = interval.tv_usec;
	        setitimer (ITIMER_PROF, &itimer, NULL);
	    }

	        /* sleep until the next wall timer fires */

	    if (m_timersImp -> wallTimers.getMinimum (timer)) {

	        wall = SystemUtil::getCurrentWall ();

	        if (TimevalUtil::compareTimevals (wall, timer.stop) < 0) {
		    interval = TimevalUtil::subTimevals (timer.stop, wall);
		    m_timersImp -> timerLock.timed_wait (interval);
	        } 

	    } else {

	        m_timersImp -> timerLock.wait ();
            }
        }

        abort_with_message ("should never get here");   // should never get here
    }
};

/* TimersImplementation::TimersImplementation

   Purpose: This procedure is the constructor for class TimersImplementation.
*/

TimersImplementation::TimersImplementation (void):
	count (0),
	timerLock (),
	wallTimers (),
	cpuTimers (),
	threadStarted (e_FALSE)
{
	// empty
}

/* Timers::disposeTimers

   Purpose: Dispose of expired timers 

     Input: None

    Output: The procedure disposes of expired timers and calls the
            corresponding handlers.  

      Note: This routine serves no purpose when the timer handler is running
	    in its own thread.  It will go away completely once I figure out 
	    how to replace the "siglongjmp" ugliness in the singled-threaded 
	    Unix implementation of tcpip_getIP.
*/

void Timers::disposeTimers (void)
{
    abort_with_message ("Timers::disposeTimers has not been implemented!");
}

/* Timers::Timers

   Purpose: This procedure is the constructor for class TIMERS.
*/

class TimerOnceTrap: public OnceTrap
{
    TimersImplementation *&m_timersImp;

public:

    TimerOnceTrap (TimersImplementation *&timersImp):
	m_timersImp (timersImp)
    {
	// empty
    }
  
    void initialize (void) {
	m_timersImp = new TimersImplementation();
    }
};

Timers::Timers (void)
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

	// allocate the timer heap if necessary

    if (onceControl != MonitorOnce::e_DONE) {
	TimerOnceTrap onceTrap (m_this);
	MonitorOnce::doOnce (onceControl, onceTrap);
    }

	// guard all further access

    Guard guard (m_this -> timerLock);
    m_this -> count += 1;

	// start up the timer thread

    if (!(m_this -> threadStarted)) {
	m_this -> threadStarted = e_TRUE;
	TimerBody *timerBody = new TimerBody (m_this);
	Thread thread ("timer-handler", Thread::e_DAEMON_THREAD, timerBody);
	thread.resume ();
    }
}

/* Timers::~Timers

   Purpose: This procedure is the destructor for class TIMER.
*/

Timers::~Timers ()
{
    {
	Guard guard (m_this -> timerLock);   // automatically released 
	m_this -> count -= 1;
    }
}

/* Timers::add

   Purpose: Add a timer to the set of timers

     Input: stop = time at which the timer should expire
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure adds the timer to the set of timers.
*/

void Timers::add (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);     // automatically released
    m_this -> wallTimers.add (stop, trap);
    m_this -> timerLock.wakeup ();
}

/* Timers::addCpu

   Purpose: Add a cpu timer to the set of timers

     Input: stop = time at which the timer should expire
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure adds the timer to the set of timers.
*/

void Timers::addCpu (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);	    // automatically released
    m_this -> cpuTimers.add (stop, trap);
    m_this -> timerLock.wakeup ();          // wakeup the timer thread
}

/* Timers::remove

   Purpose: Remove a timer from the set of timers

     Input: stop = time at which the timer was to expire 
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure removes the timer from the set of timers. 
*/

void Timers::remove (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);     // automatically released
    m_this -> wallTimers.remove (stop, trap);
}

/* Timers::removeCpu

   Purpose: Remove a cpu timer from the set of timers

     Input: stop = time at which the timer was to expire 
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure removes the timer from the set of timers. 
*/

void Timers::removeCpu (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);     // automatically released
    m_this -> cpuTimers.remove (stop, trap);
}

/* Timers::removeAllAttachedToTrap

   Purpose: Remove all timers with a particular trap (i.e., callback routine)

     Input: trap = TimerTrap instance
		   (class TimerTrap *)

    Output: The procedure removes all timers with the given trap.
*/

void Timers::removeAllAttachedToTrap (TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);	// automatically released
    m_this -> wallTimers.removeAllAttachedToTrap (trap);
    m_this -> cpuTimers.removeAllAttachedToTrap (trap);
}
