/* Bob Gray
   Agent Tcl
   13 July 1996

   platExclusionImp.cc

   This file implements the classes that provide simple mutual exclusion.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
extern "C" {
#include "sys_api.h"
#include "threads.h"
#include "oobj.h"
#include "monitor.h"
#include "interpreter.h"
}
#include "platExclusion.h"
#include "platExclusionImp.h"

    /* keys for the reference count monitors */

unsigned int MonitorLockImp::referenceKey = 0;
unsigned int MutexLockImp::referenceKey = 0;

    /* acquire, release and wait */

BOOLEAN MonitorLock::acquire (void) 
{
    BOOLEAN rc;

    monitorEnter (m_this -> key);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}
    
BOOLEAN MonitorLock::try_acquire (void)
{
    abort_with_message ("MonitorLock::try_acquire has not been implemented!");
    return (e_FALSE);
}

BOOLEAN MonitorLock::release (void) 
{
    BOOLEAN rc;

    monitorExit (m_this -> key);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}

BOOLEAN MonitorLock::wait (void) 
{
    BOOLEAN rc;

    monitorWait (m_this -> key, TIMEOUT_INFINITY);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}

BOOLEAN MonitorLock::timed_wait (struct timeval delay)
{
    BOOLEAN rc;

    int millis = delay.tv_sec * 1000 + delay.tv_usec / 1000;
    monitorWait (m_this -> key, millis);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}

BOOLEAN MonitorLock::wakeup (void)
{
    BOOLEAN rc;

    monitorNotify (m_this -> key);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}

    /* constructors, destructor and assignment operator */

MonitorLock::MonitorLock (void)
{
    assert (EE() != NULL);

	/*
	 * Note that there is no race condition here since the address of
	 * MonitorLockImp::referenceKey never changes.
	 */

#ifdef FIX_LATER
	/*
	 * However we are assuming that a pointer and an unsigned integer are
	 * the same size.
	 */
#endif

    MonitorLockImp::referenceKey = (unsigned int) &MonitorLockImp::referenceKey;
    m_this = new MonitorLockImp ();
    m_this -> key = (unsigned int) m_this;
    m_this -> referenceCount = 1;
}

MonitorLock::MonitorLock (const MonitorLock& lock)
{
    assert (EE() != NULL);
    monitorEnter (MonitorLockImp::referenceKey);
    m_this = lock.m_this;
    m_this->referenceCount += 1;
    monitorExit (MonitorLockImp::referenceKey);
}

MonitorLock& MonitorLock::operator= (const MonitorLock& lock)
{
    if (this == &lock) {
	return (*this);
    }

    assert (EE() != NULL);
    monitorEnter (MonitorLockImp::referenceKey);

    if (m_this != lock.m_this) {

	if (--m_this->referenceCount == 0) {
	    delete (m_this);
	    m_this = NULL;
	}

        m_this = lock.m_this;
        m_this->referenceCount += 1;
    }

    monitorExit (MonitorLockImp::referenceKey);
    return (*this);
}

MonitorLock::~MonitorLock ()
{
    assert (EE() != NULL);
    monitorEnter (MonitorLockImp::referenceKey);

    if (--m_this->referenceCount == 0) {
	delete (m_this);
	m_this = NULL;
    }

    monitorExit (MonitorLockImp::referenceKey);
}

static unsigned int s_onceKey = 0;

void MonitorOnce::doOnce (OnceControl &onceControl, OnceTrap &trap)
{
	// check without locking first for speed

    if (onceControl == e_DONE) {
	return;
    }

	// lock and test again

    s_onceKey = (unsigned int) &s_onceKey;
    monitorEnter (s_onceKey);

	// if the initialization trap is being called from another routine,
	// wait until the other routine finishes and then return; otherwise 
	// call the initialization trap ourselvs if it has not been called
	// already

    if (onceControl == e_IN_PROGRESS) {

	do {
	    monitorWait (s_onceKey, TIMEOUT_INFINITY); 
	} while (onceControl == e_IN_PROGRESS);
	
    } else if (onceControl == e_NEVER) {

	onceControl = e_IN_PROGRESS;
	monitorExit (s_onceKey);
	trap.initialize ();
	monitorEnter (s_onceKey);
	onceControl = e_DONE;
	monitorNotifyAll (s_onceKey);
    }

	// done

    monitorExit (s_onceKey);
}

    /* acquire, release and wait */

BOOLEAN MutexLock::acquire (void) 
{
    BOOLEAN rc;

    monitorEnter (m_this -> key);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}
    
BOOLEAN MutexLock::try_acquire (void)
{
    abort_with_message ("MutexLock::try_acquire has not been implemented!");
    return (e_FALSE);
}

BOOLEAN MutexLock::release (void) 
{
    BOOLEAN rc;

    monitorExit (m_this -> key);

    if (EE() == NULL) {
	rc = e_TRUE;
    } else {
	rc = exceptionOccurred(EE()) ? e_FALSE : e_TRUE;
    }

    return (rc);
}

    /* constructors, destructor and assignment operator */

MutexLock::MutexLock (void)
{
    assert (EE() != NULL);

	/*
	 * Note that there is no race condition here since the address of
	 * MutexLockImp::referenceKey never changes.
	 */

#ifdef FIX_LATER
	/*
	 * However we are assuming that a pointer and an unsigned integer are
	 * the same size.
	 */
#endif

    MutexLockImp::referenceKey = (unsigned int) &MutexLockImp::referenceKey;
    m_this = new MutexLockImp ();
    m_this -> key = (unsigned int) m_this;
    m_this -> referenceCount = 1;
}

MutexLock::MutexLock (const MutexLock& lock)
{
    assert (EE() != NULL);
    monitorEnter (MutexLockImp::referenceKey);
    m_this = lock.m_this;
    m_this->referenceCount += 1;
    monitorExit (MutexLockImp::referenceKey);
}

MutexLock& MutexLock::operator= (const MutexLock& lock)
{
    if (this == &lock) {
	return (*this);
    }

    assert (EE() != NULL);
    monitorEnter (MutexLockImp::referenceKey);

    if (m_this != lock.m_this) {

	if (--m_this->referenceCount == 0) {
	    delete (m_this);
	    m_this = NULL;
	}

        m_this = lock.m_this;
        m_this->referenceCount += 1;
    }

    monitorExit (MutexLockImp::referenceKey);
    return (*this);
}

MutexLock::~MutexLock ()
{
    assert (EE() != NULL);
    monitorEnter (MutexLockImp::referenceKey);

    if (--m_this->referenceCount == 0) {
	delete (m_this);
	m_this = NULL;
    }

    monitorExit (MutexLockImp::referenceKey);
}
