/* Agent Tcl
   Bob Gray
   14 February 1995

   interrupt.cc

   This file implements a reliable signal handler "install_signal".  This
   handler is based on the examples in "UNIX Network Programming" by W. Richard
   Stevens [Prentice-Hall, 1990].

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
extern "C" {
#include "interrupt.h"		// intrRegister
}

/*
 * forward declarations
 */

static int javaSignalHandler (int signo, void *arg, void *siginfo);

/*
 * static data -- list of signal handlers
 */

struct HandlerInfo
{
    friend class SignalHandlerList;
    friend int javaSignalHandler (int signo, void *arg, void *siginfo);

public:

	// constructor and destructor

    HandlerInfo (void);
   ~HandlerInfo ();

private:

	// unimplemented

    HandlerInfo (const HandlerInfo &);
    HandlerInfo& operator= (const HandlerInfo &);

	// data

    SignalHandlers::funcType m_func;
    SignalHandlers::ClientData m_clientData;
};

HandlerInfo::HandlerInfo (void):
    m_func (NULL),
    m_clientData (NULL)
{
    // empty
}

HandlerInfo::~HandlerInfo()
{
    // empty
}

class SignalHandlerList
{
    friend int javaSignalHandler (int signo, void *arg, void *siginfo);

public:

	// constructor and destructor

    SignalHandlerList (void);
   ~SignalHandlerList ();

	// install a new signal handler

    void install (int signo, 
	SignalHandlers::funcType func, SignalHandlers::ClientData clientData);

private:

	// unimplemented

    SignalHandlerList (const SignalHandlerList &);
    SignalHandlerList& operator= (const SignalHandlerList &);

	// data -- we maintain two copies of the signal handler list to
	// avoid race conditions (i.e., we modify one copy, change 
	// the inUse index to instantly put the modified copy into use,
	// and then update the now out-of-use copy to match

    sig_atomic_t m_inUseIndex;
    sig_atomic_t m_updateIndex;
    HandlerInfo *m_handlers[2];
    MutexLock m_lock;
};

SignalHandlerList::SignalHandlerList (void):
    m_inUseIndex (0),
    m_updateIndex (1),
    m_lock()
{
    for (int i = 0; i < 2; ++i) {
	m_handlers[i] = new HandlerInfo [N_INTERRUPTS];
    }
}

SignalHandlerList::~SignalHandlerList()
{
    for (int i = 0; i < 2; ++i) {
	delete (m_handlers[i]);
    }
}

void SignalHandlerList::install (int signo, SignalHandlers::funcType func, SignalHandlers::ClientData clientData)
{
	// see if we are establishing a handler for an invalid signal

    if ((signo < 0) || (signo >= N_INTERRUPTS)) {
	abort_with_message ("invalid signo");
    }

	// acquire the lock for mutual exclusion

    MutexGuard guard (m_lock);

	// determine if we are going to need to call intrRegister (i.e., if
	// there is no current handler) or intrUnregister (i.e., if there is
	// a current handler and we are turning off the handler by specifying
	// a func of NULL

    sig_atomic_t useIndex    = m_inUseIndex;
    sig_atomic_t updateIndex = m_updateIndex;
    BOOLEAN shouldUnregister = e_FALSE;
    BOOLEAN shouldRegister   = e_FALSE;

    if (func == NULL) {

	shouldUnregister = 
	    (m_handlers[useIndex][signo].m_func != NULL) ? e_TRUE : e_FALSE;

    } else {

	shouldRegister = 
	    (m_handlers[useIndex][signo].m_func == NULL) ? e_TRUE : e_FALSE;
    }
   
	// update the signal handler list

    m_handlers[updateIndex][signo].m_func       = func; 
    m_handlers[updateIndex][signo].m_clientData = clientData;

	// make the update list the in-use list (and vice verse) -- note
	// that this atomically puts the updated copy into use

    useIndex      = (useIndex + 1) % 2;
    updateIndex   = (updateIndex + 1) % 2;
    m_inUseIndex  = useIndex;
    m_updateIndex = updateIndex;

	// do the update again so that the lists match

    m_handlers[updateIndex][signo].m_func       = func; 
    m_handlers[updateIndex][signo].m_clientData = clientData;

	// register or unregister the Java signal handler if necessary

    if (shouldRegister) {

	intrRegister (signo, javaSignalHandler, (void *) NULL);

    } else if (shouldUnregister) {

	intrUnregister (signo, javaSignalHandler, (void *) NULL);
    }
}

static SignalHandlerList *s_handlerList = 
    (SignalHandlerList *) NULL;

class ListInitializer: public OnceTrap
{
private:

    static MonitorOnce::OnceControl m_onceControl;

    void initialize (void) {
	s_handlerList = new SignalHandlerList;
    }

public:

    static BOOLEAN isInitialized (void) {
	return ((m_onceControl == MonitorOnce::e_DONE) ? e_TRUE : e_FALSE);
    }

    static void init (void) {
	if (m_onceControl != MonitorOnce::e_DONE) {
	    ListInitializer initializer;
	    MonitorOnce::doOnce (m_onceControl, initializer);
	}
    }
};

MonitorOnce::OnceControl ListInitializer::m_onceControl =
    MonitorOnce::e_NEVER;

/*
 * signal handling
 */

void SignalHandlers::initializeSignals (void)
{
    // empty
}

int SignalHandlers::install_signal (int signo, funcType func, ClientData clientData)
{
	// create the signal handler list if necessary

    if (!ListInitializer::isInitialized()) {
	ListInitializer::init();
    }

	// install the new handler

    s_handlerList->install (signo, func, clientData);
 
	// done

    return 0;
}

int SignalHandlers::install_signal_intr (int signo, funcType func, ClientData clientData)
{
    return (install_signal (signo, func, clientData));
}

/*
 * Java signal handler
 */

int javaSignalHandler (int signo, void *arg, void *siginfo)
{
#ifdef FIX_LATER
    /*
     * Note that we are relying on the fact that the Java signal-handling
     * thread runs at high priority and will never be interrupted by a
     * "user" thread that is changing the handler.  In other words,
     * s_handlerList->m_inUseIndex will not change in between the line
     * that gets the func, and the line that gets the clientData.
     */
#endif

	// make sure we have a valid signo

    if ((signo < 0) || (signo >= N_INTERRUPTS)) {
	abort_with_message ("invalid signo");
    }

	// get the handler and client data

    sig_atomic_t useIndex = s_handlerList->m_inUseIndex;
    SignalHandlers::funcType func =
	s_handlerList->m_handlers[useIndex][signo].m_func;
    SignalHandlers::ClientData clientData =
	s_handlerList->m_handlers[useIndex][signo].m_clientData;

	// call the handler if specified (rc = 1 tells Java the we handled
	// the signal)
   
    int rc = 0; 

    if (func != NULL) {
	(func) (signo, clientData);
        rc = 1;
    }

    return (rc);
}
