/* Agent Tcl
   Bob Gray
   29 September 1995

   platSigioImp.cc

   This file implements the class which handles the SIGIO interrupt.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platPerThread.h"
#include "platThread.h"
#include "platThreadImp.h"
#include "platSigio.h"
#include "platSigioImp.h"
#include "truefalse.h"

#ifndef OLD_APPROACH

class SigioBody: public ThreadBody
{
    SigioHandlerImp *m_sigioHandlerImp;

public:

    SigioBody (SigioHandlerImp *sigioHandlerImp):
	m_sigioHandlerImp (sigioHandlerImp)
    {
	// empty
    }

   ~SigioBody () {
	delete (m_sigioHandlerImp);
    }

    void run (void) {

	int maxFd;
	int flags = 0;
	fd_mask readSet[MASK_SIZE];
	fd_mask writeSet[MASK_SIZE];
	fd_mask exceptSet[MASK_SIZE];
	fd_set *writeSetPtr = (fd_set *) NULL;
	fd_set *exceptSetPtr = (fd_set *) NULL;
	struct timeval nullTimeout = {0, 0};

	    /* get the max fd */

	if (m_sigioHandlerImp->fd > m_sigioHandlerImp->sigioFd) {
	    maxFd = m_sigioHandlerImp->fd + 1;
	} else {
	    maxFd = m_sigioHandlerImp->sigioFd + 1;
	}

	    /* loop forever */

        while (!(m_sigioHandlerImp->deleted)) {

	        /* do the "select" call */

	    bool repeatSelect = true;

	    while (repeatSelect) {

	        flags = 0;
		writeSetPtr = (fd_set *) NULL;
		exceptSetPtr = (fd_set *) NULL;

		FDMASK_ZERO (readSet);
	        FDMASK_SET (m_sigioHandlerImp->sigioFd, readSet);

                if (m_sigioHandlerImp -> flags & SigioTrap::e_READ_READY) {
	            FDMASK_SET (m_sigioHandlerImp -> fd, readSet);
	        }

                if (m_sigioHandlerImp -> flags & SigioTrap::e_WRITE_READY) {
	            FDMASK_ZERO (writeSet);
	            FDMASK_SET (m_sigioHandlerImp -> fd, writeSet);
		    writeSetPtr = (fd_set *) &writeSet[0];
	        }

                if (m_sigioHandlerImp -> flags & SigioTrap::e_EXCEPT_READY) {
	            FDMASK_SET (m_sigioHandlerImp -> fd, exceptSet);
	            FDMASK_ZERO (exceptSet);
		    exceptSetPtr = (fd_set *) &exceptSet[0];
	        }

	        if (select (maxFd, (fd_set *) &readSet[0], writeSetPtr, exceptSetPtr, NULL) <= 0) {

		    repeatSelect = false;

		} else if (FDMASK_ISSET (m_sigioHandlerImp->sigioFd, readSet)) {

		    unsigned char c;
		    read (m_sigioHandlerImp->sigioFd, &c, 1);

		} else {

		    repeatSelect = false; 
		    
	            if (FDMASK_ISSET (m_sigioHandlerImp->fd, readSet)) {
	 	        flags |= SigioTrap::e_READ_READY;
	            }

	            if ((writeSetPtr != (fd_set *) NULL) &&
			FDMASK_ISSET (m_sigioHandlerImp -> fd, writeSet)) {
			flags |= SigioTrap::e_WRITE_READY;
	            }

	            if ((exceptSetPtr != (fd_set *) NULL) &&
			    FDMASK_ISSET (m_sigioHandlerImp -> fd, exceptSet)) {
			    flags |= SigioTrap::e_EXCEPT_READY;
		    }
		}
	    }    /* end of repeatSelect */


	if ((flags != 0) && (!(m_sigioHandlerImp->deleted))) {

	        /* make sure that we have not been deleted */

	    m_sigioHandlerImp -> monitors.acquireAll ();

	    if (!(m_sigioHandlerImp -> deleted)) {

	        /* then redo the select to make sure that the flags are still valid */

	        flags = 0;

	        if (select (maxFd, (fd_set *) &readSet[0], writeSetPtr, exceptSetPtr, &nullTimeout) > 0) {

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, readSet)) {
	 	        flags |= SigioTrap::e_READ_READY;
	            }

	            if ((writeSetPtr != (fd_set *) NULL) &&
			FDMASK_ISSET (m_sigioHandlerImp -> fd, writeSet)) {
		        flags |= SigioTrap::e_WRITE_READY;
	            }

	            if ((exceptSetPtr != (fd_set *) NULL) &&
	               FDMASK_ISSET (m_sigioHandlerImp -> fd, exceptSet)) {
	 	       flags |= SigioTrap::e_EXCEPT_READY;
	            }
	        

	            /* then invoke the callback */

	        if (flags != 0) {
	            m_sigioHandlerImp -> callback = e_TRUE;
	            (m_sigioHandlerImp -> trap) -> sigioAlarm (m_sigioHandlerImp -> fd, flags);
	            m_sigioHandlerImp -> callback = e_FALSE;
	        }
		}
	    }
	    }

	    m_sigioHandlerImp -> monitors.releaseAll ();
        }

	    /* acquire and release the lock since we don't want to */
	    /* actually return if a caller is still inside         */
	    /* the SigioHandler destructor                         */

	m_sigioHandlerImp->lock.acquire();
	m_sigioHandlerImp->lock.release();
    }
};

SigioHandlerImp::SigioHandlerImp (int p_fd, int p_flags, SigioTrap *p_trap, const MonitorGroup& p_monitors):
    fd (p_fd),
    flags (p_flags),
    trap (p_trap),
    lock (),
    monitors (p_monitors),
    deleted (e_FALSE),
    callback (e_FALSE)
{
    int fds[2];
    monitors.push (lock); 
    
    if (pipe (fds) < 0) {
	abort();
    } else {
	sigioFd = fds[0];
	controlFd = fds[1];
    }
}

SigioHandlerImp::SigioHandlerImp (const SigioHandlerImp& sigioHandlerImp):
    fd (sigioHandlerImp.fd),
    flags (sigioHandlerImp.flags),
    trap (sigioHandlerImp.trap),
    lock (sigioHandlerImp.lock),
    monitors (sigioHandlerImp.monitors),
    deleted (e_FALSE),
    callback (e_FALSE),
    sigioFd (sigioHandlerImp.sigioFd),
    controlFd (sigioHandlerImp.controlFd)
{
    /* empty */
}

void SigioHandlerImp::createThread (void)
{
    char buffer[24];

	// construct the thread name
	
    sprintf (buffer, "sigio-handler-%d", fd);

	// create and start the thread

    SigioBody *sigioBody = new SigioBody (this);
    Thread thread (buffer, Thread::e_DAEMON_THREAD, sigioBody);
    thread.resume ();
}

SigioHandlerImp::~SigioHandlerImp ()
{
    close (sigioFd);
    close (controlFd);
}

/* SigioHandler::SigioHandler

   Purpose: This procedure is the constructor for class SigioHandler.

     Input: trap  = function to be called when SIGIO fires
		    (SigioTrap *)

	    fd    = file descritor of interest
		    (int)

	    flags = events of interest
		    (OR'ed combination of SigioTrap::StatusFlags)
*/

SigioHandler::SigioHandler (int fd, SigioTrap *trap, int flags, const MonitorGroup &group)
{
	/* assertions on the parameters */

    assert (trap != NULL);

	/* allocate an m_this and then create the thread */

    m_this = new SigioHandlerImp (fd, flags, trap, group);
    m_this -> createThread();
}

/* SigioHandler::changeFlags

   Purpose: Change the I/O events for which we want a callback

     Input: flags = I/O events for which we want a callback 
		    (OR'ed combination of SigioTrap::e_READY_READY, etc.)  
*/

void SigioHandler::changeFlags (int flags)
{
	/*
	 * lock for mutual exclusion
	 */

    m_this -> lock.acquire ();

	/*
	 * if we are inside a callback, just change the flags since we
	 * will get the correct setting once we loop back around -- otherwise
	 * set the flags and write to the control fd so that we wake up the
	 * select call
	 */

    if (m_this->callback) {

	m_this->flags = flags;

    } else {

	m_this->flags = flags;
	unsigned char c = 0;
	write (m_this->controlFd, &c, 1);
    }

	/*
	 * release the lock
	 */

    m_this -> lock.release ();
}

/* SigioHandler::~SigioHandler

   Purpose: This procedure is the destructor for class SigioHandler.
*/

SigioHandler::~SigioHandler (void)
{
    Guard guard (m_this -> lock);
    m_this -> deleted = e_TRUE;
    unsigned char c = 0;
    write (m_this->controlFd, &c, 1);
}

void SigioHandler::checkAndCallTraps (void)
{
    abort_with_message ("SigioHandler::checkAndCallTraps should never be called in a multi-threaded environment");
}

void SigioHandler::setGeneralTrap (SigioTrap */*trap*/)
{
    abort_with_message ("SigioHandler::setGeneralTrap should never be called in a multi-threaded environment");
}



#else

class SigioBody: public ThreadBody
{
    SigioHandlerImp *m_sigioHandlerImp;

public:

    SigioBody (SigioHandlerImp *sigioHandlerImp):
	m_sigioHandlerImp (sigioHandlerImp)
    {
	// empty
    }

   ~SigioBody () {
	delete (m_sigioHandlerImp);
    }

    void run (void)
    {
	int maxFd;
	int flags = 0;
	fd_mask readSet[MASK_SIZE];
	fd_mask writeSet[MASK_SIZE];
	fd_mask exceptSet[MASK_SIZE];
	struct timeval nullTimeout = {0, 0};

	    /* get the max fd */

        maxFd = m_sigioHandlerImp -> fd + 1;

	    /* loop forever */

        while (1) {

	        /* do the "select" call */

	    if (m_sigioHandlerImp -> flags != 0) {

	        FDMASK_ZERO (readSet);
	        FDMASK_ZERO (writeSet);
	        FDMASK_ZERO (exceptSet);

                if (m_sigioHandlerImp -> flags & SigioTrap::e_READ_READY) {
	             FDMASK_SET (m_sigioHandlerImp -> fd, readSet);
	        }

                if (m_sigioHandlerImp -> flags & SigioTrap::e_WRITE_READY) {
	            FDMASK_SET (m_sigioHandlerImp -> fd, writeSet);
	        }

                if (m_sigioHandlerImp -> flags & SigioTrap::e_EXCEPT_READY) {
	            FDMASK_SET (m_sigioHandlerImp -> fd, exceptSet);
	        }

	        if (select (maxFd, (SELECT_MASK *) &readSet[0], (SELECT_MASK *) &writeSet[0], (SELECT_MASK *) &exceptSet[0], NULL) > 0) {

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, readSet)) {
	 	        flags |= SigioTrap::e_READ_READY;
	            }

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, writeSet)) {
		        flags |= SigioTrap::e_WRITE_READY;
	            }

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, exceptSet)) {
		        flags |= SigioTrap::e_EXCEPT_READY;
	            }
	        }
	    }

	        /* make sure that we have not been deleted */

	    m_sigioHandlerImp -> monitors.acquireAll ();

	    if (m_sigioHandlerImp -> deleted) {
	        m_sigioHandlerImp -> monitors.releaseAll ();
		return;
	    }

	        /* then redo the select to make sure that the flags are still valid */

	    if (flags != 0) {

	        flags = 0;

	        if (select (maxFd, (SELECT_MASK *) &readSet[0], (SELECT_MASK *) &writeSet[0], (SELECT_MASK *) &exceptSet[0], &nullTimeout) > 0) {

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, readSet)) {
	 	        flags |= SigioTrap::e_READ_READY;
	            }

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, writeSet)) {
		        flags |= SigioTrap::e_WRITE_READY;
	            } 

	            if (FDMASK_ISSET (m_sigioHandlerImp -> fd, exceptSet)) {
	 	       flags |= SigioTrap::e_EXCEPT_READY;
	            }
	        }

	            /* then invoke the callback */

	        if (flags != 0) {
	            m_sigioHandlerImp -> callback = e_TRUE;
	            (m_sigioHandlerImp -> trap) -> sigioAlarm (m_sigioHandlerImp -> fd, flags);
	            m_sigioHandlerImp -> callback = e_FALSE;
	        }
	    }

	    m_sigioHandlerImp -> monitors.releaseAll ();
        }

        abort_with_message ("should never get here");   // should never get here
    }
};

SigioHandlerImp::SigioHandlerImp (int p_fd, int p_flags, SigioTrap *p_trap, const MonitorGroup& p_monitors):
    fd (p_fd),
    flags (p_flags),
    trap (p_trap),
    lock (),
    monitors (p_monitors),
    deleted (e_FALSE),
    callback (e_FALSE)
{
    monitors.push (lock); 
}

SigioHandlerImp::SigioHandlerImp (const SigioHandlerImp& sigioHandlerImp):
    fd (sigioHandlerImp.fd),
    flags (sigioHandlerImp.flags),
    trap (sigioHandlerImp.trap),
    lock (sigioHandlerImp.lock),
    monitors (sigioHandlerImp.monitors),
    deleted (e_FALSE),
    callback (e_FALSE)
{
    // empty
}

void SigioHandlerImp::createThread (void)
{
    char buffer[24];

	// construct the thread name
	
    sprintf (buffer, "sigio-handler-%d", fd);

	// create and start the thread

    SigioBody *sigioBody = new SigioBody (this);
    Thread thread (buffer, Thread::e_DAEMON_THREAD, sigioBody);
    thread.resume ();
}

SigioHandlerImp::~SigioHandlerImp ()
{
    // empty
}

/* SigioHandler::SigioHandler

   Purpose: This procedure is the constructor for class SigioHandler.

     Input: trap  = function to be called when SIGIO fires
		    (SigioTrap *)

	    fd    = file descritor of interest
		    (int)

	    flags = events of interest
		    (OR'ed combination of SigioTrap::StatusFlags)
*/

SigioHandler::SigioHandler (int fd, SigioTrap *trap, int flags, const MonitorGroup &group)
{
	/* assertions on the parameters */

    assert (trap != NULL);

	/* allocate an m_this and then create the thread */ 

    m_this = new SigioHandlerImp (fd, flags, trap, group);
    m_this -> createThread();
}

/* SigioHandler::changeFlags

   Purpose: Change the I/O events for which we want a callback

     Input: flags = I/O events for which we want a callback 
		    (OR'ed combination of SigioTrap::e_READY_READY, etc.)  
*/

void SigioHandler::changeFlags (int flags)
{
	/*
	 * lock for mutual exclusion
	 */

    m_this -> lock.acquire ();

	/*
	 * appropriate action depends on whether the new flags require a 
	 * watcher thread and whether a watcher thread currently exists
	 */

    if (m_this -> flags == 0) {

	if (flags == 0) {

		/*
		 * do nothing if we do not have a watcher thread and do not
		 * need a watcher thread
		 */

	    m_this -> lock.release ();

	} else {

		/*
		 * create the watcher thread
		 */

	    m_this -> flags = flags;
	    m_this -> lock.release ();
	    m_this -> createThread ();
	}

    } else {

	if (flags == 0) {

		/*
		 * if we have a watcher thread but do not need one anymore,
		 * set up a new SigioHandlerImp instance to remember the
		 * trap information and then mark the old SigioHandlerImp 
		 * instance as deleted
		 */

	    SigioHandlerImp *new_this = new SigioHandlerImp (*m_this);
	    new_this -> flags = 0;
	    m_this -> flags = 0;
	    m_this -> deleted = e_TRUE;
	    m_this -> lock.release ();
	    m_this = new_this;

	} else {

		/* just reset the flags if the watcher thread is at the   */
		/* right point; otherwise create a new watcher thread and */
		/* delete the old one                                     */

	    if (m_this -> callback) {

		m_this -> flags = flags;
		m_this -> lock.release ();

	    } else {

	        SigioHandlerImp *new_this = new SigioHandlerImp (*m_this);
	        new_this -> flags = flags;
		m_this -> flags = 0;
	        m_this -> deleted = e_TRUE;
	        m_this -> lock.release ();
	        m_this = new_this;
		m_this -> createThread ();
	    }
	}
    }
}

/* SigioHandler::~SigioHandler

   Purpose: This procedure is the destructor for class SigioHandler.
*/

SigioHandler::~SigioHandler (void)
{
    Guard guard (m_this -> lock);
    m_this -> deleted = e_TRUE;
}

void SigioHandler::checkAndCallTraps (void)
{
    abort_with_message ("SigioHandler::checkAndCallTraps should never be called in a multi-threaded environment");
}

void SigioHandler::setGeneralTrap (SigioTrap */*trap*/)
{
    abort_with_message ("SigioHandler::setGeneralTrap should never be called in a multi-threaded environment");
}

#endif
