/* Agent Tcl
   Bob Gray
   29 September 1995

   platThreadImp.cc

   This file implements the class which handles the creation and execution 
   of threads.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platPerThread.h"
#include "platThread.h"
#include "platThreadImp.h"
#include "truefalse.h"
extern "C" {
#include "threads.h"		// TID
}

    // forward declarations

static void java_thread_startup (TID tid);

    // statics -- 
    //
    // s_idToThreadMap associates a Java TID with a Thread instance

static PerThreadPointer *s_idToThreadMap 
	= NULL;

unsigned int ThreadImp::referenceKey 
	= 0;

    // methods

ThreadImp::ThreadImp (ThreadBody *p_threadBody)
{
	/*
	 * Note that the address of the static variable referenceKey never 
	 * changes (of course), so it is okay to just go ahead and redo the
	 * assignment for every thread.
	 */

    referenceKey = (unsigned int) &referenceKey;
    referenceCount = 1;
    attachedThread = NULL;
    threadBody     = p_threadBody;
}

ThreadImp::~ThreadImp ()
{
    delete (threadBody);
}

class MapInitializer: public OnceTrap
{
public:
    void initialize (void) {
	s_idToThreadMap = new PerThreadPointer();
    }
};

Thread::Thread (const char *name, Thread::ThreadType threadType, ThreadBody *threadBody) 
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

	// allocate the id-to-thread map if necessary 

    if (onceControl != MonitorOnce::e_DONE) {
	MapInitializer mapInitializer;
	MonitorOnce::doOnce (onceControl, mapInitializer);	
    }

	// allocate the implementation-specific part of the structure

    m_this = new ThreadImp (threadBody);

	// find the thread class

    ClassClass *cb = FindClass (NULL, "java/lang/Thread", TRUE);
    assert (cb != NULL);

	// create the Java thread instance
	
    TID tid = (TID) execute_java_constructor 
	    (NULL, NULL, cb, "(Ljava/lang/String;)", 
		makeJavaString ((char *) name, strlen((char *) name)));
    assert (tid != NULL);

	// remember the thread id

    m_id = tidToThreadId(tid);
    m_this -> javaId = tid;

	// associate the thread id with this C++ instance

    s_idToThreadMap -> setValue (m_id, (void *) m_this);

	// turn the Java thread instance into a daemon thread 

    if (threadType == e_DAEMON_THREAD) {

        execute_java_dynamic_method 
	    (EE(), (Hjava_lang_Object *) tid,
	        "setDaemon", "(Z)V", (long) 1);

	if (exceptionOccurred(EE())) {
	    abort_with_message ("Java method \"setDaemon\" failed in Thread::Thread!");
	}
    }

	// create the actual thread

    if (threadCreate (tid, THR_SYSTEM, ProcStackSize, (void * (*)()) java_thread_startup) != 0) {
	abort_with_message ("Thread::Thread was unable to create the thread");
    }

	// set the thread priority to MaximumPriority if we are a daemon and
	// to normal priority if we are not

    if (threadType == e_DAEMON_THREAD) {
	threadSetPriority (tid, MaximumPriority);
    } else {
	threadSetPriority (tid, NormalPriority);
    }
}

Thread::Thread (const Thread& thread)
{
    monitorEnter (ThreadImp::referenceKey);
    m_this	              = thread.m_this;
    m_id                      = thread.m_id;
    m_this -> referenceCount += 1;
    monitorExit (ThreadImp::referenceKey);
}

void Thread::resume (void) 
{
    m_this -> attachedThread = new Thread (*this);
    threadResume (m_this -> javaId);
}

Thread::~Thread ()
{
    monitorEnter (ThreadImp::referenceKey);

    if (--m_this->referenceCount == 0) {
	delete (m_this);
	m_this = NULL;
    }

    monitorExit (ThreadImp::referenceKey);
}

/*
 * STATIC METHODS
 */

void Thread::exit (void)
{
	// delete the dynamically-allocated Thread instance that
	// exists only as long as the thread does

    ThreadId threadId = Thread::self();
    ThreadImp *threadImp = (ThreadImp *) s_idToThreadMap -> getValue (threadId);
    assert (threadImp != NULL);
    Thread *attachedThread = threadImp -> attachedThread;
    threadImp -> attachedThread = NULL;
    delete (attachedThread);

	// then exit

    threadExit();
}

ThreadId Thread::self (void)
{
    return (tidToThreadId (threadSelf()));
}

/*
 * JAVA THREAD STARTUP FUNCTION
 */

static void java_thread_startup (TID tid)
{
    ExecEnv ee;
 
	// finish thread initialization and get the ThreadImp instance,

    InitializeExecEnv (&ee, (JHandle *) tid);
    threadInit (tid, (stackp_t) &tid);
    ThreadImp *m_this = (ThreadImp *) s_idToThreadMap -> 
	peekAtValue (tidToThreadId(tid));
    assert (m_this != NULL);

	// run the thread

    assert (m_this -> threadBody != NULL);
    (m_this -> threadBody) -> run();

	// thread has finished!

    Thread::exit();
    abort_with_message ("should never get here!");
}
