/* Bob Gray
   Agent Tcl
   13 July 1996

   platExclusionImp.cc

   This file implements the classes that provide simple mutual exclusion.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platExclusionImp.h"
#include "platSigStack.h"

#ifdef FIX_LATER

/*
 * Note that our counter is semantically correct only if all signal
 * blocking is being done through this class or, more precisely, only 
 * if no signal blocking is done by other means in between NESTED
 * acquistions of a MonitorLock.  In the current system this is true.
 * However, we need to write and expose a separate signal blocking
 * class so that Tcl extensions or a more complex server can block
 * arbitrary signals while still going through some central function
 * that uses a similar counter to avoid unnecessary calls to sigprocmask 
 * (it makes a huge speed difference).
 */

#endif

   // signal masks

sigset_t MonitorLockImp::oldMask;
sigset_t MonitorLockImp::zeroMask;
sigset_t MonitorLockImp::blockMask;

   // counter of nested MonitorLock::acquires

sig_atomic_t MonitorLockImp::count = 0;  

static void fillInBlockMask (sigset_t &blockMask)  
{
    sigemptyset (&blockMask);
    sigaddset (&blockMask, SIGIO);
    sigaddset (&blockMask, SIGALRM);
#ifndef ALLOW_PROFILING
    sigaddset (&blockMask, SIGPROF);
#endif
    sigaddset (&blockMask, SIGCHLD);
    sigaddset (&blockMask, SIGUSR1);
    sigaddset (&blockMask, SIGUSR2);
    sigaddset (&blockMask, SIGHUP);
}

static void blockSignals (void)
{
	/* only block the signals if they are not already blocked) */

#ifdef FIX_LATER
	/* There is a race condition here, but only in the rare (i.e., never  */
	/* occurs in the current system) case where a *signal handler* calls  */
	/* MonitorLock::acquire without later calling MonitorLock::release    */
#endif
	
    if (MonitorLockImp::count == 0) {
	sigprocmask 
	    (SIG_BLOCK, &MonitorLockImp::blockMask, &MonitorLockImp::oldMask);
    }

    MonitorLockImp::count += 1;
}

static void unblockSignals (void)
{
	/* revert to the old signal mask if the current lock acquisition */
	/* was not nested inside another                                 */

    assert (MonitorLockImp::count > 0);

    if (--MonitorLockImp::count == 0) {
	sigprocmask (SIG_SETMASK, &MonitorLockImp::oldMask, NULL);
    }
}

BOOLEAN MonitorLock::getConditionVariable (UINT_32 & /*handle*/) 
{
    abort_with_message ("MonitorLock::getConditionVariable has not been implemented!");
    return (e_FALSE);
}

BOOLEAN MonitorLock::acquire (void) 
{
    blockSignals();
    return (e_TRUE);
}
    
BOOLEAN MonitorLock::try_acquire (void)
{
    acquire ();			// acquire always succeeds
    return (e_TRUE);
}

BOOLEAN MonitorLock::release (void) 
{
    unblockSignals();
    return (e_TRUE);
}

BOOLEAN MonitorLock::wait (void) 
{
    sigsuspend (&MonitorLockImp::zeroMask);
    return (e_TRUE);
}

BOOLEAN MonitorLock::timed_wait (struct timeval)
{
    abort_with_message ("MonitorLock::timed_wait has not been implemented!");
    return (e_FALSE);
}

BOOLEAN MonitorLock::wakeup (void)
{
    return (e_TRUE);
}

BOOLEAN MonitorLock::wait (UINT_32 /*conditionHandle*/, enum BOOLEAN /* debug*/)
{
    abort_with_message ("MonitorLock::wait (on condition handle) has not been implemented!");
    return (e_FALSE);
}

BOOLEAN MonitorLock::timed_wait (UINT_32 /*conditionHandle*/, struct timeval /*delay*/)
{
    abort_with_message ("MonitorLock::timed_wait (on condition handle) has not been implemented!");
    return (e_FALSE);
}

BOOLEAN MonitorLock::wakeup (UINT_32 /*conditionHandle*/, enum BOOLEAN /* debug */)
{
    abort_with_message ("MonitorLock::wakeup (on condition handle) has not been implemented!");
    return (e_FALSE);
}

class MaskInitializer: public OnceTrap
{
    sigset_t &m_blockMask;
    sigset_t &m_zeroMask;

public:

    MaskInitializer (sigset_t &blockMask, sigset_t &zeroMask):
	m_blockMask (blockMask),
	m_zeroMask  (zeroMask)
   {
	// empty
   }

   void initialize (void) {
	fillInBlockMask (m_blockMask);
	sigemptyset (&m_zeroMask);
   }
};

void initializeSignalMasks (void)
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

	// initialize the signal masks

    if (onceControl != MonitorOnce::e_DONE) {
	MaskInitializer initializer
	    (MonitorLockImp::blockMask, MonitorLockImp::zeroMask);
	MonitorOnce::doOnce (onceControl, initializer);
    }
}

MonitorLock::MonitorLock (void):
    m_this (NULL)			// no instance-specific data
{
    initializeSignalMasks();
}

MonitorLock::MonitorLock (const MonitorLock& /*lock*/):
    m_this (NULL)
{
    // empty
} 

MonitorLock& MonitorLock::operator= (const MonitorLock& /*lock*/)
{
    return (*this);
}

MonitorLock::~MonitorLock ()
{
    // empty
}

void MonitorOnce::doOnce (OnceControl &onceControl, OnceTrap &trap)
{
    sigset_t oldMask;
    sigset_t blockMask;

	// in a single-threaded environment, we can ensure exclusive access
	// just by blocking signals

	// first test without blocking for speed

    if (onceControl == e_DONE) { 
	return;
    }

	// otherwise block the signals, do the initialize, and unblock the signals 

    fillInBlockMask (blockMask);
    sigprocmask (SIG_BLOCK, &blockMask, &oldMask);

    if (onceControl != e_DONE) {
	trap.initialize ();
	onceControl = e_DONE;
    }

    sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

BOOLEAN MutexLock::acquire (void) 
{
    blockSignals();
    return (e_TRUE);
}
    
BOOLEAN MutexLock::try_acquire (void)
{
    acquire ();			// acquire always succeeds
    return (e_TRUE);
}

BOOLEAN MutexLock::release (void) 
{
    unblockSignals();
    return (e_TRUE);
}

MutexLock::MutexLock (void):
    m_this (NULL)		// no instance-specific data
{
    initializeSignalMasks();
}

MutexLock::MutexLock (const MutexLock& /*lock*/):
    m_this (NULL)		// no instance-specific data
{
    // empty
}

MutexLock& MutexLock::operator= (const MutexLock& /*lock*/)
{
    return (*this);
}

MutexLock::~MutexLock ()
{
    // empty
}
