/* Agent Tcl
   Bob Gray
   29 September 1995

   platSigioImp.cc

   This file implements the class which handles the SIGIO interrupt.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platSigio.h"
#include "platSigioImp.h"
#include "platSizes.h"
#include "platSocket.h"
#include "truefalse.h"

    /* static data */

int SigioHandlerImp::count 
	= 0;

SigioHandlerImp *SigioHandlerImp::head  
	= NULL;

MonitorLock *SigioHandlerImp::lock  
	= NULL;

SigioTrap *SigioHandlerImp::sigioTrap
	= NULL;

fd_mask SigioHandlerImp::rset[MASK_SIZE];
fd_mask SigioHandlerImp::wset[MASK_SIZE];
fd_mask SigioHandlerImp::eset[MASK_SIZE];

int SigioHandlerImp::maxSockfd 
	= 0;
int SigioHandlerImp::rcount    
	= 0;
int SigioHandlerImp::wcount    
	= 0;
int SigioHandlerImp::ecount    
	= 0;

BOOLEAN SigioHandlerImp::callback 
	= e_FALSE;

/* gen_sigio_handler

   Purpose: This procedure is called when SIGIO fires.

     Input: signo      = SIGIO
		         (int)

	    clientData = client data
			 (unused)

    Output: The procedure calls the generic SIGIO trap.
*/

void gen_sigio_handler (int, SignalHandlers::ClientData)
{
	/* save errno, make the callback and restore errno */

    int oldErrno = errno;

    SigioHandler::markSigioFired ();

    if (SigioHandlerImp::sigioTrap != NULL) {
	SigioHandlerImp::sigioTrap -> sigioAlarm (0, 0);
    }

    errno = oldErrno;
}

/* SigioHandler::setGeneralTrap

   Purpose: Set the trap that is called on any SIGIO
  
     Input: trap = the trap
		   (SigioTrap *)
*/

void SigioHandler::setGeneralTrap (SigioTrap *trap) 
{
    assert (trap != NULL);
    SigioHandlerImp::sigioTrap = trap;
}

/* SigioHandler::checkAndCallTraps

   Purpose: Check all the file descriptors and call the traps for any ready
            descriptors
*/

void SigioHandler::checkAndCallTraps (void)
{
    int rc;
    UINT_32 i;
    int outFlags;
    struct timeval tv;
    fd_mask *readPtr;
    fd_mask *writePtr;
    fd_mask *exceptPtr;
    SigioHandlerImp *next;
    SigioHandlerImp *handle;
    fd_mask readSet[MASK_SIZE];		/* read set for "select"      */
    fd_mask writeSet[MASK_SIZE];        /* write set for "select"     */
    fd_mask exceptSet[MASK_SIZE];       /* exception set for "select" */

	/* mutual exclusion -- automatically released */

    Guard guard (*SigioHandlerImp::lock);

	/* loop until there is no pending I/O */

    while (1) {

	    /* initialize the masks */

	if (SigioHandlerImp::rcount > 0) {

	    for (i = 0; i < MASK_SIZE; ++i) {
		readSet[i] = SigioHandlerImp::rset[i];
	    }

	    readPtr = readSet;

        } else {

	    readPtr = NULL;
        } 

        if (SigioHandlerImp::wcount > 0) {

	    for (i = 0; i < MASK_SIZE; ++i) {
	        writeSet[i] = SigioHandlerImp::wset[i];
	    }

	    writePtr = writeSet;

        } else {

	    writePtr = NULL;
        }

        if (SigioHandlerImp::ecount > 0) {

	    for (i = 0; i < MASK_SIZE; ++i) {
	        exceptSet[i] = SigioHandlerImp::eset[i];
	    }

	    exceptPtr = exceptSet; 

        } else {

	    exceptPtr = NULL;
        }

    	    /* do the selection */
    
        tv.tv_sec  = 0;
        tv.tv_usec = 0;
    
        if ((rc = select (SigioHandlerImp::maxSockfd + 1, (SELECT_MASK *) readPtr, (SELECT_MASK *) writePtr, (SELECT_MASK *) exceptPtr, &tv)) < 0) {
	    char temp [128];
	    sprintf (temp, "select failed in SigioHandler::checkAndCallTraps (errno = %d)\n", errno);
	    abort_with_message (temp);
        }

	if (rc == 0) {
	    break;
	}
    
	    /* check for ready files */
    
        for (handle = SigioHandlerImp::head; handle != NULL; handle = next) {
    
	    next = handle -> next;
	    outFlags = 0;
    
	    if ((readPtr != NULL) && 
		((readSet[handle->fdindex]) & (handle->fdbit))) {
	        outFlags |= SigioTrap::e_READ_READY;
    	    }
    
    	    if ((writePtr != NULL) && 
		((writeSet[handle->fdindex]) & (handle->fdbit))) {
    	        outFlags |= SigioTrap::e_WRITE_READY;
    	    }
    
    	    if ((exceptPtr != NULL) && 
		((exceptSet[handle->fdindex]) & (handle->fdbit))) {
    	        outFlags |= SigioTrap::e_EXCEPT_READY;
    	    }
    
    	    if (outFlags != 0) {
	        SigioHandlerImp::callback = e_TRUE;
    	        handle -> trap -> sigioAlarm (handle->fd, outFlags);
	        SigioHandlerImp::callback = e_FALSE;
	    }
	}
    }
}
   
/* SigioHandler::SigioHandler

   Purpose: This procedure is the constructor for class SigioHandler.
*/

SigioHandler::SigioHandler (int fd, SigioTrap *trap, int flags, const MonitorGroup& /*UNUSED*/)
{
	/* assertions on the parameters */

    assert (trap != NULL);

	/* allocate the lock if necessary */

    if (SigioHandlerImp::lock == NULL) {
	SignalHandlers::install_signal_intr (SIGIO, gen_sigio_handler, 0);
	SigioHandlerImp::lock = new MonitorLock ();
	FDMASK_ZERO (SigioHandlerImp::rset);
	FDMASK_ZERO (SigioHandlerImp::wset);
	FDMASK_ZERO (SigioHandlerImp::eset);
	SigioHandlerImp::maxSockfd = 0;
    }

	/* allocate an m_this */

    m_this = new SigioHandlerImp ();

	/* remember the file descriptor and trap */

    m_this -> fd    = fd;
    m_this -> trap  = trap;
    m_this -> flags = flags;

	/* guard all further access */

    Guard guard (*(m_this -> lock));  // automatically released

	/* turn on asynchronous I/O for the file descriptor */

    (void) SocketIO::markAsynchronous (fd);

	/* construct the fd_set for the "select" call */

    int fdindex     = FDMASK_INDEX (fd);
    int fdbit       = FDMASK_BIT   (fd);
    m_this->fdindex = fdindex;
    m_this->fdbit   = fdbit;

    if (flags & SigioTrap::e_READ_READY) {
	SigioHandlerImp::rset[fdindex] |= fdbit;
	SigioHandlerImp::rcount += 1;
    }

    if (flags & SigioTrap::e_WRITE_READY) {
	SigioHandlerImp::wset[fdindex] |= fdbit;
	SigioHandlerImp::wcount += 1;
    }

    if (flags & SigioTrap::e_EXCEPT_READY) {
	SigioHandlerImp::eset[fdindex] |= fdbit;
	SigioHandlerImp::ecount += 1;
    }

    if (fd > SigioHandlerImp::maxSockfd) {
	SigioHandlerImp::maxSockfd = fd; 
    }

	/* is this the first handler? */

    if (m_this -> count == 0) {

	m_this -> head     = m_this;
	m_this -> next     = NULL;
	m_this -> previous = NULL;
	m_this -> count    = 1;

    } else {

	m_this -> head -> previous = m_this;
	m_this -> next	     	   = m_this -> head;
	m_this -> previous	   = NULL;
	m_this -> head	           = m_this;
	m_this -> count		  += 1;
    }

	// send a SIGIO to myself

    if (!SigioHandlerImp::callback) {

	SigioHandler::markSigioFired ();

	if (SigioHandlerImp::sigioTrap != NULL) {
	    SigioHandlerImp::sigioTrap -> sigioAlarm (0, 0);
	}
    }
}

/* SigioHandler::changeFlags

   Purpose: Change the I/O events for which we want a callback

     Input: flags = I/O events for which we want a callback 
		    (OR'ed combination of SigioTrap::e_READY_READY, etc.)  
*/

void SigioHandler::changeFlags (int flags) 
{
    BOOLEAN subset;
    int unionFlags;

	// lock for mutual exclusion

    Guard guard (*(m_this -> lock));  // automatically released 

	// see if the new set of I/O events is the same as the old one

    if (flags == m_this -> flags) {
	return;
    }

	// see if the new set of I/O events is a subset of the new one

    unionFlags = flags | m_this -> flags;

    if (unionFlags ^ m_this -> flags == 0) {
	subset = e_TRUE;
    } else {
	subset = e_FALSE;
    }

#ifdef FIX_LATER
	// We are doing extra work here -- we should turn the flag on or
	// off only if the setting in the new flag set is different than
	// the setting in the old flag set.
#endif

	// clear the old events

    int fdindex = m_this->fdindex;
    int fdbit   = m_this->fdbit;

    if (m_this->flags & SigioTrap::e_READ_READY) {
	SigioHandlerImp::rset[fdindex] &= ~fdbit;
	SigioHandlerImp::rcount -= 1;
    }

    if (m_this->flags & SigioTrap::e_WRITE_READY) {
	SigioHandlerImp::wset[fdindex] &= ~fdbit;
	SigioHandlerImp::wcount -= 1;
    }

    if (m_this->flags & SigioTrap::e_EXCEPT_READY) {
	SigioHandlerImp::eset[fdindex] &= ~fdbit;
	SigioHandlerImp::ecount -= 1;
    }

	// set the new events

    m_this->flags = flags;

    if (flags & SigioTrap::e_READ_READY) {
	SigioHandlerImp::rset[fdindex] |= fdbit;
	SigioHandlerImp::rcount += 1;
    }

    if (flags & SigioTrap::e_WRITE_READY) {
	SigioHandlerImp::wset[fdindex] |= fdbit;
	SigioHandlerImp::wcount += 1;
    }

    if (flags & SigioTrap::e_EXCEPT_READY) {
	SigioHandlerImp::eset[fdindex] |= fdbit;
	SigioHandlerImp::ecount += 1;
    }

	// send a SIGIO to myself if necessary

    if (!subset && !SigioHandlerImp::callback) {

	SigioHandler::markSigioFired ();

	if (SigioHandlerImp::sigioTrap != NULL) {
	    SigioHandlerImp::sigioTrap -> sigioAlarm (0, 0);
	}
    }
}

/* SigioHandler::~SigioHandler

   Purpose: This procedure is the destructor for class SigioHandler.
*/

SigioHandler::~SigioHandler (void)
{
	/* scope for the guard */
 
    {
	Guard guard (*(m_this -> lock));  // automatically released 

	    /* one less instance */

	m_this -> count -= 1;

	    /* splice this instance out of the list */

	if (m_this -> count == 0) {

	    m_this -> head = NULL;

	} else {

	    if (m_this -> next != NULL) {
		m_this -> next -> previous = m_this -> previous;
	    }

	    if (m_this -> previous != NULL) {
		m_this -> previous -> next = m_this -> next;
	    } else {
		m_this -> head = m_this -> next;
	    }
	} 

	    /* turn off asynchronous I/O for the file descriptor */

	(void) SocketIO::markSynchronous (m_this->fd);

	     /* adjust the fd mask */

	int fdindex = m_this->fdindex;
   	int fdbit   = m_this->fdbit; 

	if (m_this->flags & SigioTrap::e_READ_READY) {
	    SigioHandlerImp::rset[fdindex] &= ~fdbit;
	    SigioHandlerImp::rcount -= 1;
	}

	if (m_this -> flags & SigioTrap::e_WRITE_READY) {
	    SigioHandlerImp::wset[fdindex] &= ~fdbit;
	    SigioHandlerImp::wcount -= 1;
	}

	if (m_this -> flags & SigioTrap::e_EXCEPT_READY) {
	    SigioHandlerImp::eset[fdindex] &= ~fdbit;
	    SigioHandlerImp::ecount -= 1;
	}

#ifdef FIX_LATER
	    /* We can adjust the maximum fd much for efficiently with */
	    /* a simgle min-max heap, but the brute-force loop should */
	    /* be more than efficient enough for most cases.          */
#endif

	    /* adjust the maximum fd */

	SigioHandlerImp *next;
	SigioHandlerImp *handle;
	SigioHandlerImp::maxSockfd = 0;

        for (handle = SigioHandlerImp::head; handle != NULL; handle = next) {

	    if (handle->fd > SigioHandlerImp::maxSockfd) {
		SigioHandlerImp::maxSockfd = handle->fd;
	    }

	    next = handle->next;
	}
    }

    delete (m_this);
}
