/* Agent Tcl
   Bob Gray
   27 March 1997

   platSigtermImp.cc

   This file implements the class which handles the SIGTERM interrupt.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platInterrupt.h"
#include "platSigterm.h"
#include "platSigtermImp.h"

    /* static data */

int SigtermHandlerImp::count
	= 0;

int SigtermHandlerImp::exitcode
	= 1;

SigtermHandlerImp *SigtermHandlerImp::head
	= NULL;

    /* block and unblock SIGTERM */

static sigset_t blockSigterm (void) 
{
    sigset_t oldMask;
    sigset_t newMask;

    sigemptyset (&oldMask);
    sigemptyset (&newMask);
    sigaddset (&newMask, SIGTERM);
    sigprocmask (SIG_BLOCK, &newMask, &oldMask);

    return (oldMask);
}

static void unblockSigterm (sigset_t oldMask)
{
    sigprocmask (SIG_SETMASK, &oldMask, NULL);
}

/* gen_sigterm_handler

   Purpose: This procedure is called when SIGTERM fires.

     Input: signo      = SIGTERM
		         (int)

	    clientData = client data
			 (unused)

    Output: The procedure calls the SigtermTrap associated with each
	    instance of SigtermHandler.
*/

void gen_sigterm_handler (int, SignalHandlers::ClientData)
{
    SigtermHandlerImp *handle;
    SigtermHandlerImp *next = NULL;	// set to prevent compiler warning

	/* call the check functions */

    for (handle = SigtermHandlerImp::head; handle != NULL; handle = next) {
	next = handle -> next;
	handle -> trap -> sigtermAlarm ();
    }

	/* exit */
 
    exit (SigtermHandlerImp::exitcode);
}
   
/* SigtermHandler::SigtermHandler

   Purpose: This procedure is the constructor for class SigtermHandler.

     Input: function = function to be called when SIGIO fires
            data     = client data to be passed to the function
*/

SigtermHandler::SigtermHandler (SigtermTrap *trap)
{
	/* assertions on the parameters */

    assert (trap != NULL);

	/* allocate the lock if necessary */ 

    if (SigtermHandlerImp::count == 0) {
	SignalHandlers::install_signal_intr (SIGTERM, gen_sigterm_handler, 0);
    }

	/* allocate an m_this */

    m_this = new SigtermHandlerImp ();

	/* remember the trap */ 

    m_this -> trap  = trap;

	/* block SIGTERM */

    sigset_t oldMask = blockSigterm ();
 
	/* is this the first handler? */

    if (m_this -> count == 0) {

	m_this -> head     = m_this;
	m_this -> next     = NULL;
	m_this -> previous = NULL;
	m_this -> count    = 1;

    } else {

	m_this -> head -> previous = m_this;
	m_this -> next             = m_this -> head;
	m_this -> previous         = NULL;
	m_this -> head             = m_this;
	m_this -> count		   += 1;
    }

	/* unblock SIGTERM */

    unblockSigterm (oldMask);
}

/* SigtermHandler::~SigtermHandler

   Purpose: This procedure is the destructor for class SigtermHandler.
*/

SigtermHandler::~SigtermHandler (void)
{
	/* block SIGTEM */

    sigset_t oldMask = blockSigterm ();

	/* one less instance */

    m_this -> count -= 1;

	/* splice this instance out of the list */

    if (m_this -> count == 0) {

	m_this -> head = NULL;

    } else {

	if (m_this -> next != NULL) {
	    m_this -> next -> previous = m_this -> previous;
	}

	if (m_this -> previous != NULL) {
	    m_this -> previous -> next = m_this -> next;
	} else {
	    m_this -> head = m_this -> next;
	}
    }

	/* unblock SIGTERM and cleanup */

    unblockSigterm (oldMask);
    delete (m_this);
}

/* SigtermHandler::setExitCode

   Purpose: Set the exit code that will be used when a SIGTERM is received

     Input: exitcode = the exitcode
		       (int) 
*/

void SigtermHandler::setExitCode (int exitcode) 
{
    SigtermHandlerImp::exitcode = exitcode;
}
