/* Agent Tcl
   Bob Gray
   28 Spetember 1995

   platTimersImp.cc

   This file provides the UNIX implementation of Timers class.

   Copyright (c) 1995-1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include <sys/times.h>
#include <math.h>
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platSystem.h"
#include "platTimers.h"
#include "platTimeval.h"
#include "platTimersImp.h"
#include "truefalse.h"

     /* forward declarations */

static void gen_sigalrm_handler (int signo, SignalHandlers::ClientData);

     /* static data members */

TimersImplementation *Timers::m_this
	= NULL;

/* TimersImplementation::TimersImplementation

   Purpose: This procedure is the constructor for class TimersImplementation.
*/

TimersImplementation::TimersImplementation (void):
	count (0),
	timerLock (),
	wallTimers (),
	cpuTimers ()
{
 	// empty
}

/* Timers::disposeCpuTimers

   Purpose: Dispose of expired wall timers

     Input: None

    Output: The procedure disposes of expired wall timers by calling the
            corresponding handlers.

      Note: The procedure assumes that the lock has already been acquired.
*/

void Timers::disposeCpuTimers (void)
{
    int rc;
    struct Timer timer;      
    struct timeval cpu;
    struct timeval interval; 
    struct itimerval itimer; 

	/* dispose of the cpu timers */

    cpu = SystemUtil::getCurrentCpu();

    while (((rc = m_this -> cpuTimers.getMinimum(timer)) > 0) && (TimevalUtil::compareTimevals (cpu, timer.stop) >= 0)) {

	    /* remove an expired timer */

	m_this -> cpuTimers.removeMinimum (timer);

	    /* release the lock, call the timer trap and reacquire the lock */

	m_this -> timerLock.release ();
 
	if (timer.trap != NULL) {
	    timer.trap -> cpuAlarm ();
	} 

	m_this -> timerLock.acquire ();
	cpu = SystemUtil::getCurrentCpu ();
    }

	/* set a SIGPROF for the next timer */

    if (rc > 0) {
	interval = TimevalUtil::subTimevals (timer.stop, cpu);
        itimer.it_interval.tv_sec = 0;
	itimer.it_interval.tv_usec = 0;
	itimer.it_value.tv_sec = interval.tv_sec;
	itimer.it_value.tv_usec = interval.tv_usec;
	setitimer (ITIMER_PROF, &itimer, NULL);
    }
}

/* Timers::disposeWallTimers

   Purpose: Dispose of expired wall timers

     Input: None

    Output: The procedure disposes of expired wall timers by calling the
            corresponding handlers.

      Note: The procedure assumes that the lock has already been acquired.
*/

void Timers::disposeWallTimers (void)
{
    int rc;
    struct Timer timer;      
    struct timeval wall;
    struct timeval interval; 
    struct itimerval itimer; 

	/* dispose of the wall timers */

    wall = SystemUtil::getCurrentWall();

    while (((rc = m_this -> wallTimers.getMinimum (timer)) > 0) && (TimevalUtil::compareTimevals (wall, timer.stop) >= 0)) {

	    /* remove an expired timer */

	m_this -> wallTimers.removeMinimum (timer);

	    /* release the lock, call the timer trap and reacquire the lock */

	m_this -> timerLock.release ();

	if (timer.trap != NULL) {
	    timer.trap -> wallAlarm ();
	} 

	m_this -> timerLock.acquire ();
	wall = SystemUtil::getCurrentWall ();
    }

	/* set a SIGALRM for the next timer */

    if (rc > 0) {
	interval = TimevalUtil::subTimevals (timer.stop, wall);
        itimer.it_interval.tv_sec = 0;
	itimer.it_interval.tv_usec = 0;
	itimer.it_value.tv_sec = interval.tv_sec;
	itimer.it_value.tv_usec = interval.tv_usec;
	setitimer (ITIMER_REAL, &itimer, NULL);
    }
}

/* Timers::disposeTimers

   Purpose: Dispose of expired timers 

     Input: None

    Output: The procedure disposes of expired timers and calls the
            corresponding handlers.
*/

void Timers::disposeTimers (void)
{
    struct Timer timer;      
    struct timeval wall;
    struct timeval cpu;     
    struct timeval interval; 
    struct itimerval itimer; 

	/* nothing to do if we have no implementation */

    if (m_this == NULL) {
	return;
    }

	/* acquire the lock -- don't use Guard since we allow siglongjmp's */
	/* inside a timer trap                                             */

    m_this -> timerLock.acquire();

	/* dispose of the wall timers */

    wall = SystemUtil::getCurrentWall();

    while ((m_this -> wallTimers.getMinimum (timer)) && (TimevalUtil::compareTimevals (wall, timer.stop) >= 0)) {

	    /* remove an expired timer */

	m_this -> wallTimers.removeMinimum (timer);

	    /* release the lock, call the timer trap and reacquire the lock */

	m_this -> timerLock.release ();

	if (timer.trap != NULL) {
	    timer.trap -> wallAlarm ();
	} 

	m_this -> timerLock.acquire ();
	wall = SystemUtil::getCurrentWall ();
    }

	/* set a SIGALRM for the next timer */

    if (m_this -> wallTimers.getMinimum(timer) > 0) {
	interval = TimevalUtil::subTimevals (timer.stop, wall);
        itimer.it_interval.tv_sec = 0;
	itimer.it_interval.tv_usec = 0;
	itimer.it_value.tv_sec = interval.tv_sec;
	itimer.it_value.tv_usec = interval.tv_usec;
	setitimer (ITIMER_REAL, &itimer, NULL);
    }

	/* dispose of the cpu timers */

    cpu = SystemUtil::getCurrentCpu();

    while ((m_this -> cpuTimers.getMinimum(timer)) && (TimevalUtil::compareTimevals (cpu, timer.stop) >= 0)) {

	    /* remove an expired timer */

	m_this -> cpuTimers.removeMinimum (timer);

	    /* release the lock, call the timer trap and reacquire the lock */

	m_this -> timerLock.release ();
 
	if (timer.trap != NULL) {
	    timer.trap -> cpuAlarm ();
	} 

	m_this -> timerLock.acquire ();
	cpu = SystemUtil::getCurrentCpu ();
    }

	/* set a SIGPROF for the next timer */

    if (m_this -> cpuTimers.getMinimum(timer)) {
	interval = TimevalUtil::subTimevals (timer.stop, cpu);
        itimer.it_interval.tv_sec = 0;
	itimer.it_interval.tv_usec = 0;
	itimer.it_value.tv_sec = interval.tv_sec;
	itimer.it_value.tv_usec = interval.tv_usec;
	setitimer (ITIMER_PROF, &itimer, NULL);
    }

	/* release the lock */

    m_this -> timerLock.release (); 
}

/* Timers::Timers

   Purpose: This procedure is the constructor for class TIMERS.
*/

Timers::Timers (void)
{
	/* set up if this is the first instance */

    if (m_this == NULL) {

	    /* allocate the implementation class */

	m_this = new TimersImplementation ();
	m_this -> count += 1; 

	    /* install the SIGALRM and SIGPROF interrupts */
   
        SignalHandlers::install_signal_intr (SIGALRM, gen_sigalrm_handler, 0);
#ifndef ALLOW_PROFILING
        SignalHandlers::install_signal_intr (SIGPROF, gen_sigalrm_handler, 0);
#endif

    } else {

	Guard guard (m_this -> timerLock);    // automatically released
	m_this -> count += 1;
    }
}

/* Timers::~Timers

   Purpose: This procedure is the destructor for class TIMER.
*/

Timers::~Timers ()
{
    {
	Guard guard (m_this -> timerLock);   // automatically released 
	m_this -> count -= 1;
    }
}

/* Timers::removeAllAttachedToTrap

   Purpose: Remove all timers with a particular trap (i.e., callback routine)

     Input: trap = TimerTrap instance
		   (class TimerTrap *)

    Output: The procedure removes all timers with the given trap.
*/

void Timers::removeAllAttachedToTrap (TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);	// automatically released
    m_this -> wallTimers.removeAllAttachedToTrap (trap);
    m_this -> cpuTimers.removeAllAttachedToTrap (trap);
}

/* Timers::remove

   Purpose: Remove a timer from the set of timers

     Input: stop = time at which the timer was to expire 
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure removes the timer from the set of timers. 
*/

void Timers::remove (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);     // automatically released
    m_this -> wallTimers.remove (stop, trap);
}

/* Timers::add

   Purpose: Add a timer to the set of timers

     Input: stop = time at which the timer should expire
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure adds the timer to the set of timers.
*/

void Timers::add (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);     // automatically released
    m_this -> wallTimers.add (stop, trap);
    disposeWallTimers ();
}

/* Timers::removeCpu

   Purpose: Remove a cpu timer from the set of timers

     Input: stop = time at which the timer was to expire 
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure removes the timer from the set of timers. 
*/

void Timers::removeCpu (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);     // automatically released
    m_this -> cpuTimers.remove (stop, trap);
}

/* Timers::addCpu

   Purpose: Add a cpu timer to the set of timers

     Input: stop = time at which the timer should expire
		   (struct timeval)

	    trap = TimerTrap instance
		   (class TimerTrap *) 

    Output: The procedure adds the timer to the set of timers.
*/

void Timers::addCpu (struct timeval stop, TimerTrap *trap)
{
    Guard guard (m_this -> timerLock);	// automatically released
    m_this -> cpuTimers.add (stop, trap);
    disposeCpuTimers ();
}

/* gen_sigalrm_handler

   Purpose: Handle SIGALRM signals

     Input: signo = SIGALRM

    Output: The procedure disposes of the expired timer.
*/

static void gen_sigalrm_handler (int, SignalHandlers::ClientData)
{
    int oldErrno;

	/* save errno, dispose of the timers and restore errno */

    oldErrno = errno;
    Timers::disposeTimers ();
    errno = oldErrno;
}
