/* Agent Tcl
   Bob Gray
   8 September 1995

   tclRestrict.cc

   This file implements the library routines that handle restrictions.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif "restrictTcl.h"

#include "platPorting.h"
#include "platTimers.h"
#include "platTimeval.h"
#include "agentPermit.h"	// class Permit
#include "tcl.h"		// struct Tcl_Interp
#include "restrictTcl.h"
#include "tclAgent.h"
#include "tclRestrict.h"
#include "genManager.h"
#include "genRestrict.h"
#include "truefalse.h"

/* Restrict_ChangeBase

   Purpose: Change the base permit

     Input: interp = the current interpreter
		     (struct Tcl_Interp *)

            permit = the permit
		     (class Permit *)

    Output: The procedure adds the permit and returns TCL_OK.
*/

int Restrict_ChangeBase (Tcl_Interp *interp, Permit *permit)
{
    AGENT *agent;
    RESTRICT_TCL *restrict;

	/* assertions on the parameters */

    assert (interp != NULL);
    assert (permit != NULL);

	/* AGENT_TCL structure */

    agent = AGENT_TCL::get_agent (interp);
    assert (agent != NULL);
 
        /* RESTRUCT_TCL structure */

    restrict = (RESTRICT_TCL *) agent -> get_restrict ();
    assert (restrict != NULL);

	/* add the permit */

    restrict -> addBase (*permit);
    return TCL_OK;
}

/* Restrict_Add

   Purpose; Add a permit to the current set

     Input: interp = the current interpreter
	             (struct Tcl_Interp *)

            permit = the permit
		     (class Permit *)

    Output: The procedure adds the permit and returns TCL_OK.
*/

int Restrict_Add (Tcl_Interp *interp, Permit *permit)
{
    AGENT *agent;
    RESTRICT_TCL *restrict;

	/* assertions on the parameters */

    assert (interp != NULL);
    assert (permit != NULL);

	/* AGENT_TCL structure */

    agent = AGENT_TCL::get_agent (interp);
    assert (agent != NULL);
 
        /* RESTRUCT_TCL structure */

    restrict = (RESTRICT_TCL *) agent -> get_restrict ();
    assert (restrict != NULL);

	/* add the permit */

    restrict -> addUserPermit (*permit);
    return TCL_OK;
}

/* Restrict_Remove

   Purpose: Remove the most recent permit from the set of permits

     Input: interp = the current interpreter
	             (struct Tcl_Interp *)

    Output: The procedure returns TCL_ERROR and sets the interpreter result to
            an appropriate error message if there are no permits.  Otherwise
            the procedure returns TCL_OK.
*/

int Restrict_Remove (Tcl_Interp *interp)
{
    AGENT *agent;
    RESTRICT_TCL *restrict;

	/* assertion on the parameter */

    assert (interp != NULL);

	/* AGENT_TCL structure */ 

    agent = AGENT_TCL::get_agent (interp);
    assert (agent != NULL);

	/* error if there are no nonbase permits */

    restrict = (RESTRICT_TCL *) agent -> get_restrict ();
    assert (restrict != NULL);
 
    if (restrict -> getLevel() == 0) {
	Tcl_AppendResult (interp, "no permit to remove", (char *) NULL);
	return TCL_ERROR;
    }

	/* remove the most recent permit */

    restrict -> removeUserPermit ();
    return TCL_OK;
}
 
/* Restrict_ParsePermit

   Purpose: Parse a string that contains a list of permits

     Input: interp = the current interpreter
            string = the string
           
    Output: The procedure returns TCL_OK and sets the interpreter result to an
            appropriate error message on error.  Otherwise the procedure fills
	    in the PERMIT structure and returns TCL_OK. 
*/

int Restrict_ParsePermit (Tcl_Interp *interp, char *string, Permit *permit)
{
    int i;                  /* loop counter                  */
    int listArgc;                    /* number of permits in the list */
    char **listArgv;                 /* the permits                   */
    int permArgc;                    /* number of items in a permit   */
    char **permArgv;                 /* the items                     */
    double dscratch;
    int iscratch;
 
	/* defaults */

    listArgv = NULL;
    permArgv = NULL; 

	/* split apart the list of permits */

    if (Tcl_SplitList (interp, string, &listArgc, &listArgv) != TCL_OK) {

	Tcl_AppendResult (interp, ": permit list must be a valid Tcl list", TCL_STATIC);
	return TCL_ERROR;

    } else if (listArgc == 0) {

	Tcl_SetResult (interp, "permit list must contain at least one permit", TCL_STATIC);
	return TCL_ERROR;
    }

    SmartCharPtr listManager ((char *) listArgv);     // ensure deletion

	/* make an empty permit */

    Permit parsedPermit;

	/* parse the pieces of the permit */

    for (i = 0; i < listArgc; i++) {

	    /* split apart the permit */

	if (Tcl_SplitList (interp, listArgv[i], &permArgc, &permArgv) != TCL_OK) {

	    Tcl_AppendResult (interp, ": each permit must be a valid Tcl list: permit \"", (char *) NULL);
	    return TCL_ERROR;

	} else if (permArgc == 0) {

	    Tcl_SetResult (interp, "a permit can not be empty", TCL_STATIC);
	    return TCL_ERROR;
	}

	SmartCharPtr permManager ((char *) permArgv);

	    /* parse the permit */

	if (!strcmp(permArgv[0], "wall")) {

	    if (permArgc != 2) {
		Tcl_SetResult (interp, "a \"wall\" permit takes one parameter -- i.e., the number of seconds", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (Tcl_GetDouble (interp, permArgv[1], &dscratch) != TCL_OK) {
		Tcl_SetResult (interp, "number of seconds must be a floating point number", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (dscratch < 0.0) {
		Tcl_SetResult (interp, "number of seconds must be 0.0 or greater", TCL_STATIC);
		return TCL_ERROR;
	    }

	    parsedPermit.setWallLimit (TimevalUtil::secondsToTimeval (dscratch));

	} else if (!strcmp(permArgv[0], "cpu")) {

	    if (permArgc != 2) {
		Tcl_SetResult (interp, "a \"cpu\" permit takes one parameter", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (Tcl_GetDouble (interp, permArgv[1], &dscratch) != TCL_OK) {
		Tcl_SetResult (interp, "number of seconds must be a floating point number", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (dscratch < 0.0) {
		Tcl_SetResult (interp, "number of seconds must be 0.0 or greater", TCL_STATIC);
		return TCL_ERROR;
	    }

	    parsedPermit.setCpuLimit (TimevalUtil::secondsToTimeval (dscratch));

	} else if (!strcmp(permArgv[0], "jumps")) {

	    if (permArgc != 2) {
		Tcl_SetResult (interp, "a \"jumps\" permit takes one parameter", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (Tcl_GetInt (interp, permArgv[1], &iscratch) != TCL_OK) {
		Tcl_SetResult (interp, "number of jumps must be an integer", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (iscratch < 0) {
		Tcl_SetResult (interp, "number of jumps must be 0 or greater", TCL_STATIC);
		return TCL_ERROR;
	    }

	    parsedPermit.setJumpLimit ((UINT_32) iscratch);

	} else if (!strcmp(permArgv[0], "children")) {

	    if (permArgc != 2) {
		Tcl_SetResult (interp, "a \"children\" permit takes one parameter", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (Tcl_GetInt (interp, permArgv[1], &iscratch) != TCL_OK) {
		Tcl_SetResult (interp, "number of children must be an integer", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (iscratch < 0) {
		Tcl_SetResult (interp, "number of children must be 0 or greater", TCL_STATIC);
		return TCL_ERROR;
	    }

	    parsedPermit.setChildrenLimit ((UINT_32) iscratch);

	} else if (!strcmp(permArgv[0], "depth")) {

	    if (permArgc != 2) {
		Tcl_SetResult (interp, "a \"depth\" permit takes one parameter", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (Tcl_GetInt (interp, permArgv[1], &iscratch) != TCL_OK) {
		Tcl_SetResult (interp, "depth must be an integer", TCL_STATIC);
		return TCL_ERROR;
	    }

	    if (iscratch < 0) {
		Tcl_SetResult (interp, "depth must be 0 or greater", TCL_STATIC);
		return TCL_ERROR;
	    }

	    parsedPermit.setDepthLimit ((UINT_32) iscratch);

	} else {

	    Tcl_AppendResult (interp, "unknown permit type \"", permArgv[0], "\": should be \"wall\", \"cpu\", \"depth\", \"children\" or \"jumps\"", (char *) NULL);
	    return TCL_ERROR;
 	}
    } 
  
	/* final permit */

    *permit = parsedPermit;
    return TCL_OK;
}
