/* Agent Tcl
   Bob Gray
   25 January 1995
  
   tclRestrictCmd.cc

   This file implements the Tcl commands that handle restriction.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#include "platPorting.h"
#include "suppStrings.h"
#include "agentPermit.h"	// class Permit
#include "agentId.h"
#include "restrictTcl.h"
#include "tcl.h"
#include "tclAgent.h"
#include "tclMap.h"
#include "tclRestrict.h"
#include "truefalse.h"

/*
 * Forward declarations.
 */

#ifndef TCL8

static int restrictRestrictBody
	(ClientData data, Tcl_Interp *interp, int argc, char **argv);

#endif

static int Restrict_RestrictCmd
	(ClientData data, Tcl_Interp *interp, int argc, char **argv);

static int Restrict_RestrictBaseCmd
	(ClientData data, Tcl_Interp *interp, int argc, char **argv);

/* Restrict_RestrictBaseCmd

   Tcl syntax: restrict_base permits

      Purpose: Restrict the execution time of the entire agent
 
        Input: clientData = unused client data 
			    (ClientData)

	       interp     = the current interpreter
			    (struct Tcl_Interp *)

	       argc       = the number of command arguments
			    (int)

	       argv       = the command arguments
			    (char **)

       Output: The procedure returns TCL_ERROR and sets the interpreter result
	       to an appropriate error message on error.  Otherwise the
               procedure returns TCL_OK and changes the agent's overall permit.
*/

int Restrict_RestrictBaseCmd (ClientData, Tcl_Interp *interp, int argc, char **argv)
{
    Permit permit;
    AGENT_TCL *agent;
    RESTRICT_TCL *restrict;

	/* AGENT_TCL structure */

    agent = AGENT_TCL::get_agent (interp);
    assert (agent != NULL);

	/* RESTRICT_TCL structure */

    restrict = (RESTRICT_TCL *) agent -> get_restrict ();
    assert (restrict != NULL);

	/* check the number of arguments */

    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " permits\"", (char *) NULL);
	return TCL_ERROR;
    }

	/* parse the permit */

    if (Restrict_ParsePermit (interp, argv[1], &permit) != TCL_OK) {
	return TCL_ERROR;
    }

	/* add the new base permit */

    restrict -> addBase (permit);
    return TCL_OK;
} 
          
/* Restrict_RestrictCmd

   Tcl syntax: restrict permits script

      Purpose: Restrict the execution time of a Tcl script

        Input: clientData = client data 
			    (ClientData)	
			    (UNUSED)

	       interp     = the current interpreter
			    (struct Tcl_Interp *)

	       argc       = the number of command arguments
			    (int)

	       argv       = the command arguments
			    (char **)

       Output: The procedure returns TCL_ERROR and sets the interpreter result
	       to an appropriate error message on error.  Otherwise the         
	       procedure returns TCL_OK and sets the interpreter result to the 
               script result.
*/

#ifndef TCL8

#define CMD_RESTRICT_BODY	1

static int restrictRestrictBody (ClientData, Tcl_Interp *interp, int, char **)
{
    int flags;       
    char temp[60];
    AGENT *agent;
    RESTRICT_TCL *restrict;

	/* AGENT_TCL structure */

    agent = AGENT_TCL::get_agent (interp);
    assert (agent != NULL);

	/* RESTRICT_TCL structure */

    restrict = (RESTRICT_TCL *) agent -> get_restrict ();
    assert (restrict != NULL);
 
	/* Do we have a permit violation? */

    if (interp -> resultCode == TCL_PERMIT) {

	    /* Check for permit violations at the current level.  If the */
	    /* permit violation has occurred at a higher level, we will  */
	    /* throw the TCL_PERMIT code.		 		 */

	if ((flags = restrict -> checkCurrentPermit ()) > 0) {

	    Tcl_SetResult (interp, "permit violation", TCL_STATIC);

	    if (flags & Permit::e_PERMIT_WALL) {
		Tcl_AppendResult (interp, ": wall time exceeded", (char *) NULL);
		Tcl_SetErrorCode (interp, "PERMIT", "WALL", (char *) NULL);
	    } else if (flags & Permit::e_PERMIT_CPU) {
		Tcl_AppendResult (interp, ": cpu time exceeded", (char *) NULL);
		Tcl_SetErrorCode (interp, "PERMIT", "CPU", (char *) NULL);
	    } else {
		abort_with_message ("unexpected flag bit set");
	    }

	    restrict -> setAllResultCodes (TCL_OK);
	    interp -> resultCode = TCL_ERROR;
	}

	if (interp -> resultCode == TCL_ERROR) {

	    char *oldErrorInfo;

	    sprintf (temp, "\n    (\"restrict\" body line %d)", interp -> errorLine);
	    Tcl_AddErrorInfo (interp, temp);

	    oldErrorInfo = strcpyWithAlloc (Tcl_GetVar2 (interp, "errorInfo", (char *) NULL, TCL_GLOBAL_ONLY));
	    Tcl_SetVar2 (interp, "errorInfo", (char *) NULL, interp -> result, TCL_GLOBAL_ONLY);
	    Tcl_SetVar2 (interp, "errorInfo", (char *) NULL, oldErrorInfo, TCL_GLOBAL_ONLY | TCL_APPEND_VALUE);
	    delete (oldErrorInfo);
	}

    } else if (interp -> resultCode == TCL_ERROR) {

	sprintf (temp, "\n    (\"restrict\" body line %d)", interp -> errorLine);
	Tcl_AddErrorInfo (interp, temp);
    }
 
	/* finished with this permit */

    restrict -> removeUserPermit ();

	/* move on to the next command in the script */

    Stack_Pop (interp);
    return (interp -> resultCode);
}

#endif

int Restrict_RestrictCmd (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
#ifdef TCL8

    abort_with_message ("Tcl 8.0 does not yet support the restrict command");
    return TCL_OK;

#else

    int flag;
    Permit permit;
    AGENT *agent;
    RESTRICT_TCL *restrict;

	/* AGENT_TCL structure */

    agent = AGENT_TCL::get_agent (interp);
    assert (agent != NULL);

	/* RESTRICT_TCL structure */

    restrict = (RESTRICT_TCL *) agent -> get_restrict ();
    assert (restrict != NULL);

	/* If a state image has been captured and later restored, */
	/* we may be right in the middle of a "restrict" command  */

    Stack_GetFlag (interp, &flag);

    if (flag == CMD_RESTRICT_BODY) {
	return (restrictRestrictBody (clientData, interp, argc, argv));
    }
 
	/* check the number of arguments */

    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # of args: should be \"", argv[0], " permits script\"", (char *) NULL);
	return TCL_ERROR;
    }

	/* parse the permit */

    if (Restrict_ParsePermit (interp, argv[1], &permit) != TCL_OK) {
	return TCL_ERROR;
    }

	/* add the permit to current set */

    restrict -> addUserPermit (permit);

	/* push on the body */

    Stack_NewHandler (interp, CMD_RESTRICT_BODY, restrictRestrictBody);
    Stack_Push (interp, argv[2], TCL_STATIC);
    return TCL_OK;

#endif
}

/* Restrict_Init

   Purpose: Initialize the restrict extension

     Input: interp     = the current interpreter

    Output: The procedure returns TCL_ERROR on initialization failure
            and TCL_OK on initilization success.
*/

int Restrict_Init (Tcl_Interp *interp)
{
	/* assertions */

    assert (interp != NULL);

	/* get the master interpreter -- might be NULL */

    Tcl_Interp *masterInterp = Tcl_GetMaster (interp);

	/* install the restrict commands */

#ifdef TCL8
    Tcl_CreateCommand (interp, "restrict", Restrict_RestrictCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#else
    Tcl_CreateIntCommand (interp, "restrict", Restrict_RestrictCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif

    if (masterInterp == NULL) {

	Tcl_CreateCommand (interp, "restrict_base", Restrict_RestrictBaseCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    } else {

#ifdef TCL8	
	Tcl_CreateCommand (masterInterp, "restrict", Restrict_RestrictCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#else
	Tcl_CreateIntCommand (masterInterp, "restrict", Restrict_RestrictCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif
	Tcl_CreateCommand (masterInterp, "restrict_base", Restrict_RestrictBaseCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    }

	/* everything is okay */

    return TCL_OK;
}
