# Agent Tcl
# Bob Gray
# 2 August 1996
#
# require.tcl 
#
# This file defines the "require" command which is the interface between
# the agent and the resource managers (for built-in resources).  The
# "require" command accepts a resource name and access type, asks the
# appropriate resource manager whether the access should be allowed, and
# then updates the kernel tables to either grant or deny access.
#
# The procedure here is called "kernelRequire" since it is actually defined
# in the kernel interpreter.  The user interpreter will have a command
# called "require" that is linked to "kernelRequire".

#
# global variables:
#
# kern_seqenceNumber = sequence number for requests to the resource managers
#

set kern_sequenceNumber 0

#
# global variables initialized in safety.tcl:
#
# kern_allowAllExecs   = specificies whether execs are allowed
#
# kern_allowAllLoads   = specifies whether loads are allowed
#
# kern_allowAllNetwork = specifies whether network access is allowed
#
# kern_allowAllScreen  = specifies whether screen access is allowed
#
# kern_allowAllFile    = specifies whether *all* file access is allowed
#
# kern_allowFiles      = specifies the allowed access to *individual* files
#                        and directories
# 
# kern_allowSources    = specifies the directories that contain the scripts that
#                        we are allowed to source
#

proc kernelTalkToManager {request} glue {

    global agent kern_sequenceNumber 

#ifdef FIX_LATER
	# since masks are inefficient, this procedure is inefficient
#endif

	# mask so that we only get response from the resource manager

    set newMask [mask new]
    mask add $newMask "$agent(local-ip) resource-manager"
    set oldMask [mask_swap message $newMask]

	# send to the resource manager

    incr kern_sequenceNumber

    set message [list $kern_sequenceNumber $request]

    set code [catch {
	agent_send "$agent(local-ip) resource-manager" 0 $message
    } result]

    if {$code} {
	return -code error "permission denied: unable to contact resource manager"
    }

	# wait for the resource manager response

    while {1} {

	agent_receive code string -security securityInfo -blocking

	    # ignore responses from an agent that is attempting to masquerade
	    # as the resource manager

	if {[lindex $securityInfo 3] != "agent-auth"} {
	    continue
	}

	    # break apart the response

	set sequence [lindex $string 0]
	set answer   [lindex $string 1]
	set request  [lindex $string 2]

	    # ignore responses with the wrong sequence number

	if {$sequence != $kern_sequenceNumber} {
	    continue
	}

	    # have a valid response so bail out of the loop
	
	break
    }

    mask_replace message $oldMask

	# return the answer and request

    return [list $answer $request]
}

proc compareAccess {allowedAccess desiredAccess} glue {

     foreach access $desiredAccess {
	if {[lsearch -exact $allowedAccess $access] == "-1"} {
	    return "no-confirmed"
	}
     }

     return "yes"
}

#ifdef FIX_LATER
# kernelRequireFile, kernelRequireDirectory and kernelRequireFileOrDirectory
# have a lot of code in common 
#endif

proc kernelRequireFile {access} glue {

    global kern_allowAllFile kern_allowFiles kern_allowDirectories

	# break out the access pieces

    set desiredAccess [lindex $access 0]
    set filename      [lindex $access 1]

	# then check the overall flag

    if {$kern_allowAllFile == "yes"} {
	return "yes"
    }

	# then check the individual directories

    set path [file split $filename]

    if [info exists kern_allowFiles($path)] {

	set allowedAccess $kern_allowFiles($path)

	if {[compareAccess $allowedAccess $desiredAccess] == "yes"} {
	    return yes
	} else {
	    return no-confirmed
	}
    }

    while {1} {

	set path [lreplace $path end end]

	if {$path == ""} {
	    break
	}

	if [info exists kern_allowDirectories($path)] {

	    set allowedAccess $kern_allowDirectories($path)

	    if {[compareAccess $allowedAccess $desiredAccess] == "yes"} {
		return yes
	    } else {
		return no-confirmed
	    }
	}
    }

	# then check the overall flag again

    if {$kern_allowAllFile == "no-confirmed"} {
	return no-confirmed
    }

 	# otherwise contact the resource manager

    set request [list file [list $desiredAccess $filename]]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# now adjust kern_allowAllFile

    if {$answer == "yes"} {
	set kern_allowAllFile yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllFile no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireDirectory {access} glue {

    global kern_allowAllFile kern_allowFiles kern_allowDirectories

	# break out the access pieces

    set desiredAccess [lindex $access 0]
    set dirname       [lindex $access 1]

	# then check the overall flag

    if {$kern_allowAllFile == "yes"} {
	return "yes"
    }

	# then check the individual directories

    set path [file split $dirname]

    while {1} {

	if [info exists kern_allowDirectories($path)] {

	    set allowedAccess $kern_allowDirectories($path)

	    if {[compareAccess $allowedAccess $desiredAccess] == "yes"} {
		return yes
	    } else {
		return no-confirmed
	    }
	}

	set path [lreplace $path end end]

	if {$path == ""} {
	    break
	}
    }

	# then check the overall flag again

    if {$kern_allowAllFile == "no-confirmed"} {
	return no-confirmed
    }

 	# otherwise contact the resource manager

    set request [list file [list $desiredAccess $dirname]]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# now adjust kern_allowAllFile

    if {$answer == "yes"} {
	set kern_allowAllFile yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllFile no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireFileOrDirectory {access} glue {

    global kern_allowAllFile kern_allowFiles kern_allowDirectories

	# break out the access pieces

    set desiredAccess [lindex $access 0]
    set filename      [lindex $access 1]

	# then check the overall flag

    if {$kern_allowAllFile == "yes"} {
	return "yes"
    }

	# then check the individual directories

    set path [file split $filename]

    if [info exists kern_allowFiles($path)] {

	set allowedAccess $kern_allowFiles($path)

	if {[compareAccess $allowedAccess $desiredAccess] == "yes"} {
	    return yes
	} else {
	    return no-confirmed
	}
    }

    while {1} {

	if [info exists kern_allowDirectories($path)] {

	    set allowedAccess $kern_allowDirectories($path)

	    if {[compareAccess $allowedAccess $desiredAccess] == "yes"} {
		return yes
	    } else {
		return no-confirmed
	    }
	}

	set path [lreplace $path end end]

	if {$path == ""} {
	    break
	}
    }

	# then check the overall flag again

    if {$kern_allowAllFile == "no-confirmed"} {
	return no-confirmed
    }

 	# otherwise contact the resource manager

    set request [list file [list $desiredAccess $filename]]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# now adjust kern_allowAllFile

    if {$answer == "yes"} {
	set kern_allowAllFile yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllFile no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireScreen {display} glue {

    global kern_allowAllScreen

	# see if we already know the answer

    if {$kern_allowAllScreen == "yes"} {
	return "yes"
    }

    if {$kern_allowAllScreen == "no-confirmed"} {
	return "no-confirmed"
    }

	# otherwise contact the resource manager

    set request [list screen $display]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# adjust kern_allowAllExecs according to the answer

    if {$answer == "yes"} {
	set kern_allowAllScreen yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllScreen no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireLoad {library} glue {

    global kern_allowAllLoads

	# see if we already know the answer

    if {$kern_allowAllLoads == "yes"} {
	return "yes"
    }

    if {$kern_allowAllLoads == "no-confirmed"} {
	return "no-confirmed"
    }

	# otherwise contact the resource manager

    set request [list load $library]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# adjust kern_allowAllExecs according to the answer

    if {$answer == "yes"} {
	set kern_allowAllLoads yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllLoads no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireSource {directory} glue {

    global kern_allowSources

	# see if the directory (or one of its super-directories) is already
	# in kern_allowSources

    set path $directory

    while {1} {

	if {[info exists kern_allowSources($path)]} {
	    if {$kern_allowSources($path) == "1"} {
		return yes
	    } else {
		return no-confirmed
	    }
	}

	set path [lreplace $path end end]

	if {"$path" == ""} {
	    break
	}
    }

	# otherwise consult with the resource manager

    set request [list source $directory]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# now add to kern_allowSources

    if {$answer == "yes"} {
	set kern_allowSources($directory) 1
    } elseif {$answer == "no-confirmed"} {
	set kern_allowSources($directory) 0
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireNetwork {location} glue {

    global kern_allowAllNetwork

	# see if we already know the answer
 
    if {$kern_allowAllNetwork == "yes"} {
	return "yes"
    }

    if {$kern_allowAllNetwork == "no-confirmed"} {
	return "no-confirmed"
    }

	# otherwise contact the resource manager

    set request [list network $location]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# adjust kern_allowAllExecs according to the answer

    if {$answer == "yes"} {
	set kern_allowAllNetwork yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllNetwork no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireExec {program} glue {

    global kern_allowAllExecs 

	# see if we already know the answer

    if {$kern_allowAllExecs == "yes"} {
	return "yes"
    }

    if {$kern_allowAllExecs == "no-confirmed"} {
	return "no-confirmed"
    }

	# otherwise contact the resource manager

    set request [list exec $program]

    set code [catch {kernelTalkToManager $request} response]

    if {$code} {
	return -code error $response
    }

    set answer [lindex $response 0]
    set request [lindex $response 1]

	# adjust kern_allowAllExecs according to the answer

    if {$answer == "yes"} {
	set kern_allowAllExecs yes
    } elseif {$answer == "no-confirmed"} {
	set kern_allowAllExecs no-confirmed
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireWall {amount} glue {

    global restrict

	# get the current wall time limit

    set wallLimit [lindex [lindex $restrict(0) 0] 1]
 
	# calculate the total amount of wall time

    set desiredWall [expr $restrict(wall-elapsed) + $amount]

	# allow immediately if we are reducing the overall limit; check with
	# the resource manager if we are increasing the overall limit

    if {$desiredWall <= $wallLimit} {

	set answer yes

    } else {

	set desiredGroupWall [expr $restrict(group-wall-elapsed) + $amount]
	set request [list wall $desiredGroupWall]

	set code [catch {kernelTalkToManager $request} response]

	if {$code} {
	    return -code error $response
	}

	set answer [lindex $response 0]
	set request [lindex $response 1]
    }

	# adjust the overall amount of wall time 
    
    if {$answer == "yes"} {
	restrict_base [list [list wall $desiredWall]]
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireCpu {amount} glue {

    global restrict

	# get the current cpu time limit

    set cpuLimit [lindex [lindex $restrict(0) 1] 1]

	# calculate the total amount of cpu time

    set desiredCpu [expr $restrict(cpu-elapsed) + $amount]

	# allow imediately if we are reducing the overall limit; check with
	# the resource manager if we are increasing the overall limit

    if {$desiredCpu <= $cpuLimit} {

	set answer yes

    } else {

	set desiredGroupCpu [expr $restrict(group-cpu-elapsed) + $amount]
	set request [list cpu $desiredGroupCpu]

	set code [catch {kernelTalkToManager $request} response]

	if {$code} {
	    return -code error $response
	}

	set answer [lindex $response 0]
	set request [lindex $response 1]
    }

	# adjust the overall amount of cpu time 
    
    if {$answer == "yes"} {
	restrict_base [list [list cpu $desiredCpu]]
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireJumps {amount} glue {

    global restrict

	# get the current jumps limit

    set jumpsLimit [lindex [lindex $restrict(0) 2] 1]

	# calculate the total number of jumps

    set desiredJumps [expr $restrict(jumps) + $amount]

	# allow imediately if we are reducing the overall limit; check with
	# the resource manager if we are increasing the overall limit

    if {$desiredJumps <= $jumpsLimit} {

	set answer yes

    } else {

	set desiredGroupJumps [expr $restrict(group-jumps) + $amount]
	set request [list jumps $desiredGroupJumps]

	set code [catch {kernelTalkToManager $request} response]

	if {$code} {
	    return -code error $response
	}

	set answer [lindex $response 0]
	set request [lindex $response 1]
    }

	# adjust the overall number of jumps 
    
    if {$answer == "yes"} {
	restrict_base [list [list jumps $desiredJumps]]
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireChildren {amount} glue {

    global restrict 

	# get the current children limit

    set childrenLimit [lindex [lindex $restrict(0) 3] 1]

	# calculate the total number of children

    set desiredChildren [expr $restrict(children) + $amount]

	# allow imediately if we are reducing the overall limit; check with
	# the resource manager if we are increasing the overall limit

    if {$desiredChildren <= $childrenLimit} {

	set answer yes

    } else {

	set desiredGroupChildren [expr $restrict(group-children) + $amount]
	set request [list children $desiredGroupChildren]

	set code [catch {kernelTalkToManager $request} response]

	if {$code} {
	    return -code error $response
	}

	set answer [lindex $response 0]
	set request [lindex $response 1]
    }

	# adjust the overall number of children
    
    if {$answer == "yes"} {
	restrict_base [list [list children $desiredChildren]]
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequireDepth {amount} glue {

    global restrict 

	# get the current depth limit

    set depthLimit [lindex [lindex $restrict(0) 4] 1]

	# calculate the total depth

    set desiredDepth [expr $restrict(depth) + $amount]

	# allow imediately if we are reducing the overall limit; check with
	# the resource manager if we are increasing the overall limit

    if {$desiredDepth <= $depthLimit} {

	set answer yes

    } else {

	set desiredGroupDepth [expr $restrict(group-depth) + $amount]
	set request [list depth $desiredGroupDepth]

	set code [catch {kernelTalkToManager $request} response]

	if {$code} {
	    return -code error $response
	}

	set answer [lindex $response 0]
	set request [lindex $response 1]
    }

	# adjust the overall depth

    if {$answer == "yes"} {
	restrict_base [list [list depth $desiredDepth]]
    } else {
	set answer no
    }

    return $answer
}

proc kernelRequire {resource access} glue {

	# case-independent

    set resource [string tolower $resource]

	# switch on the resource name to decide which resource manager
	# to contact

    switch $resource {
	exec 		{set answer [kernelRequireExec $access]}
	source  	{set answer [kernelRequireSource $access]}
	load 		{set answer [kernelRequireLoad $access]}
	network		{set answer [kernelRequireNetwork $access]}
	screen		{set answer [kernelRequireScreen $access]}
	file		{set answer [kernelRequireFile $access]}
	directory	{set answer [kernelRequireDirectory $access]}
	wall		{set answer [kernelRequireWall $access]}
	cpu		{set answer [kernelRequireCpu $access]}
	jumps		{set answer [kernelRequireJumps $access]}
	children	{set answer [kernelRequireChildren $access]}
	depth		{set answer [kernelRequireDepth $access]}
	wall	 	{set answer [kernelRequireWall $access]}
	default 	{set answer "no-confirmed"}
    }

    return $answer
}

proc userRequire {interp resource access} {

    set code [
	catch {
	    kernelRequire $resource $access
	} answer
    ]

    if {$code} {
	return -code error $answer
    } elseif {$answer != "yes"} {
	return -code error "permission denied: $resource $access"
    } else {
	return $answer
    }
}
