# Bob Gray
# 2 August 1996
#
# safety.tcl 
#
# This file defines safe versions of the unsafe Tcl commands.  These safe
# versions are loaded into the kernel interpreter.  This file will eventually 
# be split into multiple files according to the type of the command.



# ---------------------------------------------------------------------------
#
# make sure that we are at the top level
#
# ---------------------------------------------------------------------------

if {[info level] != 0} {
  return -code error "\"safety.tcl\" can only be sourced at the global level"
}



# ---------------------------------------------------------------------------
#
# Utility routines
#
# ---------------------------------------------------------------------------

proc constructDeniedMessage {base arguments} glue {

    set message "permission denied: $base"

    foreach arg $arguments {
	append message " [list $arg]"
    }

    return $message 
}



#ifdef FIX_LATER
# ---------------------------------------------------------------------------
# It is unnecessary (but convenient) to have "source" as a separate action.  
# The agent should be allowed to source any file to which it has read access 
# (since if it can read the file, it just scan in the complete contents and 
# pass those contents to the Tcl eval command anyway).
# ---------------------------------------------------------------------------
#

# ---------------------------------------------------------------------------
#
# Source commands
#
# There is currently one relevant variable defined in the kernel interpreter.
#
#   kern_allowSources($dir)
#
#     If this array element exists and is set to 1, the agent is allowed to 
#     source any file that appears in directory $dir or any of its 
#     subdirectories.  If the array element does not exist or is set to 0,
#     the agent is not allowed to source the files that appear in directory
#     $dir or any of its subdirectories.  
#
# The kern_allowSources array elements default to just the Tcl library directory 
# and the Tk library directory and can only be changed by the "kernelRequire"
# command (in combination with the resource managers).
#
# Thus the overall security rule is ...
#
# The agent can either (1) source any script that can be accessed by the
# interpreter userid OR (2) source only those scripts that appear in the 
# Tcl and Tk library directories.
#
# This security rule is extremely simple -- i.e., there are only two classes
# of agents at the Tcl language level -- but provides enough flexibility to
# support both anonymous and trusted agents.  It is also reasonably easy to
# get it right and to verify its correctness which were the critical concerns
# in this release of Agent Tcl. 
#
# ---------------------------------------------------------------------------

# The following code (borrowed from Tcl 7.6) makes sure that tcl_library is 
# set to the names of the actual directories. These names may be different 
# than the assigned values if the actual directories are at the end of a 
# symbolic link. We need to do this to ensure that it will be possible to 
# load files from the Tcl and Tk libraries even when there is a symbolic link 
# somewhere in the pathname.

if {$tcl_platform(platform) == "macintosh"} {

    set tcl_library "resource"

} else {

    set curdir [pwd]
    cd $tcl_library
    set tcl_library [pwd]
    cd $curdir
}

# default is to allow a source only from the Tcl library directory

set kern_allowSources([file split $tcl_library]) 1

# safe versions of the commands (much of this is borrowed from Tcl 7.6)

proc getSourceDirectory {file} glue {

    if {$file == ""} {
	return no-confirmed
    }

    if [file isdir $file] {
	return no-confirmed
    }

    set path [file dirname $file]

    set oldPwd [pwd]

    set code [catch {
	cd $path
	set actualPath [pwd]
    }]

    cd $oldPwd

    if {$code} {
	return no-confirmed
    }

    set realpath [file split $actualPath]
    return [list yes $realpath]
}

proc kernelSource {interp file} glue {

    global errorCode errorInfo

	# get the directory name

    set code [catch {getSourceDirectory $file} answer]

    if {$code} {
	return -code error "permission denied: source $file"
    }

    if {$answer == "no-confirmed"} {
	return -code error "permission denied: source $file"
    }

    set directory [lindex $answer 1]

	# see if we are allowed to source files in that directory

    set code [catch {kernelRequireSource $directory} answer]

    if {$code} {
	return -code error $answer
    }

    if {$answer != "yes"} {
	return -code error "permission denied: source $file"
    }

	# if we make it here, it is okay

    set fd [open $file r]
    set r [read $fd]
    close $fd

	# do the source

    interp eval $interp $r
}



# ---------------------------------------------------------------------------
#
# File commands
#
# There are currently three relevant variables defined in the kernel interpreter.
# The first directly controls file access.
#
#   kern_allowAllFile 
#
#     This variable is set to "yes" if the agent can access any file that the
#     interpreter userid can access.  The variable is set to "no-confirmed" if 
#     the agent can only access some subset of those files and has already 
#     contacted the resource manager.  The variable is set to "no" if the 
#     agent can only access some subset of those files and has not yet 
#     contacted the resource manager.
#
#   kern_allowFiles($file)
#
#     If this array element exists, then its value determines the allowed
#     access to file $file.  Its value will be a list containing one or
#     more of the following: create, read, write, remove.  "create" means that
#     the agent is allowed to create the file.  "write" means that the agent
#     is allowed to write to the file *if it already exists*.  "read" means
#     that the agent is allowed to read from the file if it already exists
#     (and that the agent is allowed to obtain information about the file
#     with the "file" command).  "remove" means that the agent is allowed to 
#     remove the file.
#
#     NOTE: The "remove" element is not used in the current system since Tcl
#     7.5 does not provide built-in commands for removing a file.
#
#   kern_allowDirectories($dir) 
#
#     If this array element exists, then its value determines the allowed
#     access to directory $dir.  Its value will be a list containing one or
#     more of the following: create, read, write, remove, create-dir,
#     create-subdir, remove-subdir, remove-dir.  "create" means that the
#     agent is allowed to create files inside that directory.  "write" means
#     that the agent is allowed to write to the *existing* files in that
#     directory.  "read" means that the agent is allowed to read from the 
#     *existing* files inside the director (and that the agent is allowed to
#     obtain information about the files and subdirectories with the "file" 
#     command).  "remove" means that the agent is allowed to remove files from 
#     the directory.  "create-dir" means that the agent is allowed to create 
#     the directory.  "create-subdir" means that the agent is allowed to create 
#     subdirectories inside the directory.  "remove-subdir" means that the agent 
#     is allowed to remove subdirectories from the directory.  "remove-dir" 
#     means that the agent is allowed to remove the directory.
#
#     NOTE: "create-dir", "create-subdir", "remove-subdir" and "remove-dir"
#     are not used in the current system since Tcl 7.5 has no built-in 
#     commands for creating and deleting directories.
#
# The third variable simply keeps track of the file descriptors that the
# kernel interpreter has opened on behalf of the user interpreter.
#
#   kern_openDescriptors($fd)
#
#     If this array element exists, then $fd is a file descriptor that has 
#     been opened on behalf of the user interpreter.  The contents of the 
#     array element is a two-element list that gives the type of the file 
#     descriptor and the access permissions.
#
# kern_allowAllFile defaults to "no" and is changed to "yes" or 
# "no-confirmed" only by the "kernelRequire"command (in coordination with 
# the resource managers). 
#
# Thus the overall security rule is ...
#
# The agent can either (1) access all the files that can be accessed by the
# interpreter userid OR (2) can access only those files identified in the
# kern_allowFiles and kern_allowDirectories maps.
#
# ---------------------------------------------------------------------------

# default is to not allow any file access except for read access to the
# tcl_library directory

set kern_allowAllFile no

set kern_allowDirectories([file split $tcl_library]) [list read]

proc checkFdAccess {fd access} glue {

    global kern_fileDescriptors

	# make sure that this is a file descriptor that the kernel
	# interpreter has opened on behalf of the user interpreter

    if {![info exists kern_fileDescriptors($fd)]} {
	return no-confirmed
    }

	# make sure that we have access to the file

    set accessList [lindex $kern_fileDescriptors($fd) 1]

    if {[lsearch $accessList $access] == "-1"} {
	return no-confirmed
    }

    return yes
}

proc getActualFilename {filename} {

    if {$filename == ""} {
	return no-confirmed
    }

    set path [file dirname $filename]
    set file [file tail $filename]

    set oldPwd [pwd]

    set code [catch {
	cd $path
	set actualPath [pwd]
    }]

    cd $oldPwd

    if {$code} { 
	return no-confirmed
    }
 
    set actualFilename [file join $actualPath $file]
    return [list yes $actualFilename]
}

proc kernelOpen {interp filename {mode r} {permissions {}}} glue {

    global kern_fileDescriptors

	# convert the mode to an access type 

#ifdef FIX_LATER
	# our desiredAccess lists are extremely simplistic.
#endif

    switch $mode {
	r	{set desiredAccess {read}}
	r+	{set desiredAccess {read write}}
	w	{set desiredAccess {create write}}
	w+	{set desiredAccess {create read write}}
	a	{set desiredAccess {create write}}
	a+	{set desiredAccess {create read write}}
	default {return -code error "permission denied: invalid mode \"$mode\""}
    }

	# call into the exec security checks if this is a command pipeline

    set filename [string trim $filename]

    if {[string index $filename 0] == "|"} {
	return [kernelExecPipeline $interp $filename $mode $desiredAccess]
    }

	# get the canonical filename 

    set code [catch {getActualFilename $filename} answer]

    if {$code} {
	return -code error $answer
    }

    if {$answer == "no-confirmed"} {
	return -code error "permission denied: open $filename $mode"
    }

    set actualFilename [lindex $answer 1]

	# bail if this is a directory

    if [file isdir $filename] {
	return -code error "permission denied: open $filename $mode"
    }

        # see if we are allowed to access the file

    set code [catch {
	kernelRequireFile [list $desiredAccess $actualFilename]
    } answer]

    if {$code} {
	return -code error $answer
    }

    if {$answer != "yes"} {
	return -code error "permission denied: open $filename $mode"
    }

	# if we make it here, we can open the file

    if {$permissions == ""} {
	set code [catch {open $filename $mode} result]
    } else {
	set code [catch {open $filename $mode $permissions} result]
    }

	# if the file opened successfully, transfer file to slave

    if {$code == 0} {
	set fd $result
	interp transfer {} $fd $interp
	set kern_fileDescriptors($fd) [list file $desiredAccess]
    }

	# done

    return -code $code $result
}

proc kernelClose {interp fd} glue {

    global kern_fileDescriptors errorCode errorInfo

	# make sure that this is a file descriptor that the kernel
	# interpreter has opened on behalf of the user interpreter

    if {![info exists kern_fileDescriptors($fd)]} {
	error "permission denied: close $fd"
    }

	# if we make it here, we can close the file

    interp transfer $interp $fd {}
    set code [catch {close $fd} result]

	# remove it from the list of open descriptors

    unset kern_fileDescriptors($fd)

	# and we're done

    return -code $code -errorinfo $errorInfo -errorcode $errorCode $result
}

proc kernelFconfigure {interp fd args} glue {

    global errorCode errorInfo kern_fileDescriptors

	# make sure that this is a file descriptor that the kernel
	# interpreter has opened on behalf of the user interpreter

    if {![info exists kern_fileDescriptors($fd)]} {
	error [constructDeniedMessage "fconfigure $fd" $args]
    }

	# if we make it here, do it -- first share the file with the kernel

    interp share $interp $fd {}

	# then execute the command

    set command [concat fconfigure $fd $args]
    set code [catch {eval $command} result]

	# close the shared file -- file stays open in the user interpreter

    close $fd

	# done

    return -code $code -errorinfo $errorInfo -errorcode $errorCode $result
}

proc kernelCd {interp {dirname ~}} glue {

	# get the canonical dirname

    set code [catch {getActualFilename $dirname} result]

    if {$code} {
	return -code error $result
    }

    if {$result == "no-confirmed"} {
	return -code error "permission denied: cd $dirname"
    }

    set actualDirname [lindex $result 1]
 
	# see if we are allowed to access the directory

    set code [catch {
	kernelRequireDirectory [list read $actualDirname]
    } result]

    if {$code} {
	return -code error $result
    }

    if {$result != "yes"} {
	return -code error "permission denied: cd $dirname"
    }

	# if we make it here, we can do it

    set code [catch {cd $dirname} result]

	# done

    return -code $code $result
}

proc kernelPwd {interp} glue {

	# get the current working directory

    set pwd [pwd]

	# see if we are allowed to access that directory

    set code [catch {
	kernelRequireDirectory [list read $pwd]
    } result]

    if {$code} {
	return -code error $result
    }

    if {$result != "yes"} {
	return -code error "permission denied: pwd"
    }
     
	# if we make it here, we can do it

    return $pwd
}

proc kernelFile {interp args} glue {

	# certain operations should always be allowed

    set subcommand [lindex $args 0]
    set filename   [lindex $args 1]
 
    foreach pattern {dirname join ro.* ext.* ta.* pathtype sp.*} {

	if {[regexp $pattern $subcommand]} {

	    set command [concat file $args]
	    set code [catch {eval $command} result]
	    return -code $code $result
	}
    }

	# certain operations should be allowed if we have read access to the file

    foreach pattern {atime exe.* exi.* isd.* isf.* lstat mtime \
			owned reada.* readl.* si.* st.* ty.* writable} {

	if {[regexp $pattern $subcommand]} {

		# get the canonical filename

	    set code [catch {getActualFilename $filename} answer]

	    if {$code} {
		return -code error $answer 
	    }

	    if {$answer == "no-confirmed"} {
		return -code error [constructDeniedMessage file $args]
	    }

	    set actualFilename [lindex $answer 1]
	
		# see if we have access to the file or directory

	    set code [catch {
		kernelRequireFileOrDirectory [list read $actualFilename]
	    } answer]

	    if {$code} {
		return -code error $answer
	    }

	    if {$answer != "yes"} {
		return -code error [constructDeniedMessage file $args]
	    }

		# if we made it to here, we can do it
	
	    set command [concat file $args]
	    set code [catch {eval $command} result]
	    return -code $code $result
	}
    }

	# no other subcommands in Tcl 7.5 so execute to get the error message

    set command [concat file $args]
    set code [catch {eval $command} $result]
    return -code error $result
}

proc kernelGlob {interp args} glue {

#ifdef FIX_LATER
	# this is incredibly inefficient (since we are calling 
	# getActualFilename and kernelRequireFilename on every
	# file that matches the glob pattern)
#endif

	# do the glob command

    set command [concat glob $args]
    set code [catch {eval $command} result]

	# bail on error

    if {$code} {
	
	if {[string range $result 0 7] == "no files"} {
	    return -code error "permission denied: no accessible files matched the glob pattern"
	}

	return -code error $result
    }

	# go through the list of files and keep only those to which the
	# agent has access

    set keepFiles {}

    foreach file $result {

	    # get the canonical filename

	set code [catch {getActualFilename $file} answer]

	if {$code} {
	    continue
	}

	if {$answer == "no-confirmed"} {
	    continue
	}

	set actualFilename [lindex $answer 1]

	    # see if we have access to the file

	set code [catch {
	    kernelRequireFileOrDirectory [list read $actualFilename]
	} answer]

	if {$code} {
	    return -code error $answer
	}

	if {$answer != "yes"} {
	    continue
	}

	    # we have access to the file so add it to the list

	lappend keepFiles $file
    }

	# if the resulting list of accessible files is empty, we will either
	# return an empty list or raise an error depending on the presence or
	# absence of the -nocomplain option

    if {$keepFiles == ""} {
	
	foreach arg $args {
	    if {$arg == "-nocomplain"} {
		return {}
	    }
	}

	return -code error "permission denied: no accessible files matched the glob pattern"
    }
 
	# now the presence or absence of the -nocomplain option determines
	# whether or not we raise an error if the resultn
	# done

    return $keepFiles
}

proc kernelPuts {interp fd {string {__no_string__}} {extra {}}} glue {

    global kern_fileDescriptors errorCode errorInfo

	# remember the original command

    set arguments [list $fd $string $extra]

	# see if we have a -nonewline option

    set nonewline 0

    if {$fd == "-nonewline"} {
	set nonewline 1
	set fd $string
	set string $extra
	set extra {}
    }

	# see if we have an implicitly specified fd 

    if {$string == "__no_string__"} {
	set string $fd
	set fd stdout
    }

	# make sure that we have write access to the fd

    if {[checkFdAccess $fd write] != "yes"} {
	error [constructDeniedMessage puts $arguments]
    }

	# if we make it here, we can do it

    interp share $interp $fd {}

    if {$nonewline == 1} {
	set code [catch {puts -nonewline $fd $string} result]
    } else {
	set code [catch {puts $fd $string} result]
    }

    close $fd

	# done

    return -code $code -errorinfo $errorInfo -errorcode $errorCode $result
}



# ---------------------------------------------------------------------------
#
# Load commands
#
# There is currently one relevant variable defined in the kernel interpreter.
#
#   kern_allowAllLoads
#
#     This variable is set to "yes" if the agent can load any shared library
#     that the interpreter userid can load.  The variable is set to
#     "no_confirmed" if the agent can load any libraries.  The variable is
#     set to "no" if the agent can not load any libraries but has not yet
#     contacted the resource manager.
#
# kern_allowAllLoads defaults to "no" and is changed to "yes" or "no-confirmed"
# only by the "kernelRequire" command (in coordination with the resource
# managers).
#
# Thus the overall security rule is ...
#
# The agent can either (1) load all the libraries that can be loaded by the
# interpreter userid or (2) load no libraries at all.
#
# This security rule is extremely simple -- i.e., there are only two classes
# of agents at the Tcl language level -- but provides enough flexibility to
# support both anonymous and trusted agents.  It is also reasonably easy to
# get it right and to verify its correctness which were the critical concerns
# in this first secure release. 
#
# NOTE: 
#
# Currently the program to be loaded is identified according to its name 
# within the file system.  Thus, as the security mechanisms evolve, it might 
# be necessary to more tightly integrate the "load" and "file" security checks.
#
# ---------------------------------------------------------------------------

# default is to not allow any exec's 

set kern_allowAllLoads no

proc kernelLoad {interp fileName {packageName {}} {interpName {}}} glue {

    global kern_allowAllLoads errorCode errorInfo

	# did the user incorrectly specify an interpreter name?

    if {$interpName != ""} {
	return -code error "permission denied: can not specify an interpreter name"
    }

	# see if we are allowed to load the library

    set splitname [file split $fileName]

    set code [catch {kernelRequireLoad $splitname} answer]

    if {$code} {
	return -code error $answer
    }

    if {$answer != "yes"} {
	return -code error "permission denied: load $fileName"
    }

	# if we make it here, we can do it

    set code [catch {load $fileName $packageName $interp} result]

	# done

    return -code $code -errorcode $errorCode -errorinfo $errorInfo $result
}



# ---------------------------------------------------------------------------
#
# Exec commands
#
# There is currently one relevant variable defined in the kernel interpreter.
#
#   kern_allowAllExecs
#
#     This variable is set to "yes" if the agent can exec any program that the
#     interpreter userid can exec.  The variable is set to "no-confirmed" if 
#     the agent can not exec any programs.  The variable is set to "no"
#     if the agent can not exec any programs but has not yet contacted the
#     resource manager.
#
# kern_allowAllExecs defaults to "no" and is changed to "yes" or "no-confirmed"
# only by the "kernelRequire"command (in coordination with the resource 
# managers).  
#
# Thus the overall security rule is ...
#
# The agent can either (1) exec all the programs that can be execed by the
# interpreter userid OR (2) exec no programs at all. 
#
# This security rule is extremely simple -- i.e., there are only two classes
# of agents at the Tcl language level -- but provides enough flexibility to
# support both anonymous and trusted agents.  It is also reasonably easy to
# get it right and to verify its correctness which were the critical concerns
# in this first secure release. 
#
# NOTES: 
#
# Currently the program to be execed is identified according to its name 
# within the file system.  Thus, as the security mechanisms evolve, it might 
# be necessary to more tightly integrate the "exec" and "file" security 
# checks.
#
# The kernelExec procedure checks for security violations that are common
# to all execed programs.  It does not have any control over what the
# program does once it is actually execed.  Therefore if the agent userid
# can exec particularly dangerous programs, you might want to further 
# restrict kernelExec (even to the point of always returning a security
# exception), and then optionally provide individual Tcl commands that exec 
# the dangerous programs in a safe way.
#
# ---------------------------------------------------------------------------

# default is to not allow any exec's 

set kern_allowAllExecs no

proc breakout {listName iName arg args length} glue {

    upvar 1 $listName list
    upvar 1 $iName i

    if {[string index $arg $length] != ""} {
	lappend list [string range $arg $length end]
    } else {
	incr i
	lappend list [lindex $args $i]
    }
}

proc parseCommand {args execFormat {access {}}} glue {

    global kern_fileDescriptors

	# everything starts out empty

    set length [llength $args]
    set programs {}
    set inputFds {}
    set outputFds {}
    set values {}
    set stdinRedirected 0
    set stdoutRedirected 0
    set stderrRedirected 0
    set background 0

	# skip the options if this is an "exec" command

    set i 0

    if {$execFormat} {

	for {} {$i < $length} {incr i} {

	    set arg [lindex $args $i]

	    if {[string index $arg 0] != "-"} {
		break;
	    }

	    if {$arg == "--"} {
		incr i
		break;
	    }	
	}
    }

	# now parse out the input fd's, the output fd's and the program names

    for {} {$i < $length} {incr i} {

	set arg [lindex $args $i]

	    # skip over pipelines

	if {$arg == "|"} {		
	    continue
	} 

	if {$arg == "|&"} {	
	    continue
	} 

	    # check for redirection markers of length 3

	set r [string range $arg 0 2]

	if {($r == "2>>") || ($r == ">>&")} {	
	    error "permission denied: can not redirect to a filename"
	} 

	if {$r == "2>@"} {
	    set stderrRedirected 1
	    breakout outputFds i $arg $args 3
	    continue
	}

	if {$r == ">&@"} {
	    set stdoutRedirected 1
	    set stderrRedirected 1
	    breakout outputFds i $arg $args 3
	    continue
	}

	    # check for redirection markers of length 2

	set r [string range $arg 0 1]

	if {($r == "2>") || ($r == ">&") || ($r == ">>")} {
	    error "permission denied: can not redirect to a filename"
	}

	if {$r == ">@"} {
	    set stdoutRedirected 1
	    breakout outputFds i $arg $args 2
	    continue
	}

	if {$r == "<@"} {
	    set stdinRedirected 1
	    breakout inputFds i $arg $args 2
	    continue
	}

	if {$r == "<<"} {
	    set stdinRedirected 1
	    breakout values i $arg $args 2
	    continue
	}
	 
	    # check for redirection markers of length 1

	set r [string range $arg 0 0]

	if {$r == ">"} {
	    error "permission denied: can not redirect to a filename"
	}

	if {$r == "<"} {
	    error "permission denied: can not redirect from a filename"
	}

	    # check for background execution

	if {$arg == "&"} {
	    set background 1
	}

	    # anything else must be a program name

	lappend programs $arg
    }

	# make sure that we have write access to each outout fd

    foreach fd $outputFds {
	if {[checkFdAccess $fd write] != "yes"} {
	    error "permission denied: can not write to file descriptor \"$fd\""
        }
    }

	# make sure that we have read access to each input fd

    foreach fd $inputFds {
	if {[checkFdAccess $fd read] != "yes"} {
	    error "permission denied: can not read from file descriptor \"$fd\""
        }
    }

	# stdout and stderr MUST be redirected if we are executing in background

    if {$background} {
	if {($stdoutRedirected != "1") || ($stderrRedirected != "1")} {
	    error "permission denied: stderr and stdout must be redirected for background execution"
	}
    }

	# different checks depending on whether this is an "exec" command

    if {$execFormat} {

	    # stdin MUST be redirected

	if {$stdinRedirected != "1"} {
	    error "permission denied: standard input must be redirected"
	}

    } else {

	    # stdin MUST be redirected if we do not have WRITE access to the pipeline

	if {([lsearch $access write] == "-1") && ($stdinRedirected != "1")} {
	    error "permission denied: standard input must be redirected if pipe is opened for writing only"
	}

	    # stdout MUST be redirected if we do not have READ access to the pipeline

	if {([lsearch $access read] == "-1") && ($stdoutRedirected != "1")} {
	    error "permission denied: standard output must be redirected if pipe is opened for reading only"
	}
    }

	# check for program access

    foreach program $programs {

	set splitname [file split $program]

	set code [catch {kernelRequireExec $splitname} result]

	if {$code} {
	    return -code error $result
	}

	if {$result != "yes"} {
	    return -code error "permission denied: exec $program"
	}
    }

	# done

    return [concat $inputFds $outputFds]
}

proc kernelExecPipeline {interp filename mode access} glue {

    global kern_allowAllExecs kern_fileDescriptors errorCode errorInfo

	# remember the original arguments

    set arguments [list $filename $mode]

	# parse the command -- parseCommand checks for file and program access

    set args [string range $filename 1 end]    

    if [catch {parseCommand $args 0 $access} fds] {
	return -code error $fds
    }

	# share any file descriptors

    foreach fd $fds {
	interp share $interp $fd {}
    }

	# if we make it here, we can do it

    set command [concat open [list $filename] $mode]
    set code [catch {eval $command} result]

	# close the shared file descriptors -- they stay open in the user interpeter

    foreach fd $fds {
	close $fd
    }

	# if we successfully open the pipeline, add the fd to the table

    if {$code == 0} {
	set fd $result
	interp transfer {} $fd $interp
	set kern_fileDescriptors($fd) [list pipeline $access]
    }

	# done

    return -code $code $result
}

proc kernelExec {interp args} glue {

    global kern_allowAllExecs

	# parse the command -- parseCommand checks for file and program access

    if [catch {parseCommand $args 1} fds] {
	return -code error $fds
    }

	# share any file descriptors

    foreach fd $fds {
	interp share $interp $fd {}
    }

	# if we make it here, we can do it

    set command [concat exec $args]
    set code [catch {eval $command} result]

	# close the shared file descriptors -- they stay open in the user interpeter

    foreach fd $fds {
	close $fd
    }

	# done

    return -code $code $result
}



# ---------------------------------------------------------------------------
#
# Network commands
#
# There is currently one relevant variable defined in the kernel interpreter.
#
#   kern_allowAllNetwork
#
#     This variable is set to "yes" if the agent can access the network
#     directly.  The variable is set to "no-confirmed" if the agent can not
#     access the network.  The variable is set to "no" if the agent can not 
#     access the network but has not yet contacted the resource manager.
#
# kern_allowAllNetwork defaults to "no" and is changed to "yes" or 
# "no-confirmed" only by the "kernelRequire"command (in coordination with 
# the resource managers).  
#
# Thus the overall security rule is ...
#
# The agent can either (1) access the network in any way OR (2) not access
# the network at all.
#
# This security rule is extremely simple -- i.e., there are only two classes
# of agents at the Tcl language level -- but provides enough flexibility to
# support both anonymous and trusted agents.  It is also reasonably easy to
# get it right and to verify its correctness which were the critical concerns
# in this first secure release. 
#
# ---------------------------------------------------------------------------

set kern_allowAllNetwork no

proc kernelSocketRedirect {interp command fd host port} glue {

    global kern_fileDescriptors

	# add the fd to the table

    interp transfer {} $fd $interp
    set kern_fileDescriptors($fd) [list network [list create read write]]

	# redirect into the slave 

    interp eval $interp $command $fd $host $port
}

proc kernelSocket {interp args} glue {

    global kern_allowAllNetwork kern_fileDescriptors errorCode errorInfo

	# see if we have a -server option

    if {[lindex $args 0] == "-server"} {
	
	set command [string trim [lindex $args 1]]

	if {$command == ""} {
	    return -code error "-server must be followed by a nonempty command"
	}

	set newCommand "kernelSocketRedirect $interp [list $command]"
	set newArgs [lreplace $args 1 1 $newCommand]

    } else {

	set newArgs $args
    }

	# see if we can access the network
 
    set code [catch {kernelRequireNetwork {}} result]

    if {$code} {
	return -code error $result
    }

    if {$result != "yes"} {
	return -code error [constructDeniedMessage socket $args] 
    }

	# if we make it here, we can do it

    set command [concat socket $newArgs]
    set code [catch {eval $command} result]

	# if we successfully open the socket, add the fd to the table

    if {$code == 0} {
	set fd $result
	interp transfer {} $fd $interp
	set kern_fileDescriptors($fd) [list network [list create read write]]
    }

	# done

    return -code $code $result
}



# ---------------------------------------------------------------------------
#
# Miscellaneous commands
#
# ---------------------------------------------------------------------------

proc kernelExit {interp {code 0}} glue {

	# There is no reason not to allow exit in the current system --
	# i.e., each agent runs in its own process and the server does not
	# need explicit notification from the agent to know that the agent
	# has terminated (which only makes sense).

    exit $code
}



# ---------------------------------------------------------------------------
#
# Install some variables into the slave
#
# ---------------------------------------------------------------------------

proc tclInit {} glue {

    global tcl_library tcl_version tcl_patchLevel

    rename tclInit {}

    foreach slave [interp slaves] {
	$slave eval [list set tcl_library $tcl_library]
	$slave eval [list set tcl_version $tcl_version]
	$slave eval [list set tcl_patchLevel $tcl_patchLevel]
    }
}

tclInit


proc kernelForever {interp args} {

    puts $args

}
