@TCL_STARTUP@

# Agent Tcl
# Bob Gray
# 8 october 1996
#
# manager.tcl
#
# This script is the resource manager that implements the security policy for
# the local Agent Tcl system.  Essentially the kernel interpreter for each
# agent contacts the resource manager when the agent is attempting a
# "dangerous" action such as opening a file, opening a network connection, 
# etc.  The resource manager decides whether or not the action should be
# allowed based on the agent's (un)authenticated identity and returns this
# decision to the kernel interpreter.
#
# USAGE:
#
# manager.tcl <confname> [-nosecurity allow | -nosecurity allow-time | -nosecurity disallow]
#
# <confname> specifies the name of the configuration file.  This file
# specifies those users that should be given the same filesystem, network,
# exec, dynamic loading, and/or screen access as the special agenttcl userid.
#
# The -nosecurity option specifies that you are not using any authentication 
# subsystem.  If -nosecurity is followed by "allow", the manager will allow
# all resource accesses (i.e. the system reverts to the behavior of the
# previous Agent Tcl release).  If -nosecurity is followed by "disallow",
# the manager will deny all resource accesses.  In this latter case you
# might want to expand safety.tcl so that you give the agents slightly more
# freedom.
#
# Finally, if you -nosecurity is followed by "allow-time", the manager will
# allow the agent to exist for as long as it wants, but will deny all other
# resource accesses.  THIS IS THE RECOMMENDED CHOICE IF YOU ARE NOT USING
# AN AUTHENTICATION SUBSYSTEM.  It allows you to create long-lived child
# agents (i.e., child agents that provide services to other agents) without
# allowing other accesses.
#
# global variables
#
# noSecurity              = If this variable exists, then *no* encryption subsystem
#                           is being used.  If the variable is set to "allow", 
#                           all resource requests are granted, whereas if the 
#                           variable is set to "disallow", all resource requests 
#                           are denied.  If the variable is set to "allow-time",
#			    wall-time requests are granted, but all other
#			    requests are denied. The existence and value of this variable
#                           is determined by the presence or absence of the
#                           "-nosecurity" command line option.
#
# manager_allowedExec     = If the array element manager_allowExec(KEY) exists,
#                           then the user with key KEY is allowed to execute
#                           external programs.  This variable is set inside the
#			    manager configuration file.
#
# manager_allowedSource   = If the array element manager_allowedSource(KEY) exists,
#                           then the user with key KEY is allowed to source
#                           any script file.  This variable is set inside the
#			    manager configuration file.
#
# manager_allowedLoad     = If the array element manager_allowedLoad(KEY) exists,
#                           then the user with key KEY is allowed to load any 
#                           library.  This variable is set inside the manager
#			    configuration file.
#
# manager_allowedNet      = If the array element manager_allowedNet(KEY) exists,
#                           then the user with key KEY is allowed to access the
#			    network in any way.  This variable is set inside the
#			    the manager configuration file.
# 
# manager_allowedScreen   = If the array element manager_allowedScreen(KEY) exists,
#                           then the user with key KEY is allowed to access the
#			    screen in any way.  This variable is set inside the
#			    the manager configuration file.
#
# manager_allowedFile     = If the array element manager_allowedFile(KEY) exists,
#                           then the user with key KEY is allowed to access the
#			    filesystem in any way.  This variable is set inside the
#			    the manager configuration file.
#
# manager_allowedChildren = If the array element manager_allowedChildren(KEY) exists,
#                           then its value is an integer n which specifies the
#                           maximum number of children that can be created by
#                           an agent belonging to the user with key KEY.  This
#                           variable is set inside the manager configuration
#                           file.
#
# manager_allowedDepth    = If the array element manager_allowedDepth(KEY) 
#			    exists, then its value is an integer n which 
#			    specifies the maximum depth of the parent-child
#                           agent hierarchy that can be created by an agent 
#		 	    belonging to the user with key KEY.  This variable
#                           is set inside the manager configuration file.
#
# manager_allowedJumps    = If the array element manager_allowedJumps(KEY) 
#			    exists, then its value is an integer n which 
#			    specifies the maximum number of jumps that can be
# 			    made by an agent belonging to the owner with key
#                           KEY.  This variable is set inside the manager
#                           configuration file. 
#
# manager_allowedWall     = If the array element manager_allowedWall(KEY) 
#			    exists, then its value is an integer n which 
#			    specifies the maximum lifetime (in seconds) of 
# 			    of an agent belonging to the owner with key KEY.
#                           This variable is set inside the manager
#                           configuration file. 
#


#
# code
#

    # utility routines to assemble the access lists

proc add_exec key {
    global manager_allowedExec
    set manager_allowedExec($key) {}
}

proc add_source key {
    global manager_allowedSource
    set manager_allowedSource($key) {}
}

proc add_load key {
    global manager_allowedLoad
    set manager_allowedLoad($key) {}
}

proc add_net key {
    global manager_allowedNet
    set manager_allowedNet($key) {}
}

proc add_screen key {
    global manager_allowedScreen
    set manager_allowedScreen($key) {}
}

proc add_file key {
    global manager_allowedFile
    set manager_allowedFile($key) {}
}

proc add_children {key n} {
    global manager_allowedChildren
    set manager_allowedChildren($key) $n
}

proc add_depth {key n} {
    global manager_allowedDepth
    set manager_allowedDepth($key) $n
}

proc add_jumps {key n} {
    global manager_allowedJumps
    set manager_allowedJumps($key) $n
}

proc add_wall {key n} {
    global manager_allowedWall
    set manager_allowedWall($key) $n
}

proc add_cpu {key n} {
    global manager_allowedCpu
    set manager_allowedCpu($key) $n
}

    # handlers for different resource types

proc handleCpuRequests {key access} {

    global manager_allowedCpu

    if {[info exists manager_allowedCpu($key)]} {

	set code [catch {
	    if {$access <= $manager_allowedCpu($key)} {
		set answer yes
	    } else {
		set answer no
	    }
	}]

	if {$code} {
	    set answer no
	}

    } else {

	set answer no
    }

    return $answer
}

proc handleWallRequests {key access} {

    global manager_allowedWall

    if {[info exists manager_allowedWall($key)]} {

	set code [catch {
	    if {$access <= $manager_allowedWall($key)} {
		set answer yes
	    } else {
		set answer no
	    }
	}]

	if {$code} {
	    set answer no
	}

    } else {

	set answer no
    }

    return $answer
}

proc handleChildrenRequests {key access} {

    global manager_allowedChildren 

    if {[info exists manager_allowedChildren($key)]} {

	set code [catch {
	    if {$access <= $manager_allowedChildren($key)} {
		set answer yes
	    } else {
		set answer no
	    }
	}]

	if {$code} {
	    set answer no
	}

    } else {

	set answer no
    }

    return $answer
}

proc handleDepthRequests {key access} {

    global manager_allowedDepth

    if {[info exists manager_allowedDepth($key)]} {

	set code [catch {
	    if {$access <= $manager_allowedDepth($key)} {
		set answer yes
	    } else {
		set answer no
	    }
	}]

	if {$code} {
	    set answer no
	}

    } else {

	set answer no
    }

    return $answer
}

proc handleJumpsRequests {key access} {

    global manager_allowedJumps

    if {[info exists manager_allowedJumps($key)]} {

	set code [catch {
	    if {$access <= $manager_allowedJumps($key)} {
		set answer yes
	    } else {
		set answer no
	    }
	}]

	if {$code} {
	    set answer no
	}

    } else {

	set answer no
    }

    return $answer
}

proc handleScreenRequests {key access} {

    global manager_allowedScreen

    if {[info exists manager_allowedScreen($key)]} {
	set answer "yes"
    } else {
	set answer "no-confirmed"
    }

    return $answer
}

proc handleNetRequests {key access} {

    global manager_allowedNet

    if {[info exists manager_allowedNet($key)]} {
	set answer "yes"
    } else {
	set answer "no-confirmed"
    }

    return $answer
}

proc handleExecRequests {key access} {

    global manager_allowedExec

    if {[info exists manager_allowedExec($key)]} {
	set answer "yes"
    } else {
	set answer "no-confirmed"
    }

    return $answer
}

proc handleSourceRequests {key access} {

    global manager_allowedSource

    if {[info exists manager_allowedSource($key)]} {
	set answer "yes"
    } else {
	set answer "no-confirmed"
    }

    return $answer
}

proc handleLoadRequests {key access} {

    global manager_allowedLoad

    if {[info exists manager_allowedLoad($key)]} {
	set answer "yes"
    } else {
	set answer "no-confirmed"
    }

    return $answer
}

proc handleFileRequests {key type access} {

    global manager_allowedFile

    if {[info exists manager_allowedFile($key)]} {
	set answer "yes"
    } else {
	set answer "no-confirmed"
    }

    return $answer
}

    # parse the command arguments

if {($argc != 1) && ($argc != 3)} {
    puts stderr "\nUSAGE: $argv0 <confname> \[-nosecurity allow | -nosecurity disallow\]\n"
    exit 1
}

if {$argc == 3} {

    set option [lindex $argv 1]

    if {$option != "-nosecurity"} {
	puts stderr "\nERROR: unknown option \"$option\": should be \"-nosecurity\"\n"
        puts stderr "\nUSAGE: $argv0 <confname> \[-nosecurity allow | -nosecurity disallow\]\n"
	exit 1
    }

    set flag [lindex $argv 2]

    if {$flag == "allow"} {
	set noSecurity allow
    } elseif {$flag == "disallow"} {
	set noSecurity disallow
    } elseif {$flag == "allow-time"} {
	set noSecurity allow-time
    } else {
	puts stderr "\nERROR: \"-nosecurity\" must be followed by \"allow\", \"allow-time\" or \"disallow\"\n"
        puts stderr "\nUSAGE: $argv0 <confname> \[-nosecurity allow | -nosecurity disallow\]\n"
	exit 1
    }
}

    # load the configuration file

set confFilename [lindex $argv 0]

if {![file exists $confFilename]} {
    puts stderr "\nERROR: configuration file \"$confFilename\" does not exist\n"
    exit 1
}

file stat $confFilename statArray

if {$statArray(mode) & 022} {
    puts stderr "\nERROR: configuration file \"$confFilename\" is group or other writeable\n"
    puts stderr "       Please verify the contents of \"$confFilename\", fix the permission"
    puts stderr "       bits, and then rerun \"$argv0\".\n"
    exit 1
}

source $confFilename
unset statArray

    # register with the agent server

if {![info exists noSecurity]} {
    security signatures on
}

agent_begin

    # get the symbolic name "resource-manager"

catch {
    agent_force "$agent(local-ip) resource-manager"
}

agent_name resource-manager

    # enter an infinite loop where we wait for resource requests

while {1} {

	# receive the resource request

    set source [agent_receive code string -security securityInfo -blocking]

	# break out the message pieces

    set sequence [lindex $string 0]
    set request  [lindex $string 1]
    set resource [string tolower [lindex $request 0]]
    set access   [lindex $request 1]

	# if the sending agent is not authenticated, then the request is
	# coming from an agent that is on some other machine and should be
	# denied out of hand; otherwise, if the sending agent is authenticated,
	# then the request is coming from an agent on the local machine and
	# must be considered -- in this latter case the request is rejected if
	# the agent's owner is anonymous but run through the appropriate access
	# table if the agent's owner is *not* anonymous

    set agentAuth [lindex $securityInfo 3]
    set owner [lindex $securityInfo 0]
    set ownerKey [lindex $owner 0]
    set ownerAuth [lindex $owner 1]

    if {$agentAuth != "agent-auth"} {

	set answer "no-confirmed"

    } elseif [info exists noSecurity] {
 
	if {$noSecurity == "allow"} {

	    set answer "yes"

	} elseif {$noSecurity == "disallow"} {

	    set answer "no-confirmed"

	} elseif {$noSecurity == "allow-time"} {

	    if {$resource == "wall"} {
		return "yes"
	    } else {
		return "no-confirmed"
	    }

	} else {

	    set answer "no-confirmed"
	}

    } elseif {$ownerAuth != "owner-auth"} {

	set answer "no-confirmed"

    } else {

	switch $resource {
	    exec	{ set answer [handleExecRequests $ownerKey $access] }
	    source	{ set answer [handleSourceRequests $ownerKey $access] }
	    load	{ set answer [handleLoadRequests $ownerKey $access] }
	    network	{ set answer [handleNetRequests $ownerKey $access] }
	    screen	{ set answer [handleScreenRequests $ownerKey $access] }
	    file	{ set answer [handleFileRequests $ownerKey file $access] }
	    directory	{ set answer [handleFileRequests $ownerKey directory $access] }
	    children    { set answer [handleChildrenRequests $ownerKey $access] }
	    depth       { set answer [handleDepthRequests $ownerKey $access] }
	    jumps       { set answer [handleJumpsRequests $ownerKey $access] }
	    wall        { set answer [handleWallRequests $ownerKey $access] }
	    cpu	        { set answer [handleCpuRequests $ownerKey $access] }
	    default	{ set answer "no-confirmed" }
	}
    }

	# send the response to the requesting agent

    set response [list $sequence $answer $request]

    catch {
	agent_send $source 0 $response
    }
} 
