# Agent Tcl
# Bob Gray
# 23 October 1996 
#
# tksafety.tcl 
#
# This file defines safe versions of the unsafe Tk commands.  These safe
# versions are loaded into the kernel interpreter.  



# ---------------------------------------------------------------------------
#
# make sure that we are at the top level
#
# ---------------------------------------------------------------------------

if {[info level] != 0} {
  return -code error "\"tksafety.tcl\" can only be sourced at the global level"
}


# ---------------------------------------------------------------------------
#
# File commands 
#
# The safe versions of the file commands are defined in the "safety.tcl"
# script.  Here we just add the "tk_library" directory to the set of
# directories from which the agent can source Tcl files.
#
# ---------------------------------------------------------------------------

if {$tcl_platform(platform) == "macintosh"} {

    set tk_library "resource"

} else {

    set curdir [pwd]
    cd $tk_library
    set tk_library [pwd]
    cd $curdir
}

# allow a source from the Tk library directory

set kern_allowSources([file split $tk_library]) 1

# allow access to the files inside the Tk library directory

set kern_allowDirectories([file split $tk_library]) [list read]

# ---------------------------------------------------------------------------
#
# Screen commands
#
# There is currently one relevant variable defined in the kernel interpreter.
#
#   kern_allowAllScreen
#
#     This variable is set to "yes" if the agent can access the screen.
#     The variable is set to "no_confirmed" if the agent can not access the
#     screen.  The variable is set to "no" if the agent can not access the
#     screen but has not yet contacted the resource manager.
#
# kern_allowAllScreen defaults to "no" and is changed to "yes" or "no-confirmed"
# only by the "kernelRequire" command (in coordination with the resource
# managers).
#
# Thus the overall security rule is ...
#
# The agent can either (1) access the screen in any way or (2) can not access
# the screen at all.
#
# This security rule is extremely simple -- i.e., there are only two classes
# of agents at the Tcl language level -- but provides enough flexibility to
# support both anonymous and trusted agents.  It is also reasonably easy to
# get it right and to verify its correctness which were the critical concerns
# in this first secure release. 
#
# ---------------------------------------------------------------------------

# default is to not allow any exec's 

set kern_allowAllScreen no 

proc kernelMain {interp args} {

    global kern_allowAllScreen 

	# check for the -interp argument

    foreach arg $args {
	if {$arg == "-interp"} {
	    return -code error "permission denied: \"-interp\" option can not be specified"
	}
    }

	# see if we can access the screen

    set code [catch {kernelRequire screen {}} answer]

    if {$code} {
	return -code error $answer
    }

    if {$answer != "yes"} {
	return -code error [constructDeniedMessage main $args]
    }

	# if we make it here, we can do it

    set command [concat main $args -interp $interp]
    set code [catch {eval $command} result]

	# done

    return -code $code $result
}


# ---------------------------------------------------------------------------
#
# Install some variables into the slave
#
# ---------------------------------------------------------------------------

proc tclInit {} glue {

    global tk_library tk_version tk_patchLevel env

    rename tclInit {}

    foreach slave [interp slaves] {

	$slave eval [list set tk_library $tk_library]
	$slave eval [list set tk_version $tk_version]
	$slave eval [list set tk_patchLevel $tk_patchLevel]

	if {[info exists env(DISPLAY)]} {
	    $slave eval [list set env(DISPLAY) $env(DISPLAY)]
	}
    }
}

tclInit
